/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.juneau.rest;

import static org.apache.juneau.rest.util.RestUtils.*;

import jakarta.servlet.http.*;

/**
 * A wrapper class that allows you to override basic fields.
 */
class OverrideableHttpServletRequest extends HttpServletRequestWrapper {

	private String pathInfo, servletPath;

	/**
	 * Constructor.
	 *
	 * @param request The wrapped servlet request.
	 */
	public OverrideableHttpServletRequest(HttpServletRequest request) {
		super(request);
	}

	@Override /* Overridden from HttpServletRequest */
	public String getPathInfo() {
		// Note that pathInfo can never be empty.
		return pathInfo == null ? super.getPathInfo() : pathInfo.charAt(0) == (char)0 ? null : pathInfo;
	}

	@Override /* Overridden from HttpServletRequest */
	public String getServletPath() { return servletPath == null ? super.getServletPath() : servletPath; }

	public OverrideableHttpServletRequest pathInfo(String value) {
		validatePathInfo(value);
		if (value == null)
			value = "\u0000";
		pathInfo = value;
		return this;
	}

	public OverrideableHttpServletRequest servletPath(String value) {
		validateServletPath(value);
		servletPath = value;
		return this;
	}
}