/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 *
 *   SPDX-License-Identifier: Apache-2.0
 */

package org.apache.jena.system;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertNull;

import org.junit.jupiter.api.Test;

import org.apache.jena.atlas.lib.Pair;
import org.apache.jena.riot.system.PrefixLib;
import org.apache.jena.riot.system.PrefixMap;
import org.apache.jena.riot.system.PrefixMapFactory;

public class TestPrefixLib {
    private PrefixMap create() {
        return PrefixMapFactory.create();
    }
    protected String pref1  = "pref1";
    protected String pref1a = "pref1:";
    protected String pref2  = "pref2";

    @Test
    public void abbreviate_1() {
        PrefixMap prefixes = create();
        prefixes.add(pref1, "http://example.net/ns#");

        String x = PrefixLib.abbreviate(prefixes, "http://example.net/ns#xyz");
        assertEquals("pref1:xyz", x);
    }

    @Test
    public void abbreviate_2() {
        PrefixMap prefixes = create();
        prefixes.add(pref1, "http://example.net/ns#");

        String x = PrefixLib.abbreviate(prefixes, "http://other/ns#xyz");
        assertNull(x);
    }

    @Test
    public void abbrev_1() {
        PrefixMap prefixes = create();
        prefixes.add(pref1, "http://example.net/ns#");

        Pair<String, String> x = PrefixLib.abbrev(prefixes, "http://example.net/ns#xyz");

        assertEquals("pref1", x.getLeft());
        assertEquals("xyz", x.getRight());
    }

    @Test
    public void abbrev_2() {
        PrefixMap prefixes = create();
        prefixes.add(pref1, "http://example.net/ns#");
        Pair<String, String> x = PrefixLib.abbrev(prefixes, "http://other/ns#xyz");
        assertNull(x);
    }

    @Test
    public void expand_1() {
        PrefixMap prefixes = create();
        prefixes.add(pref1, "http://example.net/ns#");
        String x = PrefixLib.expand(prefixes, "pref1:abc");
        assertEquals("http://example.net/ns#abc", x);
        String x2 = PrefixLib.expand(prefixes, "pref1z:abc");
        assertNull(x2);
    }

    @Test
    public void expand_2() {
        PrefixMap prefixes = create();
        prefixes.add(pref1, "http://example.net/ns#");
        String x2 = PrefixLib.expand(prefixes, "pref1z:abc");
        assertNull(x2);
    }
}
