/*
 * Copyright 2012 Splunk, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"): you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

package com.splunk;

import java.text.SimpleDateFormat;
import java.util.Date;

/**
 * The {@code SavedSearchDispatchArgs} class contains arguments for dispatching 
 * a saved search using the {@link SavedSearch#dispatch} method.
 */
public class SavedSearchDispatchArgs extends Args {

    /**
     * Class constructor.
     */
    public SavedSearchDispatchArgs() { super(); }
    
    /* BEGIN AUTOGENERATED CODE */
    
    /**
     * Sets a time that dispatches the search as though the specified time were the current time.
     * 
     * @param dispatchNow
     *      A string with the specified time.
     */
    public void setDispatchNow(String dispatchNow) {
        this.put("dispatch.now", dispatchNow);
    }
    
    /**
     * Sets the maximum number of timeline buckets. 
     * 
     * @param dispatchBuckets
     *      The maximum number of timeline buckets.
     */
    public void setDispatchBuckets(int dispatchBuckets) {
        this.put("dispatch.buckets", dispatchBuckets);
    }
    
    /**
     * Specifies the earliest time for this search. This value can be a relative or absolute time. If this value is an absolute time, use {@link #setDispatchTimeFormat} to format the value.
     * 
     * @param dispatchEarliestTime
     *      A time string that specifies the earliest time for this search.
     */
    public void setDispatchEarliestTime(String dispatchEarliestTime) {
        this.put("dispatch.earliest_time", dispatchEarliestTime);
    }
    
    /**
     * Sets the earliest time for this search. 
     * 
     * @param dispatchEarliestTime
     *      A date that specifies the earliest time for this search.
     */
    public void setDispatchEarliestTime(Date dispatchEarliestTime) {
        // Documented format:        "%FT%T.%Q%:z"
        // strftime original format: "%FT%T.%Q%:z"
        // strftime expanded format: "%Y-%m-%dT%H:%M:%S.%Q%:z"
        String javaFormatString =    "yyyy-MM-dd'T'HH:mm:ss.SSSZ";
        String timeString = new SimpleDateFormat(javaFormatString).format(dispatchEarliestTime);
        this.put("dispatch.earliest_time", timeString);
    }
    
    /**
     * Specifies the latest time for this search. This value can be a relative or absolute time. If this value is an absolute time, use {@link #setDispatchTimeFormat} to format the value.
     * 
     * @param dispatchLatestTime
     *      A time string that specifies the latest time for this search.
     */
    public void setDispatchLatestTime(String dispatchLatestTime) {
        this.put("dispatch.latest_time", dispatchLatestTime);
    }
    
    /**
     * Sets the latest time for this search. 
     * 
     * @param dispatchLatestTime
     *      A date that specifies the latest time for this saved search.
     */
    public void setDispatchLatestTime(Date dispatchLatestTime) {
        // Documented format:        "%FT%T.%Q%:z"
        // strftime original format: "%FT%T.%Q%:z"
        // strftime expanded format: "%Y-%m-%dT%H:%M:%S.%Q%:z"
        String javaFormatString =    "yyyy-MM-dd'T'HH:mm:ss.SSSZ";
        String timeString = new SimpleDateFormat(javaFormatString).format(dispatchLatestTime);
        this.put("dispatch.latest_time", timeString);
    }
    
    /**
     * Indicates whether to enable lookups for this search.
     * 
     * @param dispatchLookups
     *      {@code true} to enable lookups, {@code false} if not.
     */
    public void setDispatchLookups(boolean dispatchLookups) {
        this.put("dispatch.lookups", dispatchLookups);
    }
    
    /**
     * Sets the maximum number of results before finalizing the search. 
     * 
     * @param dispatchMaximumCount
     *      The maximum number of results.
     */
    public void setDispatchMaximumCount(int dispatchMaximumCount) {
        this.put("dispatch.max_count", dispatchMaximumCount);
    }
    
    /**
     * Sets the the maximum amount of time before finalizing the search.
     * 
     * @param dispatchMaximumTime
     *      The maximum amount of time, in seconds. 
     */
    public void setDispatchMaximumTime(int dispatchMaximumTime) {
        this.put("dispatch.max_time", dispatchMaximumTime);
    }
    
    /**
     * Set the frequency for how often Splunk should run the MapReduce reduce phase on accumulated map values.
     * 
     * @param dispatchReduceFrequency
     *      The frequency, in seconds.
     */
    public void setDispatchReduceFrequency(int dispatchReduceFrequency) {
        this.put("dispatch.reduce_freq", dispatchReduceFrequency);
    }
    
    /**
     * Indicates whether to back fill the real-time window for this search. This
     * value only applies to real-time searches. 
     * 
     * @param dispatchRealTimeBackfill
     *      {@code true} to back fill the real-time window, {@code false} if not.
     */
    public void setDispatchRealTimeBackfill(boolean dispatchRealTimeBackfill) {
        this.put("dispatch.rt_backfill", dispatchRealTimeBackfill);
    }
    
    /**
     * Indicates whether the search should run in a separate spawned process. Searches against indexes must run in a separate process.
     * 
     * @param dispatchSpawnProcess
     *      {@code true} to run the search in a separate process, {@code false} if not.
     */
    public void setDispatchSpawnProcess(boolean dispatchSpawnProcess) {
        this.put("dispatch.spawn_process", dispatchSpawnProcess);
    }
    
    /**
     * Sets a time format for Splunk to use to specify the earliest and latest times.
     * 
     * @param dispatchTimeFormat
     *      A time format string. 
     */
    public void setDispatchTimeFormat(String dispatchTimeFormat) {
        this.put("dispatch.time_format", dispatchTimeFormat);
    }
    
    /**
     * Indicates the time to live (TTL) for the artifacts of the scheduled search, if no actions are triggered. If an action is triggered, Splunk changes the TTL to that action's TTL. If multiple actions are triggered, Splunk applies the maximum TTL to the artifacts. To set the action's TTL, refer to alert_actions.conf.spec.
     * 
     * @param dispatchTtl
     *      The time to live, in seconds. If the value is a number followed by "p", it is the number of scheduled search periods.
     */
    public void setDispatchTtl(String dispatchTtl) {
        this.put("dispatch.ttl", dispatchTtl);
    }
    
    /**
     * Indicates whether to trigger alert actions.
     * 
     * @param triggerActions
     *      {@code true} to trigger alert actions, {@code false} if not.
     */
    public void setTriggerActions(boolean triggerActions) {
        this.put("trigger_actions", triggerActions);
    }
    
    /**
     * Indicates whether to start a new search, even if another instance of this search is already running.
     * 
     * @param forceDispatch
     *      {@code true} to start a new search, {@code false} if not.
     */
    public void setForceDispatch(boolean forceDispatch) {
        this.put("force_dispatch", forceDispatch);
    }
    
    /* END AUTOGENERATED CODE */
}
