
    | path |
    path := Prompter prompt: 'Installing from: ' default: 'a:\hotdrawv\'. 
    path isNil ifTrue: [^self]. 
    Smalltalk at: #HotDrawLibrary put: Dictionary new.
    (Smalltalk at: #HotDrawLibrary)
        at: #SelectionBitmap put: (Bitmap fromFile: (path, 'bitmaps\select.bmp'));
        at: #ScrollBitmap put: (Bitmap fromFile: (path, 'bitmaps\hand.bmp'));
        at: #BringToFrontBitmap put: (Bitmap fromFile: (path, 'bitmaps\front.bmp'));
        at: #SendToBackBitmap put: (Bitmap fromFile: (path, 'bitmaps\back.bmp'));
        at: #EraserBitmap put: (Bitmap fromFile: (path, 'bitmaps\erase.bmp'));
        at: #LineBitmap put: (Bitmap fromFile: (path, 'bitmaps\line.bmp'));
        at: #ArrowBitmap put: (Bitmap fromFile: (path, 'bitmaps\arrow.bmp'));
        at: #RectangleBitmap put: (Bitmap fromFile: (path, 'bitmaps\rect.bmp'));
        at: #EllipseBitmap put: (Bitmap fromFile: (path, 'bitmaps\ellipse.bmp'));
        at: #TextBitmap put: (Bitmap fromFile: (path, 'bitmaps\text.bmp'))!

Object subclass: #Drawing
  instanceVariableNames: 'figures '
  classVariableNames: ''
  poolDictionaries: ''!

Object subclass: #DrawingEditor
  instanceVariableNames: 'tools currentTool fileName drawing '
  classVariableNames: ''
  poolDictionaries: 'ColorConstants '!

Object subclass: #Figure
  instanceVariableNames: 'container '
  classVariableNames: 'ColorTable '
  poolDictionaries: ''!

Figure subclass: #CachedFigure
  instanceVariableNames: 'cache origin '
  classVariableNames: ''
  poolDictionaries: ''!

CachedFigure subclass: #BitmapFigure
  instanceVariableNames: ''
  classVariableNames: ''
  poolDictionaries: ''!

CachedFigure subclass: #GroupFigure
  instanceVariableNames: 'figures '
  classVariableNames: ''
  poolDictionaries: ''!

Figure subclass: #CompositeFigure
  instanceVariableNames: 'figures visibleArea visibleFigures showVisibleArea '
  classVariableNames: ''
  poolDictionaries: 'WinConstants '!

Figure subclass: #EllipseFigure
  instanceVariableNames: 'ellipse width color fillColor '
  classVariableNames: ''
  poolDictionaries: ''!

Figure subclass: #PolylineFigure
  instanceVariableNames: 'points origin extent width color closed fillColor '
  classVariableNames: ''
  poolDictionaries: 'ColorConstants '!

PolylineFigure subclass: #LineFigure
  instanceVariableNames: ''
  classVariableNames: ''
  poolDictionaries: ''!

LineFigure subclass: #ArrowFigure
  instanceVariableNames: ''
  classVariableNames: ''
  poolDictionaries: ''!

LineFigure subclass: #DependentLineFigure
  instanceVariableNames: ''
  classVariableNames: ''
  poolDictionaries: ''!

PolylineFigure subclass: #RectangleFigure
  instanceVariableNames: ''
  classVariableNames: ''
  poolDictionaries: 'ColorConstants '!

Figure subclass: #TextFigure
  instanceVariableNames: 'origin text readOnly font '
  classVariableNames: ''
  poolDictionaries: 'WinConstants ColorConstants '!

TextFigure subclass: #FixedTextFigure
  instanceVariableNames: 'length '
  classVariableNames: ''
  poolDictionaries: ''!

FixedTextFigure subclass: #NumberFigure
  instanceVariableNames: 'number '
  classVariableNames: ''
  poolDictionaries: ''!

Object subclass: #Handle
  instanceVariableNames: 'locator '
  classVariableNames: ''
  poolDictionaries: ''!

Handle subclass: #ConnectionHandle
  instanceVariableNames: 'className connectionAction '
  classVariableNames: ''
  poolDictionaries: ''!

Handle subclass: #TrackHandle
  instanceVariableNames: 'sense change '
  classVariableNames: ''
  poolDictionaries: ''!

TrackHandle subclass: #SelectionTrackHandle
  instanceVariableNames: ''
  classVariableNames: ''
  poolDictionaries: ''!

Object subclass: #Locator
  instanceVariableNames: 'receiver selector arguments '
  classVariableNames: 'CopiedFigures '
  poolDictionaries: ''!

Object subclass: #MultiheadedConstraint
  instanceVariableNames: 'sources sink action '
  classVariableNames: ''
  poolDictionaries: ''!

Object subclass: #NumberHolder
  instanceVariableNames: 'number '
  classVariableNames: ''
  poolDictionaries: ''!

Object subclass: #PositionConstraint
  instanceVariableNames: 'location receiver message '
  classVariableNames: ''
  poolDictionaries: ''!

Object subclass: #Tool
  instanceVariableNames: 'icon drawingPane '
  classVariableNames: ''
  poolDictionaries: ''!

Tool subclass: #CreationTool
  instanceVariableNames: 'className '
  classVariableNames: ''
  poolDictionaries: ''!

CreationTool subclass: #TextCreationTool
  instanceVariableNames: 'typingTarget '
  classVariableNames: ''
  poolDictionaries: 'VirtualKeyConstants '!

Tool subclass: #DrawingActionTool
  instanceVariableNames: ''
  classVariableNames: ''
  poolDictionaries: ''!

Tool subclass: #FigureActionTool
  instanceVariableNames: 'actionBlock '
  classVariableNames: ''
  poolDictionaries: ''!

Tool subclass: #ScrollingTool
  instanceVariableNames: ''
  classVariableNames: ''
  poolDictionaries: ''!

Tool subclass: #SelectionTool
  instanceVariableNames: ''
  classVariableNames: ''
  poolDictionaries: 'WinConstants '!

SubPane subclass: #HotDrawPane
  instanceVariableNames: 'editor buffer '
  classVariableNames: ''
  poolDictionaries: 'WinConstants '!

HotDrawPane subclass: #DrawingPane
  instanceVariableNames: 'drawing handles '
  classVariableNames: 'CopyBuffer '
  poolDictionaries: 'VirtualKeyConstants '!

HotDrawPane subclass: #ToolPalettePane
  instanceVariableNames: ''
  classVariableNames: ''
  poolDictionaries: 'ColorConstants '!


!ArrowFigure class methods !

start: startPoint stop: stopPoint
    "Return a new instance of the receiver with start aStart and stop aStop point"

    | array |
    array := Array new: 5.
    array at: 1 put: startPoint.
    array at: 2 put: stopPoint.
    array at: 3 put: 0@0.
    array at: 4 put: stopPoint.
    array at: 5 put: 0@0.
    ^(self withPoints: array) width: 1.

!

creationTool
    "Return a creation tool"

    ^CreationTool
        icon: (HotDrawLibrary at: #ArrowBitmap)
        class: self

!

COMMENT
"
    An ArrowFigure is a line figure with an arrow head at the ending point.

    Instance Variables:
    None

    Class Variables:
    None
"
! !


!ArrowFigure methods !

connectionPosition
    "Return the connection position (used by ConnectionHandle to find the target)"

    ^self startPoint
!

changed
    "The receiver changed, recompute the arrow head"

    self updateArrow.
    super changed
!

updateArrow
    "Recompute the arrow head"

    | vector unit aPoint stopPoint |
    stopPoint := self stopPoint.
    vector := stopPoint - self startPoint.
    vector = 0
        ifTrue: [
            unit := 0.
            aPoint := stopPoint]
        ifFalse: [
            unit := vector normal unitVector.
            aPoint := stopPoint - (vector unitVector * width * 8)].
    self points at: 3 put: (aPoint + (unit * (width * 4))) rounded.
    self points at: 5 put: (aPoint - (unit * (width * 4))) rounded.
    self points at: 4 put: stopPoint.
    self calculateBoundingBox

!

locator
    "Return a locator for receiver"

    ^Locator on: self at: #startPoint
! !


!BitmapFigure class methods !

COMMENT
"
    BitmapFigures represent arbitrary images in a drawing.

    Instance Variables:
    None

    Class Variables:
    None
"
! !


!BitmapFigure methods ! !


!CachedFigure class methods !

COMMENT
"
    CachedFigure is an abstract class that captures figures who wish to save
    their representation to a bitmap to speed up motion.

    Subclasses should implement:
    <fillCache> : Draw the representation onto the cache, a bitmap

    Instance Variables:
    <cache> : A bitmap
    <origin> : A point for the origin

    Class Variables:
    None
"
! !


!CachedFigure methods !

origin: aPoint
    "Set the origin of the receiver to aPoint"

    origin := aPoint
!

basicTranslateBy: aPoint
    "Translate the receiver by aPoint"

    origin := origin translateBy: aPoint
!

release
    "Release the receiver"

    cache release.
    super release
!

extent
    "Return the extent of the receiver"

    cache isNil ifTrue: [self fillCache].
    ^cache extent
!

displayOn: aGraphicsMedium
    "Display cache onto aGraphicsMedium"

    cache isNil ifTrue: [^nil].
    aGraphicsMedium pen
        copyBitmap: cache
        from: cache boundingBox
        at: self origin
!

origin
    "Return the origin of the receiver"

    ^origin
!

fillCache
    "Draw components onto the cache"

    ^self implementedBySubclass
! !


!CompositeFigure class methods !

visibleArea: aRectangle figures: aCollection
    "Return a new instance of the receiver"

    ^self new setFigures: aCollection visibleArea: aRectangle
!

figures: aCollection
    "Return a new instance of the receiver with each figure in aCollection
    as a visible figure of the composition. Set the visible area to cover all figures"

    | aRect |
    aRect := aCollection inject: aCollection first displayBox into: [:sum :each | sum merge: each displayBox].
    ^self visibleArea: aRect figures: aCollection
!

COMMENT
"
    CompositeFigure is an abstract subclass that permits composed figures.  A
    CompositeFigure is like a GroupFigure in that it contains component figures,
    but it is different in that the components are still accessible.  Furthermore, a
    CompositeFigure has a bounding region (called the visibleArea).  Only the
    components in this area are visible.  The components outside the visibleArea
    are not drawn.  This permits active components that are not visible (or accessible)
    to the casual user.

    When selected, a CompositeFigure offers handles on the corners of its visibleArea.
    If this area is contracted so that it touches or covers a component figure, the figure
    is no longer displayed.  This is how active component figures are masked from view.
    The component figures can be edited (including moved) by shift-selecting them. That
    is, while the CompositeFigure is not selected, hold the left shift key and select the
    component figure.  The handles for the figure will be offered and the figure can be
    moved or edited in the normal fashion.  Note that if you move the figure (actually its
    bounding box) outside of the visibleArea of the CompositeFigure, the component
    figure will disappear.  You may still move it further but once released, the figure
    becomes an invisible component of the CompositeFigure.

    Several visibleArea operations are currently supported via the yellow button menu.
    You can show or hide the visibleArea which appears as a solid black rectangular
    outline.  You can also resize the visibleArea to include all the component figures.
    This is valuable in development if you accidently move a component figure outside
    the visibleArea and release the mouse button.  It is expected that subclasses of
    CompositeFigure not necessarily permit users to manipulate the visibleArea in such
    a casual way unless it is important to the operation of the figure.

    Concrete subclasses of CompositeFigure need not implement any new methods.
    However, it is expected that certain behaviour would want to be changed.  For
    example, it is likely that subclasses will inhibit the ability to change the visibleArea
    of a CompositeFigure.  Other behaviour changes are also likely.

    Instance Variables:
    <figures> : A collection of component figures
    <visibleArea> : A rectangle
    <visibleFigures> : A collection of visible component figures
    <showVisibleArea> : A boolean

    Class Variables:
    None
"
! !


!CompositeFigure methods !

showVisibleAreaIndicator
    "Show the visible area border"

    showVisibleArea := true.
    self changed
!

hideVisibleAreaIndicator
    "Don't show the visible area border"

    showVisibleArea := false.
    self changed
!

extent
    "Return the extent of the receiver"

    ^self visibleArea extent
!

setFigures: aCollection visibleArea: aRectangle
    "Private - Initialize the receiver"

    figures := aCollection.
    visibleArea := aRectangle.
    self resetVisibleFigures.
    figures do: [:each | each addDependent: self].
    showVisibleArea := false.
!

showVisibleArea
    "Return a boolean"

    ^showVisibleArea
!

resetVisibleFigures
    "Recalculate which figures are visible"

    visibleFigures := figures select: [:each | visibleArea contains: each displayBox]
!

visibleFigures: aCollection
    "Set the collection of visible figures"

    visibleFigures := aCollection
!

displayOn: aGraphicsMedium
    "Draw each of my figures that fall entirely within my visible region on aGraphicsContext."

    self visibleFigures reverseDo: [:each | each displayOn: aGraphicsMedium].
    self showVisibleArea ifTrue: [
        aGraphicsMedium pen
            defaultNib: 1;
            place: self visibleArea origin;
            box: self visibleArea corner]
!

growBy: aRectangle
    "Change the size of visible area and update the figures within it"

    (visibleArea origin < aRectangle origin negated or: [visibleArea extent < aRectangle extent negated])
        ifTrue:
            [^self].
    visibleArea := (visibleArea origin + aRectangle origin) corner: (visibleArea corner + aRectangle corner).
    self resetVisibleFigures.
    self changed
!

visibleArea
    "Return the visible area of the receiver"

    ^visibleArea
!

resetVisibleArea
    "Recalculate the visible area"

    self visibleArea: ((figures inject: figures first displayBox into: [:rect :each | rect merge: each displayBox]) insetBy: -1@-1).
    self resetVisibleFigures.
    self changed
!

figures: aCollection
    "Set the receiver component figures"

    figures := aCollection
!

containsPoint: aPoint
    "Return true if the receiver contains aPoint, else return false"

    ^(Notifier isKeyDown: VkShift) not and: [visibleArea containsPoint: aPoint]
!

menu
    "Return the composite figure menu"

    ^Menu
        labels: 'show visible area\hide visible area\reset visible area' withCrs
        lines: #(2)
        selectors: #(menuShowVisibleAreaIndicator menuHideVisibleAreaIndicator menuResetVisibleArea )

!

copy
    | aFigure |
    aFigure := super copy.
    aFigure setFigures: (figures collect: [:each | each copy])
        visibleArea: visibleArea copy.
    showVisibleArea ifTrue: [aFigure showVisibleAreaIndicator].
    ^aFigure
!

showVisibleArea: aBoolean
    "Show or hide the visible area border"

    showVisibleArea := aBoolean
!

update: aFigure
    "Update the receiver"

    self resetVisibleFigures.
    self changed
!

figures
    "Return the receiver component figures"

    ^figures
!

figureAt: aPoint
    "Return the figure at aPoint or nil"

    | aFigure |
    (Notifier isKeyDown: VkShift)
        ifTrue: [
            visibleFigures do: [:each |
                aFigure := each figureAt: aPoint.
                aFigure notNil ifTrue: [^aFigure]].
            ^nil]
        ifFalse: [^super figureAt: aPoint]
!

menuShowVisibleAreaIndicator
    "Menu selector"

    self showVisibleAreaIndicator.
    self container update
!

menuHideVisibleAreaIndicator
    "Menu selector"

    self hideVisibleAreaIndicator.
    self container update
!

visibleArea: aRect
    "Set the visible area of the receiver"

    visibleArea := aRect
!

menuResetVisibleArea
    "Menu selector"

    self resetVisibleArea.
    self container update
!

visibleFigures
    "Return the collection of visible figures"

    ^visibleFigures
!

basicTranslateBy: aPoint
    "Translate all the component figures by a point"

    visibleArea := visibleArea translateBy: aPoint.
    figures do: [:each | each translateBy: aPoint]
!

origin
    "Return the receiver origin"

    ^self visibleArea origin
! !


!ConnectionHandle class methods !

COMMENT
"
    A ConnectionHandle is a Handle that creates a connecting Figure
    from the Figure to which it belongs and the nearest other figure.
    After the ConnectionHandle is released, the connecting figure is
    added to the Drawing. It has an optional connectionAction, which
    is a block that gets evaluated with the two figures. This makes it
    easy to set up specialized constraints between objects that are
    connected.

    Instance Variables:
    <className> : The name of the class of the connecting figure which will be created
    <connectionAction> : A block with the first argument the source figure and the second the target figure

    Class Variables:
    None
"
!

on: aFigure at: aSymbol class: aClass
    "Return a new instance of the class"

    ^(super on: aFigure at: aSymbol) setClassName: aClass name asSymbol
!

on: aFigure at: aSymbol
    "Return a new instance of the receiver"

    ^self on: aFigure at: aSymbol class: DependentLineFigure

! !


!ConnectionHandle methods !

setClassName: aSymbol
    "Set the class"

    className := aSymbol
!

findTarget: aPane
    "Find the target figure in aPane"

    | aFigure startPoint stopPoint |
    aPane noSelections.
    startPoint := stopPoint := self center.
    Notifier consumeInputUntil: [:event |
        stopPoint := aPane mouseLocation.
        aFigure := aPane figureAt: stopPoint.
        (aFigure = self or: [
            (self isFor: aFigure) or: [
                aFigure notNil and: [
                    aFigure canBeConnected not]]]) ifTrue: [aFigure := nil].
        stopPoint := aFigure isNil
            ifTrue: [stopPoint]
            ifFalse: [aFigure connectionPosition].
        aPane pen
            place: startPoint;
            goto: stopPoint.
        aPane update.
        event selector = #button1Up:].
    ^aFigure


!

connectionClass
    "Return the class"

    ^Smalltalk at: className
!

action: aBlock
    "Set he action block of the receiver"

    connectionAction := aBlock
!

invoke: aPane
    "Invoke aPane"

    | aFigure |
    aFigure := self findTarget: aPane.
    aFigure isNil ifTrue: [^self].
    self connectTo: aFigure in: aPane.
    connectionAction isNil ifFalse: [connectionAction value: self owner value: aFigure]
!

connectTo: aFigure in: aPane
    "Connect the receiver to aFigure in aPane"

    | aConnectionFigure |
    aConnectionFigure := self connectionClass startLocation: locator stopLocation: aFigure locator.
    aPane addFigure: aConnectionFigure.
    aPane sendToBack: aConnectionFigure
! !


!CreationTool class methods !

COMMENT
"
    A CreationTool is an abstract subclass of Tool.
    CreationTools are tools that create new figures in
    a drawing. A creation tool knows the class of the
    figure to be created when the tool is pressed in the
    background of a drawing.

    Instance Variables:
    <className> : The class name of the figure to be created

    Class Variables:
    None
"
!

icon: aBitmap class: aClass
    "Return a new instance of the class"

    ^(self icon: aBitmap) setClassName: aClass name asSymbol
! !


!CreationTool methods !

setClassName: aSymbol
    "Set the class name"

    className := aSymbol
!

pressBackground
    "The user has pressed the tool on the drawing background"

    self createFigure


!

pressFigure: aFigure
    "The user has pressed the tool on top of another figure"

    self createFigure
!

createFigure
    "Create the figure"

    | aFigure |
    aFigure := self creationClass createNotifying: self drawingPane.
    aFigure notNil ifTrue: [self drawingPane addFigure: aFigure].
    ^aFigure
!

creationClass
    "Return the class of the figure to be created"

    ^Smalltalk at: className
! !


!DependentLineFigure class methods !

COMMENT
"
    A DependentLineFigure is a LineFigure that can only be moved by constraints.

    Instance Variables:
    None

    Class Variables:
    None
"
! !


!DependentLineFigure methods !

basicTranslateBy: aPoint
    "Do nothing - the only way to move the receiver is to move the end points"
! !


!Drawing class methods !

COMMENT
"
    A Drawing captures the essence of a graphical drawing.  It contains a number
    of Figures each of which have some visual representation.

    Instance Variables :
    <figures> : The figures contained in the drawing

    Class Variables:
    None
"
!

new
    "Return a new instance of the receiver"

    ^super new initialize
! !


!Drawing methods !

figures
    "Return the figures of the receiver"

    ^figures
!

addFigure: newFigure behind: oldFigure
    "Add newFigure behind oldFigure"

    self figures add: newFigure after: oldFigure
!

figuresIntersecting: aRectangle
    "Return the figures intersecting aRectangle"

    ^figures select: [:each | each intersects: aRectangle]
!

displayOn: aGraphicsMedium
    "Display the receiver on aGraphicsMedium"

    figures reverseDo: [:each |
        (each displayBox intersects: aGraphicsMedium boundingBox) ifTrue: [
            each displayOn: aGraphicsMedium]]
!

top
    "Return the first figure"

    ^self figures first
!

sendToBack: aFigure
    "Send aFigure to the back of the drawing"

    (figures includes: aFigure) ifTrue: [
        figures remove: aFigure.
        figures addLast: aFigure]
!

bottom
    "Return the last figure"

    ^figures last
!

installInEditor: anEditor
    "Change the tools, menu etc of the editor"
!

figuresIn: aRectangle
    "Return the figures in aRectangle"

    ^figures select: [:each | each containedBy: aRectangle]
!

release
    "Release the receiver"

    super release.
    figures do: [:each | each isFigure ifTrue: [each release]].
    figures := nil
!

isDrawing
    "Return a boolean"

    ^true
!

tools
    "Return the tools of the receiver"

    ^self class tools
!

figures: aCollection
    "Set the figures of the receiver to aCollection"

    figures := aCollection
!

addFigure: aFigure
    "Add a figure to the receiver"

    self figures addFirst: aFigure
!

addFigures: aCollection
    "Add aCollection of figures to the receiver"

    aCollection reverseDo: [:each | self addFigure: each]
!

copy
    "Return a copy of the receiver"

    | aDrawing |
    aDrawing := self class new initialize.
    Locator copyWhile: [figures reverseDo: [:each | aDrawing addFigure: each copy]].
    ^aDrawing
!

boundingBox
    "Return the bounding box of the receiver"

    figures isEmpty
        ifFalse: [^figures inject: figures first displayBox into: [:sum :each | sum merge: each displayBox]]
        ifTrue: [^0@0 extent: 0@0]
!

initialize
    "Private - Initialize the receiver"

    self figures: OrderedCollection new
!

figureAt: aPoint
    "Return the figure located at aPoint"

    | aFigure |
    figures do: [:each |
        aFigure := each figureAt: aPoint.
        aFigure notNil ifTrue: [^aFigure]].
    ^nil
!

bringToFront: aFigure
    "Bring aFigure to front"

    (figures includes: aFigure) ifTrue: [
        figures remove: aFigure.
        figures addFirst: aFigure]
!

addFigureLast: aFigure
    "Add aFigure last"

    self add: aFigure behind: figures last
!

isFigure
    "Return a boolean"

    ^false
!

removeFigure: aFigure
    "Remove aFigure from the receiver"

    figures remove: aFigure ifAbsent: []
!

removeFigures: aCollection
    "Remove aCollection of figures from the receiver"

    aCollection do: [:each | self removeFigure: each]
! !


!DrawingActionTool class methods !

COMMENT
"
    DrawingActionTools are tools which operate on the entire Drawing.
    When pressed (or alternately when the tool is selected) the actionBlock
    is evaluated with the DrawingEditor as an argument

    Instance Variables:
    <activateBlock> : The block to be executed when the tool is pressed (single argument, the drawing editor)
    <deactivateBlock> : The block to be executed when the tool is released (drawing editor as argument)

    Class Variables:
    None
"
! !


!DrawingActionTool methods ! !


!DrawingEditor class methods !

open
    "DrawingEditor open"

    self openOn: self drawingClass new
!

drawingClass
    "Return the drawing class"

    ^Drawing
!

openOn: aDrawing
    "Open a new instance of the receiver with drawing aDrawing"

    | aDrawingView aToolPaletteView aWindow container toolPaletteSize aDrawingEditor |
    aDrawingEditor := self new setDrawing: aDrawing.
    toolPaletteSize := aDrawingEditor tools size * (Tool toolImageSize + Tool toolIconSpacing + Tool toolHiliteSize) x + Tool toolIconSpacing x + 4.
    aDrawingView := DrawingPane on: aDrawingEditor.
    aDrawingView drawing: aDrawing.
    aToolPaletteView := ToolPalettePane on: aDrawingEditor.
    aWindow := ViewManager new.
    aWindow
        labelWithoutPrefix: self label;
        noSmalltalkMenuBar;
        backColor: ClrLightgray;
        addSubpane: (aToolPaletteView
            owner: aWindow;
            backColor: ClrWhite;
            framingBlock: [:box | 0@0 extent: (4 + Tool toolImageSize x + Tool toolHiliteSize x + (2 * Tool toolIconSpacing) x) @toolPaletteSize]);
        addSubpane: (aDrawingView
            owner: aWindow;
            backColor: ClrWhite;
            framingBlock: [:box | (4 + Tool toolImageSize x + Tool toolHiliteSize x + (2 * Tool toolIconSpacing) x) @0 corner: box corner]).
    aWindow mainView openWindow.




!

COMMENT
"
    DrawingEditor represents the model of a graphical drawing editor.

    Instance Variables:
    <tools> : A collection of tools that can be used on the current drawing
    <currentTool> : One of the tools from the tools collection
    <fileName> : The filename where the current drawing is stored
    <drawing> : The drawing beeing currently edited

    Class Variables:
    None
"
!

label
    "Return the label of the editor"

    ^'Drawing Editor'
!

defaultTools
    "Answer an OrderedCollection of the tools I can use."

    ^(OrderedCollection new)
            add: SelectionTool new;
            add: ScrollingTool new;
            add: FigureActionTool bringToFront;
            add: FigureActionTool sendToBack;
            add: FigureActionTool delete;
            add: LineFigure creationTool;
            add: ArrowFigure creationTool;
            add: RectangleFigure creationTool;
            add: EllipseFigure creationTool;
            add: TextFigure creationTool;
            yourself

!

new
    "Return a new instance of the class"

    | editor |
    editor := super new.
    editor tools: self defaultTools.
    ^editor
! !


!DrawingEditor methods !

currentTool: aTool
    "Set the current tool to aTool"

    currentTool := aTool
!

saveDrawing
    "Save a drawing to a file"

    Menu message: 'Not implemented yet !!'
!

currentTool
    "Return the current tool"

    ^currentTool
!

setDrawing: aDrawing
    "Private - Initialize the receiver"

    drawing := aDrawing.
    aDrawing installInEditor: self
!

loadDrawing
    "Read a drawing from a file"

    Menu message: 'Not implemented yet !!'



!

tools
    "Return the tools of the receiver"

    ^tools
!

tools: aCollection
    "Set the tools to aCollection"

    tools := aCollection.
    tools isNil ifFalse: [self currentTool: tools first]
!

drawing
    "Return a copy of the drawing"

    ^drawing copy
! !


!DrawingPane class methods !

on: aDrawingEditor
    "Return a new instance of the receiver"

    | aDrawingView |
    aDrawingView := super new initialize.
    aDrawingEditor tools do: [:each | each drawingPane: aDrawingView].
    ^aDrawingView editor: aDrawingEditor

! !


!DrawingPane methods !

removeSelections
    "Remove the selected figures"

    | aCollection |
    aCollection := self selections.
    aCollection do: [:each | each changed].
    drawing removeFigures: aCollection.
    self noSelections.
    self update

!

isSelected: aFigure
    "Return a boolean"

    self handles
        detect: [:each | each isFor: aFigure]
        ifNone: [^false].
    ^true
!

menuGroup
    "Group the selected figures"

    | aCollection aGroup |
    aCollection := self selections.
    aCollection isEmpty ifTrue: [^self].
    self noSelections.
    drawing removeFigures: aCollection.
    aGroup := GroupFigure figures: aCollection asArray.
    self addFigure: aGroup.
    aGroup changed.
    self selection: aGroup.
    self update
!

bufferHandles
    "Display the handles onto the cache"

    self handles do: [:each |
        each displayOn: self buffer at: each origin]
!

initialize
    "Private - initialize the receiver"

    super initialize.
    self handles: OrderedCollection new
!

handlesFor: aFigure
    "Return the handles corresponding to aFigure"

    ^handles select: [:each | each isFor: aFigure]
!

addFigure: aFigure
    "Add aFigure"

    self drawing addFigure: aFigure.
    aFigure container: self.        "wrong - container should be drawing !!!!"
    aFigure changed.
    self update
!

bufferFigures
    "Display the drawing onto the cache"

    self drawing displayOn: self buffer
!

menuDecompose
    "Decompose the selected figures"

    | aCollection |
    aCollection := self selections.
    aCollection isEmpty ifTrue: [^self].
    self noSelections.
    self selections: (aCollection inject: Set new into: [:sum :each |
        each hideVisibleAreaIndicator.
        each changed.
        each release.
        drawing removeFigure: each.
        each figures reverseDo: [:aFigure |
            self addFigure: aFigure.
            aFigure dependents remove: each ifAbsent: []].
        sum addAll: each figures; yourself]).
    self update
!

removeFigure: aFigure
    "Remove aFigure"

    aFigure changed.
    self drawing removeFigure: aFigure.
    self update
!

handles: aCollection
    "Set the handles"

    handles := aCollection
!

characterInput: aChar
    "User inputs a character from the keyboard"

    self editor currentTool characterInput: aChar.

!

menu
    "Return the standard editor menu"

    ^Menu
        labels: 'Save\Load\Copy\Cut\Paste\Group\Ungroup\Compose\Decompose' withCrs
        lines: #(2 5 7)
        selectors: #(menuSaveDrawing menuLoadDrawing menuCopy menuCut menuPaste menuGroup menuUngroup menuCompose menuDecompose)
!

drawing: aDrawing
    "Set the drawing"

    drawing := aDrawing
!

button1Down: aPoint
    "User pressed the left mouse button"

    self editor currentTool press.
!

menuCompose
    "Compose all the selected figures into a ComposedFigure."

    | aCollection aCF |
    aCollection := self selections asOrderedCollection.
    aCollection isEmpty ifTrue: [^self].
    self noSelections.
    drawing removeFigures: aCollection.
    aCF := CompositeFigure figures: aCollection.
    drawing figures isEmpty
        ifTrue: [self addFigure: aCF]
        ifFalse: [self addFigure: aCF behind: drawing figures last].
    aCF changed.
    self selection: aCF.
    self update
!

selections: aCollection
    "Set the selected figures to aCollection"

    self noSelections.
    aCollection do: [:figure |
        figure handles do: [:ahandle |
            self handles add: ahandle]]

!

sendToBack: aFigure
    "Send aFigure to back"

    self drawing sendToBack: aFigure.
    self update
!

menuCut
    "Cut the current selected figures"

    self editor currentTool canCutCopyPaste ifTrue: [^self editor currentTool cut].
    self menuCopy.
    self removeSelections
!

selection: aFigure
    "Set the selection to aFigure"

    self noSelections.
    self selections: (Array with: aFigure)

!

menuCopy
    "Copy the currently selected figures"

    | anArray aRectangle |
    self editor currentTool canCutCopyPaste ifTrue: [^self editor currentTool copy].
    anArray := self selections asArray.
    anArray isEmpty ifFalse: [
        aRectangle := anArray
            inject: anArray first displayBox
            into: [:sum :each | sum merge: each displayBox].
    Locator copyWhile: [CopyBuffer := anArray collect: [:each | each copy basicTranslateBy: (0@0) - aRectangle origin]]]
!

createBuffer
    "Fill the cache"

    super createBuffer.
    self bufferFigures.
    self bufferHandles.


!

close
    "Release the drawing"

    self drawing release.
    super close
!

menuSaveDrawing
    "Save drawing to a file"

    ^self editor saveDrawing
!

handles
    "Return the handles"

    ^handles
!

scrollBy: aPoint
    "Scroll the window"

    self drawing figures do: [:each | each translateBy: aPoint]
!

menuLoadDrawing
    "Load a drawing from a file"

    ^self editor loadDrawing
!

activate
    "The receiver has just been activated,
    set the focus for keboard input"

    super activate.
    self setFocus
!

addAndSelectAll: aCollection
    "Add figures"

    self noSelections.
    self drawing addFigures: aCollection.
    aCollection do: [:each |
        each container: self.       "wrong - container should be drawing, fix later !!!!"
        each changed].
    self selections: aCollection.
    self update
!

figureAt: aPoint
    "Return the figure at aPoint"

    self drawing isNil ifTrue: [^self].
    ^self handles
            detect: [:each | each containsPoint: aPoint]
            ifNone: [self drawing figureAt: aPoint]
!

drawing
    "Return the drawing"

    ^drawing
!

figuresIn: aRectangle
    "Return the figures in aRectangle"

    ^self drawing figuresIn: aRectangle
!

removeHandlesFor: aFigure
    "Remove the handles corresponding to aFigure"

    (self handlesFor: aFigure) do:
        [:each | self removeHandle: each]
!

addHandle: aHandle
    "Add aHandle"

    self handles add: aHandle.
    self update
!

noSelections
    "Reset handles"

    handles := OrderedCollection new.
!

doPopupMenuAt: aPoint
    "Pop the corresponding menu"

    | popMenu aFigure menuOwner menuSelector |
    aFigure := self figureAt: self mouseLocation.
    aFigure isNil
        ifTrue: [
            popMenu := self menu.
            menuOwner := self]
        ifFalse: [
            popMenu := aFigure menu.
            menuOwner := aFigure].
    menuSelector := popMenu popUp.
    menuSelector isNil ifFalse: [
        menuOwner perform: menuSelector]


!

toggleSelections: aCollection
    "Toggle selection for aCollection of figures"

    aCollection do: [:each | self toggleSelection: each]
!

toggleSelection: aFigure
    "Toggle selection for aFigure"

    (self isSelected: aFigure)
        ifTrue: [self removeHandlesFor: aFigure]
        ifFalse: [self addHandlesFor: aFigure]
!

button1DownShift: aPoint
    "Shift + left mouse button"

    self editor currentTool press.
    self editor currentTool release

!

button1Up: aPoint
    "User released left mouse button"

    self editor currentTool release.

!

selections
    "Return the selected figures"

    ^Set new addAll: (self handles collect: [:each | each owner]); yourself
!

menuUngroup
    "Ungroup the selected figures"

    | aCollection |
    aCollection := self selections.
    aCollection isEmpty ifTrue: [^self].
    self noSelections.
    self selections: (aCollection inject: Set new into: [:sum :each |
        each release.
        drawing removeFigure: each.
        each figures reverseDo: [:aFigure | self addFigure: aFigure].
        sum addAll: each figures; yourself]).
    self update
!

menuPaste
    "Paste figures"

    | newFigures aPoint |
    self editor currentTool canCutCopyPaste ifTrue: [^self editor currentTool paste].
    CopyBuffer isNil ifFalse: [
        aPoint := self mouseLocation.
        Locator copyWhile: [newFigures := CopyBuffer collect: [:each | each copy basicTranslateBy: aPoint]].
        self noSelections.
        self addAndSelectAll: newFigures]
!

addHandlesFor: aFigure
    "Add handles for aFigure"

    aFigure handles do: [:each | self addHandle: each]
!

bringToFront: aFigure
    "Bring aFigure to front"

    self drawing bringToFront: aFigure.
    self update
!

removeHandle: aHandle
    "Remove aHandle"

    self handles remove: aHandle ifAbsent: [].
    self update
!

controlKeyInput: aKey
    "aKey has been pressed"

    aKey asciiValue = BackspaceKey ifTrue: [self editor currentTool backspace]
! !


!EllipseFigure class methods !

createNotifying: aPane
    "Return a new instance of the receiver, an ellipse figure.
    aPane is updated dynamically; i.e. while the figure is growing"

    | aFigure |
    aFigure := self ellipse: (aPane mouseLocation extent: 0@0).
    aFigure growNotifying: aPane.
    ^aFigure
!

ellipse: anEllipse
    "Return a new instance of the receiver"

    ^(self new
        width: 1
        color: (GraphicsTool red: 0 green: 0 blue: 0)
        fillColor: (GraphicsTool red: 255 green: 255 blue: 255))
        setEllipse: anEllipse!

creationTool
    "Return a creation tool"

    ^CreationTool
        icon: (HotDrawLibrary at: #EllipseBitmap)
        class: self
!

COMMENT
"
    An EllipseFigure represents an ellipse.

    Instance Variables:
    <ellipse> : A rectangle representing the bounding box of the ellipse
    <width> : An integer representding the widht of the ellipse border
    <color> : An rgb value representing the color of the border
    <fillColor> : An rgb value representing the color of the interior

    Class Variables:
    None
"
! !


!EllipseFigure methods !

menuDarkGrayFill
    "Set the fill color"

    self fillColor: 3.
    self container update


!

borderWidthBy: anInteger
    "Increase border width by anInteger"

    self borderWidth: ((width + (anInteger // 4) min: 25) max: 1)!

basicTranslateBy: aPoint
    "Translate the receiver by aPoint"

    ellipse := ellipse translateBy: aPoint
!

borderWidth: anInteger
    "Set the border width to anInteger"

    width := anInteger.
    self changed
!

menu
    "Return the menu for a polyline figure"

    | lineWidthMenu lineColorMenu fillColorMenu |
    lineWidthMenu := Menu
            labels: '1 pixel\2 pixel\3 pixel\4 pixel' withCrs
            lines: #()
            selectors: #(menu1PixelBorder menu2PixelBorder menu3PixelBorder menu4PixelBorder).
    lineWidthMenu title: 'line width'.
    lineColorMenu := Menu
            labels: 'black\dark gray\light gray\white' withCrs
            lines: #()
            selectors: #(menuBlackBorder menuDarkGrayBorder menuLightGrayBorder menuWhiteBorder).
    lineColorMenu title: 'line color'.
    fillColorMenu := Menu
            labels: 'black\dark gray\light gray\white' withCrs
            lines: #()
            selectors: #(menuBlackFill menuDarkGrayFill menuLightGrayFill menuWhiteFill).
    fillColorMenu title: 'fill color'.
    ^Menu new
        appendSubMenu: lineWidthMenu;
        appendSubMenu: lineColorMenu;
        appendSubMenu: fillColorMenu




!

growNotifying: aPane
    "Grow the receiver and notify aPane afterwards"

    | aHandle |
    aHandle := TrackHandle bottomRightOf: self.
    aPane addFigure: self.
    aHandle invoke: aPane.
    "Remove figure because it is going to be added later."
    aPane removeFigure: self.

!

darkenBy: anInteger
    "Darken the receiver by anInteger"

    | index |
    index := ColorTable indexOf: fillColor.
    index := (index + ((anInteger / 5) truncated) min: ColorTable size) max: 1.
    fillColor := ColorTable at: index.
    self changed!

menuLightGrayBorder
    "Set the border color"

    self borderColor: 4.
    self container update
!

origin
    "Return the origin of the receiver"

    ^ellipse origin!

menuBlackFill
    "Set the fill color"

    self fillColor: 1.
    self container update
!

growBy: aRectangle
    "Grow the receiver"

    ellipse := Rectangle
        origin: (ellipse origin + aRectangle origin)
        corner: (ellipse corner + aRectangle corner).
    self changed
!

displayBox
    "Return the bounding box of the receiver in document coordinates"

    ^Rectangle origin: self origin - (0.5 + width // 2) extent: self extent + width!

fillColor: anInteger
    "Set the border color of the receiver to aColor"

    fillColor := ColorTable at: anInteger.
    self changed
!

menuWhiteFill
    "Set the fill color"

    self fillColor: 5.
    self container update




!

menuLightGrayFill
    "Set the fill color"

    self fillColor: 4.
    self container update



!

handles
    "Return an ordered collection with selection handles for the receiver"

    ^(super handles)
        add: (SelectionTrackHandle colorOf: self);
        add: (SelectionTrackHandle borderColorOf: self);
        add: (SelectionTrackHandle widthOf: self);
        add: (ConnectionHandle on: self at: #center);
        yourself
!

borderColor: anInteger
    "Set the border color of the receiver to aColor"

    color := ColorTable at: anInteger.
    self changed
!

displayOn: aGraphicsMedium
    "Display the receiver on aGraphicsMedium"

    | pen oldWidth oldColor oldFillColor |
    pen := aGraphicsMedium pen.
    oldWidth := pen width.
    oldColor := pen foreColor.
    oldFillColor := pen backColor.
    pen
        setLineWidth: width;
        foreColor: color;
        backColor: fillColor;
        ellipse: ellipse;
        setLineWidth: oldWidth;
        foreColor: oldColor;
        backColor: oldFillColor
!

menuBlackBorder
    "Set the line color"

    self borderColor: 1.
    self container update
!

setEllipse: anEllipse
    "Private - Set ellipse to anEllipse"

    ellipse := anEllipse!

scaleBy: aPoint
    "Scale the receiver by aPoint"

    ellipse := ellipse scaleBy: aPoint.
    self changed
!

menuDarkGrayBorder
    "Set the border color"

    self borderColor: 3.
    self container update
!

extent
    "Return the extent of the receiver"

    ^ellipse extent rounded!

width: anInteger color: aColorValue fillColor: anotherColorValue
    "Private - initialize the receiver"

    width := anInteger.
    color := aColorValue.
    fillColor := anotherColorValue!

menuWhiteBorder
    "Set the border color"

    self borderColor: 5.
    self container update
!

menu2PixelBorder
    "Set border thickness to 2 pixel"

    self borderWidth: 2.
    self container update
!

menu3PixelBorder
    "Set border thickness to 3 pixel"

    self borderWidth: 3.
    self container update
!

borderDarkenBy: anInteger
    "Darken the border by anInteger"

    | index |
    index := ColorTable indexOf: color.
    index := (index + (anInteger / 5) truncated min: ColorTable size) max: 1.
    color := ColorTable at: index.
    self changed!

menu4PixelBorder
    "Set border thickness to 4 pixel"

    self borderWidth: 4.
    self container update
!

menu1PixelBorder
    "Set border thickness to 1 pixel"

    self borderWidth: 1.
    self container update
!

center
    "Return the center of the receiver"

    ^(ellipse left + ellipse right / 2) rounded @ (ellipse top + ellipse bottom / 2) rounded! !


!Figure class methods !

initializeColorTable
    "Initialize an array of grayscale colors, starting with black and ending with white"

    ColorTable := (Array new: 5)
        at:1 put: (Pen red: 0 green: 0 blue: 0);
        at:2 put: (Pen red: 57 green: 57 blue: 57);
        at:3 put: (Pen red: 128 green: 128 blue: 128);
        at:4 put: (Pen red: 189 green: 189 blue: 189);
        at:5 put: (Pen red: 255 green: 255 blue: 255);
        yourself!

initialize
    "   (Figure initialize)     "

    self initializeColorTable!

COMMENT
"
    Figure is an abstract class.

    A subclass of Figure must implement:
    <origin> : The origin of the graphical image of the subclass
    <extent> : The extent of the graphical image of the subclass
    <displayOn:> : Display the graphical image of the subclass on aGraphicsMedium
    <basicTranslateBy:> : Move a figure by aPoint

    Instance Variables:
    <container> : The drawing pane containing the currently edited drawing.
                        It should be the drawing itself, this will change.
                        Used for updating the pane (i.e. container update)

    ClassVariables:
    <ColorTable> : An array of colors used in figure coloring
"

!

new
    "Return a new instance of the receiver"

    ^super new initialize!

allCreationTools
    "Return the creation tools of all subclasses"

    | definingClasses |
    definingClasses := self allSubclasses select: [:each | each class includesSelector: #creationTool].
    ^definingClasses collect: [:each | each creationTool]
! !


!Figure methods !

bottomRight
    "Return the bottom right of the receiver"

    ^self displayBox corner
!

do: aBlock
    "Evaluate aBlock with self as argument"

    aBlock value: self!

isActive
    "Return true if the receiver is active, else return false"

    ^false!

senseColor: deltaPoint
    "Sense color"

    ^deltaPoint y!

initialize
    "Do nothing"
!

senseTopLeft: deltaPoint
    "Sense top left"

    ^Rectangle origin: deltaPoint extent: deltaPoint negated!

isFigure
    "Return true if the receiver is a figure, else return false"

    ^true!

displayBox
    "Return the bounding box of the receiver in document coordinates"

    ^self origin extent: self extent!

kindsOfFigures
    "Return an array with the class name"

    ^Array with: self class!

height
    "Return the height of the receiver"

    ^self displayBox height
!

origin
    "Return the origin of the receiver"

    ^self subclassResponsibility!

noMenu
    "Do nothing"
!

menuBindings
    "Return the collection of symbols in menu that represent messages that
    should be sent to the receiver instead of the controller."

    ^#()!

menu
    "Return the corresponding menu"

    ^Menu
        labels: 'no menu' withCrs
        lines: #()
        selectors: #(noMenu)
!

topLeft
    "Return the top left coordinate of the receiver"

    ^self displayBox topLeft!

owner
    "Return the owner of the receiver"

    ^self!

release
    "Release the receiver"

    self container: nil.
    super release
!

extent
    "Return the extent of the receiver"

    ^self implementedBySubclass!

bottom
    "Return the bottom of the receiver"

    ^self displayBox bottom!

containedBy: aRectangle
    "Return true if the receiver is contained by aRectangle"

    ^aRectangle contains: self displayBox!

displayOn: aGraphicsMedium
    "Display the receiver on aGraphicsMedium"

    ^self implementedBySubclass
!

top
    "Return the top coordinate of the receiver"

    ^self displayBox top!

handles
    "Return an ordered collection with selection handles for the receiver"

    ^SelectionTrackHandle allCornersOf: self.

!

container: anObject
    "Set the container"

    container := anObject
!

left
    "Return the left coordinate of the receiver"

    ^self displayBox left!

intersects: aRectangle
    "Return true if the receiver intersects aRectangle, else return false"

    ^self displayBox intersects: aRectangle!

dependentFigures
    "Return the dependent figures"

    ^self dependents select: [:each | each isFigure]!

connectionPosition
    "Return the position that a line should be connected to the receiver"

    ^ self center!

= aFigure
    "Equality operator"

    ^self class == aFigure class and: [self displayBox = aFigure displayBox]!

figureAt: aPoint
    "Return yourself if containing aPoint else return nil"

    ^(self containsPoint: aPoint)
            ifTrue: [self]
            ifFalse: [nil]
!

senseBottomLeft: deltaPoint
   "Sense bottom left"

    ^Rectangle origin: (deltaPoint x)@0 extent: (deltaPoint x negated)@(deltaPoint y)!

canBeConnected
    "Return true if the receiver can be connected, else return false"

    ^ true!

topRight
    "Return the top right coordinate of the receiver"

    ^self displayBox topRight!

isConnectionFigure
    "Return true if the receiver is a connection figure, else return false"

    ^false!

figures
    "Return an array with yourself"

    ^Array with: self!

asBitmap
    "Make the figure into a bitmap"

    ^self implementedBySubclass
!

align: alignmentPoint with: relativePoint
    "Align the receiver to alignmentPoint relative to relativePoint"

    self translateBy: relativePoint - alignmentPoint!

acceptsTyping
    "Return true if the receiver accepts typing, else return false"

    ^false!

container
    "Return the container"

    ^container
!

bottomCenter
    "Return the bottom center of the receiver"

    ^self displayBox bottomCenter!

bottomLeft
    "Return the bottom left of the receiver"

    ^self displayBox bottomLeft!

corner
    "Return the corner of the receiver"

    ^self displayBox corner!

topCenter
    "Return the center of the top edge of the receiver"

    ^self displayBox topCenter!

scaleBy: aPoint
    "Scale the receiver by aPoint"

    self
        align: self displayBox center
        with: (self displayBox center scaleBy: aPoint)!

changed
    "Inform all dependents to update"

    self dependents do: [:each | each update: self]
!

right
    "Return the right coordinate of the receiver"

    ^self displayBox right!

rightCenter
    "Return the center of the right edge of the receiver"

    ^self displayBox rightCenter!

dependentDrawings
    "Return the dependent drawings"

    ^self dependents select: [:each | each isDrawing]!

offOrigin: deltaPoint
    "Return a point which is deltaPoint off the origin"

    ^self displayBox origin + deltaPoint!

displayOn: aGraphicsMedium at: aPoint
    "Display the receiver on aGraphicsMedium translated by aPoint"

    aGraphicsMedium pen place: aPoint.
    self displayOn: aGraphicsMedium
!

locator
    "Return a locator on the center of the receiver"

    ^Locator on: self at: #center!

width
    "Return the width of the receiver"

    ^self displayBox width
!

center
    "Return the center of the receiver"

    ^self displayBox center!

absolute: aRelativePoint
    "Return the absolute value of the extent wrt aRelativePoint"

    | aRectangle |
    aRectangle := self displayBox.
    ^aRectangle extent * aRelativePoint + aRectangle origin!

translateBy: aPoint
    "Translate the receiver by aPoint and update after"

    self basicTranslateBy: aPoint.
    self changed
!

senseBottomRight: deltaPoint
    "Sense bottom right"

    ^Rectangle origin: 0 @ 0 extent: deltaPoint!

leftCenter
    "Return the center of the left edge of the receiver"

    ^self displayBox leftCenter!

translateTo: aPoint
    "Translate the receiver by aPoint"

    self translateBy: aPoint - self displayBox corner!

relative: anAbsolutePoint
    "Return a point which is anAbsolutePoint relative to the origin"

    | aRectangle |
    aRectangle := self displayBox.
    ^anAbsolutePoint - aRectangle origin * 1.0 / aRectangle extent!

senseTopRight: deltaPoint
    "Sense top right"

    ^Rectangle origin: 0@(deltaPoint y) extent: (deltaPoint x)@(deltaPoint y negated)!

boundingBox
    "Return the bounding box of the receiver in figure coordinates"

    ^0@0 extent: self extent!

containsPoint: aPoint
    "Return true if the receiver contains aPoint, else return false"

    ^self displayBox containsPoint: aPoint!

figure
    "Return yourself"

    ^self!

offCorner: deltaPoint
    "Return a point which is deltaPoint off the corner"

    ^self displayBox corner + deltaPoint!

isDrawing
    "Return true if the receiver is a drawing, else return false"

    ^false!

offCenter: deltaPoint
    "Return a point which is deltaPoint of the center"

    ^self center + deltaPoint!

basicTranslateBy: aPoint
    "Translate the receiver by aPoint"

    ^self implementedBySubclass! !


!FigureActionTool class methods !

COMMENT
"
    FigureActionTools are tools which operate on a figure when
    they are pressed. The FigureActionTool has an actionBlock
    that gets evaluated when the tool is pressed on a figure.
    The arguments of the block are the figure on which the tool
    was pressed and the view in which the cursor is located.

    Instance Variables:
    <actionBlock> : The block that gets evaluated when the tool is pressed.
                             The arguments are a figure and a view.

    Class Variables:
    None
"
!

delete
    "Return a creation tool"

    | tool |
    tool := FigureActionTool icon: (HotDrawLibrary at: #EraserBitmap).
    ^tool actionBlock: [:aFigure :aDrawingPane |
        aDrawingPane removeFigure: aFigure]
!

sendToBack
    "Return a creation tool"

    | tool |
    tool := FigureActionTool icon: (HotDrawLibrary at: #SendToBackBitmap).
    ^tool actionBlock: [:aFigure :aDrawingPane |
        aDrawingPane sendToBack: aFigure]



!

bringToFront
    "Return a creation tool"

    | tool |
    tool := FigureActionTool icon: (HotDrawLibrary at: #BringToFrontBitmap).
    ^tool actionBlock: [:aFigure :aDrawingPane |
        aDrawingPane bringToFront: aFigure]

! !


!FigureActionTool methods !

pressBackground
    "Do nothing"

!

actionBlock: aBlock
    "Set the action block"

    actionBlock := aBlock
!

actionBlock
    "Return the action block"

    ^actionBlock
!

pressFigure: aFigure
    "The user pressed the tool on top of another figure"

    ^self actionBlock value: aFigure value: self drawingPane
! !


!FixedTextFigure class methods !

string: aString at: aPoint
    "Return a new instance of the class"

    ^self new text: aString origin: aPoint
!

COMMENT
"
    A FixedTextFigure is just like any other TextFigure except the right
    margin is limited to length pixels.  This prohibits the user from causing
    the text to wrap around to the next line.

    Instance Variables:
    <length> : The maximum size of the text

    Class Variables:
    None
"
! !


!FixedTextFigure methods !

replaceFrom: start to: stop with: string notifying: aPane
    "Replace a portion of the text and notify aPane"

    self text
        replaceFrom: start
        to: stop
        with: string.
    aPane update
!

length: anInteger
    "Set the max length of the text"

    length := anInteger
!

length
    "Return the length of the text"

    ^length
! !


!GroupFigure class methods !

figures: aCollection
    "Return a new instance of the receiver"

    ^self new setFigures: aCollection
!

COMMENT
"
    A GroupFigure is a collection of figures.
    Scalings are not permitted on GroupFigures.

    Instance Variables:
    <figures> : The figures that make up the group figure

    Class Variables:
    None
"
! !


!GroupFigure methods !

figures
    "Return a collection of figures"

    ^figures
!

scaleBy: aPoint
    "GroupFigures cant be scaled."
!

kindsOfFigures
    "Return a collection"

    ^self figures inject: (Set with: self class) into: [:sum :each | sum addAll: each kindsOfFigures; yourself]
!

growBy: aPoint
    "GroupFigures can't grow"
!

senseTopLeft: deltaPoint
    "No scaling of GroupFigures"
!

release
    "Release the receiver"

    figures do: [:each |
        each dependents remove: self ifAbsent: [].
        each translateBy: self origin].
    super release

!

do: aBlock
    "For each figure evaluate aBlock"

    aBlock value: self.
    figures do: [:each | each do: aBlock]
!

copy
    "Return a copy of the receiver"

    ^self class figures: (figures collect: [:each | each copy])
!

fillCache
    "Draw each of the component Figures onto the cache"

    | extent |
    extent := (figures inject: figures first displayBox into: [:sum :each | sum merge: each displayBox]) extent.
    cache isNil ifFalse: [cache release].
    cache := Bitmap screenExtent: extent.
    figures reverseDo: [:each | each displayOn: cache at: 0@0]
!

senseBottomLeft: deltaPoint
    "No scaling of GroupFigures"
!

transformBy: aTransformation
    "No transformation permitted on GroupFigures"
!

update: aFigure
    "Return error"

    self error: 'Not supposed to be called !!'
!

senseBottomRight: deltaPoint
    "No scaling of GroupFigures"
!

setFigures: aCollection
    "Set the component figures of the receiver"

    figures := aCollection.
    origin := (figures inject: figures first displayBox into: [:sum :each | sum merge: each displayBox]) origin.
    figures do: [:each | each translateBy: self origin negated].
    self fillCache
!

senseTopRight: deltaPoint
    "No scaling of GroupFigures"
! !


!Handle class methods !

on: aFigure at: aSymbol
    "Return a new instance of the receiver"

    ^self new setLocator: (Locator on: aFigure at: aSymbol)
!

handleSize
    "Return the default handle size"

    ^6 @ 6



!

on: aFigure at: aSymbol with: anArgument
    "Return a new instance of the receiver"

    ^self new setLocator: (Locator on: aFigure at: aSymbol with: anArgument)
!

COMMENT
"
    A Handle is used to manipulate another figure.
    Handles operate through Locators to change a certain
    aspect of a Figure.  Handles use invoke: and invokeStep:
    methods to permit the handle to affect its change while it
    is being pressed.

    Instance Variables:
    <locator> : A Locator, an intermediate link between a handle and its figure

    Class Variables:
    None
"
! !


!Handle methods !

extent
    "Return the extent of the receiver"

    ^Handle handleSize
!

invoke: aPane
    "Invoke aPane"

    | oldPoint newPoint |
    oldPoint := aPane mouseLocation.
    Notifier consumeInputUntil: [:event |
        newPoint := aPane mouseLocation.
        self invokeStep: newPoint - oldPoint.
        oldPoint := newPoint.
        aPane update.
     event selector = #button1Up:]


!

invokeStep: deltaPoint
    "Step by delta"
!

isActive
    "Return a boolean"

    ^true
!

center
    "Return the center of the receiver"

    ^self displayBox center
!

locator
    "Return the locator for the receiver"

    ^locator
!

handles
    "Return the handles of the receiver, none"

    ^Array new
!

owner
    "Return the owner of the receiver"

    ^locator object
!

displayOn: aGraphicsMedium at: aPoint
    "Display the receiver on aGraphicsMedium at aPoint"

    aGraphicsMedium pen reverse: (aPoint extent: Handle handleSize)
!

containsPoint: aPoint
    "Return a boolean"

    ^self displayBox containsPoint: aPoint
!

menu
    "Return the menu for a handle"

    ^Menu
        labels: 'no menu' withCrs
        lines: #()
        selectors: #(noMenu)
!

origin
    "Return the origin of the receiver"

    ^locator value - (Handle handleSize // 2) rounded
!

displayBox
    "Return the display box of the receiver"

    ^Rectangle origin: self origin extent: self extent
!

canBeConnected
    "Return a boolean"

    ^false
!

setLocator: aLocator
    "Set the locator for the receiver"

    locator := aLocator
!

isFor: aFigure
    "Return a boolean"

    ^self owner == aFigure
!

noMenu
    "Do nothing"
!

owner: aFigure
    "Set the owner of the receiver"

    locator := locator copyOn: aFigure
! !


!HotDrawPane class methods ! !


!HotDrawPane methods !

close
    "The pane is closed, release the buffer"

    self buffer isNil ifFalse: [self buffer release].
    super close
!

getGraphicsTool
    "Private - Initialize the graphics tool to be a pen"

    | dc |
    dc := self getDC.
    graphicsTool :=  Pen forDC: dc medium: self.
    ^graphicsTool


!

mouseLocation
    "Return the mouse location relative to the window"

    ^Cursor sense mapScreenToClient: self
!

buffer: aBitmap
    "Set the buffer to aBitmap"

    buffer := aBitmap
!

buffer
    "Return a Bitmap instance"

    ^buffer
!

editor
    "Return a DrawingEditor instance"

    ^editor

!

createBuffer
    "Create a bitmap the size of the receiver"

    self buffer isNil ifFalse: [self buffer release].
    self buffer: (Bitmap screenExtent: self rectangle extent)

!

display
    "Copy the buffer to the screen"

    super display.
    self createBuffer.
    self buffer displayAt: 0@0 with: self pen


!

update
    "Update the receiver"

    super update.
    self display
!

editor: aDrawingEditor
    "Set the editor to aDrawingEditor"

    editor := aDrawingEditor

!

defaultStyle
    "Return the default window style, i.e. no scroll bars, no border"

    ^WsChild   |
     WsVisible |
     WsClipsiblings |
     WsClipchildren

!

updateSliders
        "Private - Reimplemented here because HotDrawPanes
         have no scrollbars."

! !


!LineFigure class methods !

createNotifying: aPane
    "Return a new instance of the receiver, a line figure.
    aPane is updated dynamically; i.e. while the figure is growing"

    | startPoint stopPoint pen line |
    pen := aPane pen.
    startPoint := aPane mouseLocation.
    line := self start: startPoint stop: (startPoint + (1@0)).
    aPane addFigure: line.
    Notifier consumeInputUntil: [:event |
        stopPoint := aPane mouseLocation.
        startPoint = stopPoint ifFalse: [
            line stopPoint: stopPoint.
            line changed.
            aPane update].
        event selector = #button1Up:].
    aPane removeFigure: line.
    ^line

!

start: aStartPoint stop: aStopPoint
    "Return a new instance of the receiver with start aStart and stop aStop point"

    ^(self withPoints: (Array with: aStartPoint with: aStopPoint)) width: 1

!

startLocation: startLocation stopLocation: stopLocation
    "Return a new instance of the receiver with start aStart and stop
    aStop point.  Add constraints to the startLocation and stopLocation"

    | aFigure |
    aFigure := self start: startLocation value stop: stopLocation value.
    startLocation object addDependent: (
        PositionConstraint new
            location: startLocation
            receiver: aFigure
            sending: #startPoint:).
    stopLocation object addDependent: (
        PositionConstraint new
            location: stopLocation
            receiver: aFigure
            sending: #stopPoint:).
    ^aFigure

!

COMMENT
"
    A LineFigure is the concrete class which represents a simple, straight line.
    The endpoints are kept in the points instance variable defined in the superclass
    PolylineFigure.  The line width and color are also kept in instance variables.
    It is not assumed that the points collection contain Points.  It is assumed that
    whatever is contained therein responds to the asPoint message with a Point.
    This was partly done to generalize a LineFigure into a ConnectionFigure where
    the points are really Locators.

    Instance Variables:
    None

    Class Variables:
    None
"
!

creationTool
    "Return a creation tool"

    ^CreationTool
        icon: (HotDrawLibrary at: #LineBitmap)
        class: self

! !


!LineFigure methods !

isHorizontalOrVertical
    "Return true if the receiver is vertical or horizontal, else return false"

    | p q |
    p := self startPoint.
    q := self stopPoint.
    ^p x = q x | p y = q y
!

moveStartBy: deltaPoint
    "Move the start point by delta"

    self startPoint: self startPoint + deltaPoint.
    self changed
!

moveStopBy: deltaPoint
    "Move the stop point by delta"

    self stopPoint: self stopPoint + deltaPoint.
    self changed
!

distanceTo: aPoint
    "Returns the distance from the receiver to aPoint"

    | p q d a b c |
    p := self startPoint.
    q := self stopPoint.
    d := p - q.
    a := 1.
    b := (d x / d y) negated.
    c := (p x + (p x - q x / (q y - p y) * p y)) negated.
    ^(aPoint x * a + (aPoint y * b) + c) abs / (a squared + b squared) sqrt
!

containsPoint: aPoint
    "Returns true if the receiver contains point aPoint"

    ^(self displayBox containsPoint: aPoint) and:
        [self isHorizontalOrVertical or: [(self distanceTo: aPoint) < 3]].
!

handles
    "Return a collection of handles"

    ^Array
        with: (TrackHandle
            on: self
            at: #startPoint
            change: #moveStartBy:)
        with: (TrackHandle
            on: self
            at: #stopPoint
            change: #moveStopBy:)

!

stopPoint
    "Return the stop point"

    ^points at: 2
!

startPoint: aPoint
    "Set the start point to aPoint"

    self points at: 1 put: aPoint
!

startPoint
    "Return the start point"

    ^self points first
!

stopPoint: aPoint
    "Set the stop point to aPoint"

    self points at: 2 put: aPoint
! !


!Locator class methods !

on: anObject at: aSymbol withArguments: anArray

    ^self new setReceiver: anObject selector: aSymbol arguments: anArray
!

copyAt: aFigure ifAbsent: aBlock

    ^CopiedFigures == nil
        ifTrue: [aBlock value]
        ifFalse:[CopiedFigures at: aFigure ifAbsent: [
            CopiedFigures at: aFigure put: aBlock value]]
!

on: anObject at: aSymbol

    ^self new setReceiver: anObject selector: aSymbol arguments: nil
!

copyAt: aFigure

    ^self copyAt: aFigure ifAbsent: [aFigure copy]
!

on: anObject at: aSymbol with: anElement

    ^self new setReceiver: anObject selector: aSymbol arguments: (Array with: anElement)
!

copyWhile: aBlock
        "Answer a copy of aFigure preserving the identity of shared Figures."

        | anObject |
        CopiedFigures := IdentityDictionary new.
        anObject := aBlock value.
        CopiedFigures := nil.
        ^anObject
! !


!Locator methods !

copy
    "Return a copy of the receiver"

    ^CopiedFigures notNil
                ifTrue: [self copyOn: (Locator copyAt: receiver)]
                ifFalse: [super copy]
!

asPoint
    "Make the receiver into a Point"

    ^self value
!

copyOn: anObject

    ^self species on: anObject at: selector withArguments: arguments
!

object
    "Return the receiver"

    ^receiver
!

setReceiver: anObject selector: aSymbol arguments: anArray
    "Private - Initialize the receiver"

    receiver := anObject.
    selector := aSymbol.
    arguments := anArray
!

value
    "Perform some action"

    ^arguments == nil
        ifTrue: [receiver perform: selector]
        ifFalse: [receiver perform: selector withArguments: arguments]
! !


!MultiheadedConstraint class methods !

COMMENT
"
    A MultiheadedConstraint makes the state of one object be the function
    of the states of many other objects.  The one object is the <sink> and the
    many objects are the <sources>.  For example, if the value of one cell
    in a spreadsheet is the sum of five other cells, the five cells are the
    sources and the cell with the sum is the sink.  A MultiheadedConstraint
    also has an <action>, which is a block with two arguments that is
    evaluated when any of the sources is changed.  The first block is
    the sources, the second is the sink.  A block to compute a sum would be
    [:sources :sink | sink value: (sources inject: 0 into: [:sum :each: | sum + each value]

    Adding a source to the constraint makes the constraint a dependent of
    the source.

    Instance Variables:
    <sources> : A collection of objects dependent on the sink
    <sink> : An object
    <action> : A block evaluated when one of the sources changed

    Class Variables:
    None
"
! !


!MultiheadedConstraint methods !

sink: aSink
    "Set the sink object"

    sink := aSink
!

addSource: aFigure
    "Add a source object to the receiver"

    sources add: aFigure.
    aFigure addDependent: self.
    self update: aFigure
!

sources
    "Return the source objects"

    ^sources
!

sources: aCollection
    "Set the source objects"

    sources := aCollection
!

update: aFigure
    "Update the receiver"

    action value: sources value: sink
!

sink
    "Return the sink object"

    ^sink
!

action: aBlock
    "Set the action"

    action := aBlock
!

for: aFigure
    "Change the sink"

    sink := aFigure.
    sources := OrderedCollection new.
!

for: aFigure action: actionBlock
    "Change the sink with actionBlock"

    sink := aFigure.
    sources := OrderedCollection new.
    action := actionBlock
!

action
    "Return the action"

    ^action
! !


!NumberFigure class methods !

createNotifying: aPane
    "Return a new instance of the receiver, a number figure.
    aPane is updated dynamically; i.e. while the figure is growing"

    | aFigure |
    aFigure := self string: '0' at: aPane mouseLocation.
    aFigure number: 0.
    ^aFigure
!

COMMENT
"
    A NumberFigure is a TextFigure that holds numbers, and that supports protocol
    number and number: to fetch and set this number.  A NumberFigure can be
    edited by the user just like any other TextFigure

    Instance Variables:
    <number> : An integer

    Class Variables:
    None
"

! !


!NumberFigure methods !

replaceFrom: start to: stop with: aString notifying: aPane
    "Replace a portion of the text and notify aPane"

    number := aString asInteger.
    super replaceFrom: start to: stop with: aString notifying: aPane.


!

number: aNumber
    "Set the number to aNumber"

    number := aNumber
!

text: aString
    "Set the text to aString"

    number := aString asInteger.
    super text: aString.
!

number
    "Return the number"

    ^number
! !


!NumberHolder class methods ! !


!NumberHolder methods !

number: aNumber
    "Set the number value"

    number := aNumber
!

number
    "Return the number value"

    ^number
! !


!Point class methods ! !


!Point methods !

unitVector
    "Return the unit vector"

    ^self / self radius asInteger
!

distanceTo: aPoint
    "Return the distance to aPoint"

    ^((aPoint x - x) squared + (aPoint y - y) squared) sqrt
!

normal
    "Return the normal of a vector"

    ^(y negated @ x) unitVector
!

translateBy: delta
    "Answer a new point translated by delta"

    ^(delta x + x) @ (delta y + y)
!

radius
    "Return the radius of a vector"

    ^(self dotProduct: self) sqrt
! !


!PolylineFigure class methods !

COMMENT
"
    PolylineFigure is an abstract class designed to capture figures drawn
    with one or more lines.  A PolylineFigure can be closed or open.  If
    it is closed, there is a fillColor that is used to color the area inside the
    polygon.

    A subclass of PolylineFigure must implement:
    <handles> : The figure's handles
    <handle manipulation methods> : Methods like growBy:

    InstanceVariables:
    <points> : The collection of points in the polyline
    <origin> : An integer representing the origin of the figure
    <extent> : An integer representing the extent of the figure
    <width> : An integer representing the pixel widht at wich lines are drawn
    <color> : An rgb value representing the color of the lines
    <closed> : A boolean showing if the polyline is closed; i.e. first point == last point
    <fillColor> : An rgb value representing the fill color

    ClassVariables:
    None
"
!

withPoints: aCollection
    "Return a new instance of the receiver"

    ^self new
        points: aCollection
        width: 1
        color: ClrBlack
        closed: false
        fillColor: nil
! !


!PolylineFigure methods !

basicTranslateBy: aPoint
    "Translate the receiver by aPoint"

    self points: (self points collect: [:each | each + aPoint]).
    self origin: (self origin moveBy: aPoint)
!

menuDarkGrayLine
    "Set the line color"

    self lineColor: 3.
    self container update
!

fillColor
    "Return the fillColor of the receiver"

    ^fillColor
!

calculateBoundingBox
    "Calculate the receiver's bounding box"

    self origin: ((self points inject: self points first into: [:aPoint :minPoint |
        minPoint min: aPoint]) - (self width // 2) rounded).
    self extent: ((self points inject: self points first into: [:aPoint :maxPoint |
        maxPoint max: aPoint]) + (self width // 2 + 0.5) rounded - origin)

!

polylineFillColor: anInteger
    "Set the line color to anInteger"

    fillColor := ColorTable at: anInteger.
    self changed
!

menu
    "Return the menu for a polyline figure"

    | lineWidthMenu lineColorMenu |
    lineWidthMenu := Menu
            labels: '1 pixel\2 pixel\3 pixel\4 pixel' withCrs
            lines: #()
            selectors: #(menu1PixelLine menu2PixelLine menu3PixelLine menu4PixelLine).
    lineWidthMenu title: 'line width'.
    lineColorMenu := Menu
            labels: 'black\dark gray\light gray\white' withCrs
            lines: #()
            selectors: #(menuBlackLine menuDarkGrayLine menuLightGrayLine menuWhiteLine).
    lineColorMenu title: 'line color'.
    ^Menu new
        appendSubMenu: lineWidthMenu;
        appendSubMenu: lineColorMenu




!

color: aColor
    "Set the color value of the receiver to aColor"

    color := aColor
!

origin
    "Return the origin of the receiver"

    ^origin
!

lineColor: anInteger
    "Set the line color to anInteger"

    color := ColorTable at: anInteger.
    self changed
!

lineWidth: anInteger
    "Set the border width to anInteger"

    width := anInteger.
    self changed

!

menu4PixelLine
    "Set line thickness to 4 pixels"

    self lineWidth: 4.
    self container update

!

width: anInteger
    "Set the width of the receiver to anInteger"

    width := anInteger
!

points
    "Return the points of the receiver"

    ^points
!

menu3PixelLine
    "Set line thickness to 3 pixels"

    self lineWidth: 3.
    self container update

!

menu1PixelLine
    "Set line thickness to 1 pixel"

    self lineWidth: 1.
    self container update

!

menu2PixelLine
    "Set line thickness to 2 pixel"

    self lineWidth: 2.
    self container update

!

fillColor: aColor
    "Set the fillColor of the receiver to aColor"

    fillColor := aColor
!

menuBlackLine
    "Set the line color"

    self lineColor: 1.
    self container update
!

points: aPointCollection width: anInteger color: aColorValue closed: aBoolean fillColor: anotherColorValue
    "Private - Initialize the receiver"

    points := aPointCollection.
    self width: anInteger.
    self color: aColorValue.
    self closed: aBoolean.
    self fillColor: anotherColorValue.
    self calculateBoundingBox
!

points: aCollection
    "Set the points of the receiver to aCollection"

    points := aCollection.
    self calculateBoundingBox
!

displayOn: aGraphicsMedium
    "Display the receiver onto aGraphicsMedium"

    | pen |
    pen := aGraphicsMedium pen.
    pen
        foreColor: self color;
        backColor: self fillColor;
        setLineWidth: self width.
    self closed
        ifFalse: [
            pen place: self points first.
            self points do: [:each | pen goto: each]]
        ifTrue: [
            pen polygonFilled: self points]
!

menuWhiteLine
    "Set the line color"

    self lineColor: 5.
    self container update
!

scaleBy: aPoint
    "Scale the receiver by aPoint"

    self points: (self points collect: [:each | each scaleBy: aPoint]).
    self changed
!

closed
    "Return true if the receiver is closed, else return false"

    ^closed
!

closed: aBoolean
    "Set closed to aBoolean"

    closed := aBoolean
!

menuLightGrayLine
    "Set the line color"

    self lineColor: 4.
    self container update
!

extent
    "Return the extent of the receiver"

    ^extent
!

extent: aPoint
    "Set the extent of the receiver to aPoint"

    extent := aPoint
!

changed
    "The receiver changed, recompute the bounding box"

    self calculateBoundingBox.
    super changed
!

color
    "Return the color value of the receiver"

    ^color
!

origin: aPoint
    "Set the origin of the receiver to aPoint"

    origin := aPoint
!

width
    "Return the widht of the receiver"

    ^width
! !


!PositionConstraint class methods ! !


!PositionConstraint methods !

update: aFigure
    "Update"

    receiver perform: message with: location value.
    receiver changed
!

location: aPoint receiver: aReceiver sending: aMessage
    "Private - Initialize the receiver"

    location := aPoint.
    receiver := aReceiver.
    message := aMessage
! !


!Rectangle class methods ! !


!Rectangle methods !

rightCenter
    "Answer the point at the center of the receiver's right vertical line."

    ^self right @ self center y
!

bottomLeft
    "Answer the point at the left edge of the bottom horizontal line of the receiver."

    ^self origin x @ self corner y
!

topRight
    "Answer the point at the top right corner of the receiver's top horizontal line."

    ^self corner x @ self origin y

!

leftCenter
    "Return the left center of the receiver"

    ^self left@self center y
!

bottomRight
    "Answer the point at the right edge of the bottom horizontal line of the receiver."

    ^self corner
!

topCenter
    "Answer the point at the center of the receiver's top horizontal line."

    ^((self origin x + self corner x) // 2) @ self origin y
!

contains: aRectangle
    "Answer whether aRectangle is contained whithin the receiver"

    ^(self containsPoint: aRectangle origin) and: [
        self containsPoint: aRectangle corner]
!

bottomCenter
    "Answer the point at the center of the bottom horizontal line of the receiver."

    ^((self origin x + self corner x) // 2) @ self corner y
!

topLeft
    "Answer the point at the top left corner of the receiver's top horizontal line."

    ^self origin
! !


!RectangleFigure class methods !

createNotifying: aPane
    "Return a new instance of the receiver, a rectangle figure.
    aPane is updated dynamically; i.e. while the figure is growing"

    | aFigure |
    aFigure := self rectangle: (aPane mouseLocation extent: 0@0).
    aPane addFigure: aFigure.
    aFigure growNotifying: aPane.
    aPane removeFigure: aFigure.
    ^aFigure
!

COMMENT
"
    RectangleFigure is the concrete subclass which represents a Rectangular region.
    The four points that make up the RectangleFigure are kept in the points instance
    variable of PolylineFigure.  The order of these points is topLeft, topRight,
    bottomRight, and bottomLeft.  This order is used on occasion in the methods of
    RectangleFigure.

    Instance Variables:
    None

    Class Variables:
    None
"
!

creationTool
    "Return a creation tool"

    ^CreationTool
        icon: (HotDrawLibrary at: #RectangleBitmap)
        class: self

!

rectangle: aRectangle
    "Return a new instance of the receiver"

    ^(self new
        points: (Array with: 0@0)
        width: 1
        color: ClrBlack
        closed: true
        fillColor: ClrWhite)
    setRectangle: aRectangle
! !


!RectangleFigure methods !

rectangle: aRectangle width: anInteger color: aColorValue closed: aBoolean fillColor: anotherColorValue
    "Private - initialize the receiver"

    points := aRectangle asPolyline asOrderedCollection.
    width := anInteger.
    color := aColorValue.
    closed := aBoolean.
    fillColor := anotherColorValue.
    self calculateBoundingBox
!

borderWidthBy: anInteger
    "Modify the border width by anInteger"

    self width: ((width + (anInteger // 4) min: 25) max: 1)
!

menuLightGrayFill
    "Set the fill color"

    self polylineFillColor: 4.
    self container update



!

darkenBy: anInteger
    "Darken the interior by anInteger"

    | index |
    index := ColorTable indexOf: fillColor.
    index := (index + ((anInteger / 5) truncated) min: ColorTable size) max: 1.
    self fillColor: (ColorTable at: index).
    self changed
!

menuDarkGrayFill
    "Set the fill color"

    self polylineFillColor: 3.
    self container update



!

handles
    "Return a collection of handles"

    ^(super handles)
        add: (SelectionTrackHandle colorOf: self);
        add: (SelectionTrackHandle borderColorOf: self);
        add: (SelectionTrackHandle widthOf: self);
        add: (ConnectionHandle on: self at: #center);
        yourself
!

growNotifying: aPane
    "Grow the receiver and notify aPane afterwards"

    | aHandle |
    aHandle := TrackHandle bottomRightOf: self.
    aHandle invoke: aPane
!

borderDarkenBy: anInteger
    "Darken border by anInteger"

    | index |
    index := ColorTable indexOf: self color.
    index := (index + (anInteger / 5) truncated min: ColorTable size) max: 1.
    self color: (ColorTable at: index)
!

growBy: aRectangle
    "Grow the receiver"

    self setRectangle: (aRectangle origin + self points first corner: aRectangle corner + (self points at: 3)).
    self changed
!

menuBlackFill
    "Set the fill color"

    self polylineFillColor: 1.
    self container update



!

setRectangle: aRectangle
    "Private - initialize the receiver"

    | collection |
    collection := OrderedCollection
        with: aRectangle origin
        with: aRectangle rightTop
        with: aRectangle corner
        with: aRectangle leftBottom.
    self points: collection
!

menuWhiteFill
    "Set the fill color"

    self polylineFillColor: 5.
    self container update



!

menu
    "Return the menu for a polyline figure"

    | lineWidthMenu lineColorMenu fillColorMenu |
    lineWidthMenu := Menu
            labels: '1 pixel\2 pixel\3 pixel\4 pixel' withCrs
            lines: #()
            selectors: #(menu1PixelLine menu2PixelLine menu3PixelLine menu4PixelLine).
    lineWidthMenu title: 'line width'.
    lineColorMenu := Menu
            labels: 'black\dark gray\light gray\white' withCrs
            lines: #()
            selectors: #(menuBlackLine menuDarkGrayLine menuLightGrayLine menuWhiteLine).
    lineColorMenu title: 'line color'.
    fillColorMenu := Menu
            labels: 'black\dark gray\light gray\white' withCrs
            lines: #()
            selectors: #(menuBlackFill menuDarkGrayFill menuLightGrayFill menuWhiteFill).
    fillColorMenu title: 'fill color'.
    ^Menu new
        appendSubMenu: lineWidthMenu;
        appendSubMenu: lineColorMenu;
        appendSubMenu: fillColorMenu




! !


!ScrollingTool class methods !

COMMENT
"
    A ScrollingTool is a Tool which permits the drawing to
    be interactively scrolled. When pressed on the drawing,
    the drawing is told to scroll by the appropriate amount.

    Instance Variables:
    None

    Class Variables:
    None
"
!

new
    "Return a new instance of the receiver"

    ^self icon: (HotDrawLibrary at: #ScrollBitmap)

! !


!ScrollingTool methods !

press
    "The tool has been pressed"

    | oldPoint newPoint |
    oldPoint := self drawingPane mouseLocation.
    Notifier consumeInputUntil: [:event |
        newPoint := self drawingPane mouseLocation.
        newPoint ~= oldPoint ifTrue: [
            self drawingPane scrollBy: newPoint - oldPoint.
            self drawingPane update].
        oldPoint := newPoint.
        event selector = #button1Up:].
! !


!SelectionTool class methods !

COMMENT
"
    A SelectionTool is a Tool which is used to make a selection
    of one or more figures in a drawing. If pressed while over
    a figure, the figure is added to the current selection of the
    drawing. If the left shift key is held during the button presses,
    the selection is toggled. If the SelectionTool is pressed in
    the background, a travelling marquee is drawn and is used
    to enclose the desired figures.

    Instance Variables:
    None

    Class Variables:
    None
"
!

new
    "Return a new instance of the class"

    ^self icon: (HotDrawLibrary at: #SelectionBitmap)
! !


!SelectionTool methods !

pressBackground
    "The user has pressed the tool in the background"

    self selectGroup.
    self drawingPane update
!

selectFigure: aFigure
    "Select aFigure"

    (Notifier isKeyDown: VkShift)
        ifTrue: [self drawingPane toggleSelection: aFigure]
        ifFalse: [
            (self drawingPane isSelected: aFigure) ifFalse: [
                self drawingPane selection: aFigure.
                self drawingPane update]]
!

pressFigure: aFigure
    "The user has pressed the tool on top of another figure"

    (aFigure isActive and: [(self drawingPane isSelected: aFigure) not])
        ifTrue: [self pressHandle: aFigure]
        ifFalse: [
            self selectFigure: aFigure.
            self moveFigure: aFigure]
!

marqueeFromUser
    "Answer a Rectangle of the size selected by the user."

    | origin corner rect aPoint pen |
    Notifier activeMainWindow captureMouseInput.
    pen := self drawingPane pen.
    pen setRop2: R2Notxorpen.
    origin := corner := self drawingPane mouseLocation.
    Notifier consumeInputUntil: [: event |
        event selector = #button1Move:
            ifTrue: [
                aPoint := self drawingPane mouseLocation.
                corner  ~= aPoint
                    ifTrue: [
                        pen
                            place: origin;
                            box: corner;
                            place: origin;
                            box: aPoint.
                        corner := aPoint]].
        event selector = #button1Up:].
    pen
        place: origin;
        box: corner;
        setRop2: R2Copypen.
    UserLibrary releaseCapture.
    ^self properMarquee: (origin corner: corner)



!

properMarquee: aRectangle
    "Return a real rectangle"

    | originX originY cornerX cornerY |
    originX := aRectangle origin x min: aRectangle corner x.
    originY := aRectangle origin y min: aRectangle corner y.
    cornerX := aRectangle origin x max: aRectangle corner x.
    cornerY := aRectangle origin y max: aRectangle corner y.
    ^originX@originY corner: cornerX@cornerY



!

selectGroup
    "Select figures using the marquee"

    | aRectangle aCollection |
    self drawingPane noSelections.
    aRectangle := self marqueeFromUser.
    aCollection := self drawingPane figuresIn: aRectangle.
    (Notifier isKeyDown: VkShift)
        ifFalse: [self drawingPane selections: aCollection]
        ifTrue: [self drawingPane toggleSelections: aCollection]
!

moveFigure: aFigure
    "Move aFigure"

    self pressHandle: (SelectionTrackHandle positionOf: aFigure)
!

pressHandle: aHandle
    "The user has pressed the tool on top of a handle"

    ^aHandle invoke: self drawingPane
! !


!SelectionTrackHandle class methods ! !


!SelectionTrackHandle methods ! !


!TextCreationTool class methods !

COMMENT
"
    A TextCreationTool has the posibility of accepting typed text.

    Instance Variables:
    <typingTarget> : A TextFigure

    Class Variables:
    None
"
! !


!TextCreationTool methods !

backspace
    "User pressed backspace, erase last character typed"

    self typingTarget text: (
        self typingTarget text copyFrom: 1 to: (self typingTarget text size - 1)).
    self drawingPane update

!

pressBackground
    "User pressed the tool in the background of the drawing"

    | aFigure |
    aFigure := self createFigure.
    self typingTarget: aFigure.
    self drawingPane selection: aFigure.
    self drawingPane update
!

pressFigure: aFigure
    "User pressed the tool on top of another figure"

    aFigure acceptsTyping
        ifTrue: [
            self typingTarget: aFigure.
            self drawingPane selection: aFigure.
            self drawingPane update]
        ifFalse: [self pressBackground]
!

characterInput: aChar
    "User entered aChar, append aChar"

    self typingTarget isNil ifTrue: [^self ].
    self typingTarget text: (self typingTarget text, aChar asString).
    self drawingPane update

!

typingTarget: aFigure
    "Set the text figure to aFigure"

    typingTarget := aFigure
!

typingTarget
    "Return a text figure"

    ^typingTarget
! !


!TextFigure class methods !

createNotifying: aPane
    "Return a new instance of the receiver, a text figure.
    aPane is updated dynamically; i.e. while the figure is growing"

    ^self text: String new at: aPane mouseLocation
!

COMMENT
"
    A TextFigure is a figure that permits the displaying of text in a drawing.

    Instance Variables:
    <origin> : The starting point to display the text
    <text> : A string containing the information to be displayed
    <readOnly> : A boolean that indicates whether the text can be edited or not
    <font> : A font

    Class Variables:
    None
"
!

creationTool
    "Return a creation tool"

    ^TextCreationTool
        icon: (HotDrawLibrary at: #TextBitmap)
        class: self

!

text: aString at: aPoint
    "Return a new instance of the class"

    ^self new text: aString origin: aPoint
! !


!TextFigure methods !

menuFont
    "Menu selector for the font"

    | newFont |
    newFont := Font chooseAFont: 'Select a font !!'.
    newFont isNil ifTrue: [^nil].
    font := newFont.
    self container update
!

extent
    "Return the extent of receiver"

    ^(self font stringWidth: self text)@self font height

!

changeFontSizeBy: aPoint
    "Change the current font size by aPoint"
    "Do nothing, handled from the menu"
!

text: aString
    "Set the text to aString"

    text := aString.
    self changed
!

font
    "Return the font of the receiver"

    ^font
!

text
    "Return the text"

    ^text
!

readOnly
    "Return a boolean"

    ^readOnly
!

fontSizeHandleLocation
    "Return the location of the font size handle"

    ^(self origin x + self extent x + 5)@ self center y

!

font: aFont
    "Set the font of the receiver"

    font := aFont
!

displayOn: aGraphicsMedium
    "Display the receiver on aGraphicsMedium"

    | aPen |
    aPen := aGraphicsMedium pen.
    aPen
        foreColor: ClrBlack;
        backColor: ClrWhite;
        font: self font;
        place: self origin;
        setTextAlign: TaTop;
        displayText: self text


!

text: aString origin: aPoint
    "Private - initialize the receiver"

    origin := aPoint.
    text := aString.
    font := SysFont.
    readOnly := false.
!

handles
    "Return a collection of handles"

    ^Array
        with: (TrackHandle
            on: self
            at: #fontSizeHandleLocation
            change: #changeFontSizeBy:)
        with: (TrackHandle
            on: self
            at: #fontHandleLocation
            change: #changeFontBy:)
!

changeFontBy: aPoint
    "Change the current font by aPoint"
    "Do nothing, handled from the menu"



!

fontHandleLocation
    "Return the location of the font handle"

    ^self center x@ (origin y + self extent y)
!

acceptsTyping
    "Returns true if the receiver accepts typing, else return false"

    ^readOnly not
!

readOnly: aBoolean
    "Set text read only or not"

    readOnly := aBoolean
!

menu
    "Return the corresponding menu"

    ^Menu
        labels: 'font' withCrs
        lines: #()
        selectors: #(menuFont)
!

basicTranslateBy: aPoint
    "Translate the receiver by aPoint"

    self origin: (self origin moveBy: aPoint)


!

origin: aPoint
    "Set the origin of the receiver to aPoint"

    origin := aPoint
!

origin
    "Return the origin of the receiver"

    ^origin
! !


!Tool class methods !

toolIconSpacing
    "Answer a Point which is the size between adjacent tool icons"

    ^2@2
!

toolImageSize
    "Answer a Point which is the size of my icon"

    ^16@16
!

COMMENT
"
    A Tool is an abstract class capturing the behaviour of a drawing tool.

    Instance Variables:
    <icon> : The bitmap showing in the tool palette
    <drawingPane> : The drawing pane containing the current drawing

    Class Variables:
    None
"
!

icon: anImage
    "Return a new instance of the receiver"

    ^super new setIcon: anImage
!

toolHiliteSize
    "Answer a point which is the size of the border around the receiver icon when selected"

    ^4@4
! !


!Tool methods !

icon
    "Return the icon of the receiver. The icon is a bitmap"

    ^icon
!

canCutCopyPaste
    "Return false"

    ^false
!

drawingPane
    "Return the drawing pane"

    ^drawingPane
!

backspace
    "Do nothing"
!

icon: aBitmap
    "Set the icon of the receiver"

    icon := aBitmap
!

displayOn: aGraphicsMedium
    "Display the receiver on aGraphicsMedium"

    | pen |
    pen := aGraphicsMedium pen.
    pen
        copyBitmap: self icon
        from: self icon boundingBox
        at: pen location
!

displayOn: aGraphicsMedium at: aPoint
    "Display the receiver on aGraphicsMedium at aPoint"

    aGraphicsMedium pen place: aPoint.
    self displayOn: aGraphicsMedium
!

deactivate
    "Do nothing"
!

figureAtCursor
    "Return the figure under the cursor in the drawing pane"

    ^self drawingPane figureAt: self drawingPane mouseLocation
!

drawingPane: aDrawingPane
    "Set the drawing pane"

    drawingPane := aDrawingPane
!

activate
    "This message is sent when a tool is selected. The default
    behavior is that changing tools cancels any selections"

    self drawingPane noSelections
!

characterInput: aChar
    "Do nothing"
!

setIcon: aBitmap
    "Private - initialize the receiver"

    self icon: aBitmap
!

press
    "The user pressed the tool in the drawing pane"

    | aFigure |
    aFigure := self figureAtCursor.
    aFigure isNil
        ifTrue: [self pressBackground]
        ifFalse: [self pressFigure: aFigure]
! !


!ToolPalettePane class methods !

on: aDrawingEditor
    "Return a new instance of the receiver"

    ^self new editor: aDrawingEditor
! !


!ToolPalettePane methods !

currentTool
    "Return the editor's current tool"

    ^self editor currentTool
!

bufferTools
    "Private - Display the tools on the buffer"

    | position |
    position := Tool toolIconSpacing + (Tool toolHiliteSize / 2) + (2@2).
    self tools do: [:tool |
        tool = self currentTool
            ifTrue: [
                self buffer pen
                    backColor: ClrBlack;
                    place: (position - Tool toolIconSpacing);
                    boxFilled: (position - Tool toolIconSpacing + Tool toolImageSize + Tool toolHiliteSize)].
        tool displayOn: self buffer at: position.
        position := position + (0@((Tool toolImageSize + Tool toolHiliteSize) y + Tool toolIconSpacing y))].


!

bufferFrame
    "Private - Display the frame around the tools"

    self buffer pen
        fill: ClrLightgray;
        foreColor: ClrWhite;
        backColor: ClrWhite;
        place: 1@1;
        boxFilled: self rectangle corner - (3@3);
        foreColor: ClrBlack;
        backColor: ClrBlack;
        place: 2@2;
        boxFilled: self rectangle corner - (1@1);
        foreColor: ClrLightgray;
        backColor: ClrLightgray;
        place: 2@2;
        boxFilled: self rectangle corner - (2@2)
!

currentTool: aTool
    "Set the editor's current tool to aTool"

    self editor currentTool deactivate.
    self editor currentTool: aTool.
    aTool activate
!

button1Down: aPoint
    "Select the tool under the mouse cursor and highlight it"

    | aTool |
    aTool := self toolAt: self mouseLocation.
    (aTool isNil not and: [(aTool = self currentTool) not]) ifTrue: [
        self currentTool: aTool].
    self update

!

createBuffer
    "Fill the cache"

    super createBuffer.
    self bufferFrame.
    self bufferTools

!

toolAt: aPoint
    "Answer the tool icon located at aPoint or nil if there is none"

    | position area |
    position := Tool toolIconSpacing + (Tool toolHiliteSize / 2).
    self tools do: [:each |
        area := Rectangle origin: position  extent: Tool toolImageSize.
        (area containsPoint: aPoint) ifTrue: [^each].
        position := position + (0 @ ((Tool toolImageSize + Tool toolHiliteSize) y + Tool toolIconSpacing y))].
    ^nil

!

tools
    "Return the editor tools"

    ^self editor tools
!

tools: aCollection
    "Set the editor tools"

    self editor tools: aCollection
! !


!TrackHandle class methods !

on: aFigure at: aSymbol with: anArgument change: changeSelector
    "Return a new instance of the receiver"

    ^(self on: aFigure at: aSymbol with: anArgument)
                setSense: nil change: changeSelector
!

bottomRightOf: aFigure
    "Return the bottomRight handle of aFigure"

    ^self
        on: aFigure
        at: #bottomRight
        sense: #senseBottomRight:
        change: #growBy:
!

bottomLeftOf: aFigure
    "Return the leftBottom handle of aFigure"

    ^self
        on: aFigure
        at: #bottomLeft
        sense: #senseBottomLeft:
        change: #growBy:
!

widthOf: aFigure
    "Return a handle used to change the border widht of aFigure"

    ^self
        on: aFigure
        at: #offCenter:
        with: -10@0
        sense: #senseColor:
        change: #borderWidthBy:
!

topLeftOf: aFigure
    "Return the top left handle for aFigure"

    ^self
        on: aFigure
        at: #topLeft
        sense: #senseTopLeft:
        change: #growBy:
!

positionOf: aFigure
    "Return a handle"

    ^self
        on: aFigure
        at: #center
        change: #translateBy:
!

COMMENT
"
    A TrackHandle is a handle used to make changes to a figure
    by direct manipulation of the TrackHandle. The TrackHandle,
    using its Locator, senses a certain aspect of the figure and
    uses the information to change the figure.

    Instance Variables:
    <sense> : Method sent to the figure to extract the desired aspect
    <change> : Method sent to the figure to change the desired aspect

    Class Variables:
    None
"
!

on: aFigure at: aSymbol sense: senseSelector change: changeSelector
    "Return a new instance of the receiver"

    ^(self on: aFigure at: aSymbol)
                setSense: senseSelector change: changeSelector
!

topRightOf: aFigure
    "Return the top right handle for aFigure"

    ^self
        on: aFigure
        at: #topRight
        sense: #senseTopRight:
        change: #growBy:
!

colorOf: aFigure
    "Return a handle used to change the fill color of aFigure"

    ^self
        on: aFigure
        at: #offCenter:
        with: 10@0
        sense: #senseColor:
        change: #darkenBy:
!

allCornersOf: aFigure
    "Return a collection with corner handles for aFigure"

    ^OrderedCollection new
            add: (self topLeftOf: aFigure);
            add: (self topRightOf: aFigure);
            add: (self bottomLeftOf: aFigure);
            add: (self bottomRightOf: aFigure);
            yourself
!

on: aFigure at: aSymbol with: anArgument sense: senseSelector change: changeSelector
    "Return a new instance of the receiver"

    ^(self on: aFigure at: aSymbol with: anArgument)
                setSense: senseSelector change: changeSelector
!

borderColorOf: aFigure
    "Return a handle used to change the border color of aFigure"

    ^self
        on: aFigure
        at: #offCenter:
        with: 0@10
        sense: #senseColor:
        change: #borderDarkenBy:
!

on: aFigure at: aSymbol change: changeSelector
    "Return a new instance of the receiver"

    ^(self on: aFigure at: aSymbol)
                setSense: nil change: changeSelector
! !


!TrackHandle methods !

change: aFigure by: anObject
    ^change notNil ifTrue: [ aFigure perform: change with: anObject ]
!

invokeStep: deltaPoint
    | anObject |
    anObject := self sense: deltaPoint.
    anObject notNil ifTrue: [self change: self owner by: anObject]
!

sense: deltaPoint
    ^sense isNil
        ifTrue: [deltaPoint]
        ifFalse: [self owner perform: sense with: deltaPoint]
!

changeBy: aFigure by: anObject
    ^change notNil ifTrue: [ aFigure perform: change with: anObject ]
!

setSense: senseSelector change: changeSelector
    sense := senseSelector.
    change := changeSelector
! !


Figure initialize! 
              
Transcript cr; tab; show: 'HotDraw/V R1.27, Sunday 25 September 1994'.
Transcript cr; tab; show: 'DrawingEditor open' !
