Q   Courier10.Scn.Fnt  n  Oberon10.Scn.Fnt              g     (************************************************************************

Name          : BTasks
Zweck         : Cooperating background tasks
Version       : 2.3
Vorgaenger	: 2.2
Aenderungen   : log output only if name#""
Status, Bugs  : 
Autor         : Frank Hrebabetzky, M.A.McGaw
Computer      : PC >=386
Compiler      : Oberon
Betriebssystem: Oberon 3 Version 2.3.6, as modified to System 7
Datum         : September, 2000

************************************************************************)

MODULE BTasks;

(**The procedure Install installs and starts BTaskProc as a background task. 
   Thus it will be called automatically and repetitively by the operating 
   system. The client module could have the following structure:

   MODULE Client;

   IMPORT BTasks;
   
   TYPE CSD = ...;	/* client specific data */
   
   VAR  csd: CSD;
        bt : BTasks.BTask;

   PROCEDURE init (VAR csd:CSD);
   BEGIN ... END init;

   PROCEDURE work (VAR csd:CSD);
   BEGIN ... END work;

   PROCEDURE stop (csd:CSD): BOOLEAN;
   BEGIN ... END stop;

   PROCEDURE BTaskProc;
   BEGIN
     IF stop(csd) THEN BTasks.Remove(bt) END;
     /* could be removed automatically too with Install parameters */
     work (csd);
   END BTaskProc;

   PROCEDURE Execute*;
   VAR bt: BTasks.BTask;
   BEGIN
     init (csd);
     bt:= BTasks.Install (BTaskProc, "Test", 1000, 0, 0, 100);
   END Execute;

   END Client.

   work should have a short execution time in order not to freeze the system.
   The formal parameters of Install are:

     BTaskProc: Procedure to be installed as background task;
     Name     : Appears in installation and de-inst. message (System.Log);
     Dt       : waiting time in ms (accuracy: 10/3 ms)
      between successive calls
                of BTaskProc by the operating system.
                 Dt=0 is allowed.
   After excess of one of the limits
     tTotMax : time in ms since installation (accuracy: 10/3 ms),
     tCalcMax: total calculation time in ms (accuracy: 10/3 ms),
     NCallMax: no. of calls,
   BTaskProc will automatically be de-installed. To deactivate this feature,
   the corresponding parameter has to be set to 0.

   bt=NIL may serve as check whether the task is already de-installed.
*)


IMPORT 
  Oberon, Out;

CONST
  NameLen* = 31;

TYPE   
  CycleIndicator* = PROCEDURE;
  BProc*    = PROCEDURE;
  BTask*    = POINTER TO BTaskDesc;
  BTaskDesc*= RECORD (Oberon.TaskDesc)
                proc    : BProc;
                name    : ARRAY NameLen+1 OF CHAR;
                tBegin, tEnd,     (* last call of Wait, CheckTask *)
                tNext, Dt,        (* next call, 
                waiting time      *)
                tStop,            (* absolute time limit          *)
                tCalc, tCalcM,    (* calculation time and ~ limit *)
                NCall, NCallM     (* no. of calls and ~ limit     *)
                        : LONGINT;
              END;

VAR
  count*, n		: INTEGER;
  tmeas*, tlast	: LONGINT;	(** tmeas: cycle frequ. meas. time in ms *)
  measbt           : BTask;	(* background task for meas. cycle freq. *)
  CycleInd*		: CycleIndicator;
  

PROCEDURE Remove* (bt:BTask);
(** Remove background task. bt will be set to NIL. Call with NIL argument doesn't
   harm. *)
BEGIN
  IF bt#NIL THEN
    Oberon.BRemove (bt);
    IF bt.name[0]#0X THEN
      Out.String("task '");      Out.String(bt.name);
      Out.String("' removed");   Out.Ln;
    END;
    bt:= NIL;
  END
END Remove;



PROCEDURE wait(bt:BTask): BOOLEAN;
(* Wait if it isn't yet the turn of bt. *)
BEGIN
  bt.tBegin:= Oberon.Time();
  IF bt.tBegin<bt.tNext THEN RETURN TRUE
  ELSE
    bt.tNext:= bt.tBegin + bt.Dt;
    RETURN FALSE;
  END;
END wait;



PROCEDURE check (bt:BTask);
(* Check whether bt is to be removed and remove then. *)
BEGIN
  bt.tEnd:= Oberon.Time();
  IF (bt.tCalc>=bt.tCalcM) OR (bt.tEnd>=bt.tStop) OR (bt.NCall>=bt.NCallM)
  THEN Remove(bt) END;
  INC (bt.tCalc, bt.tEnd-bt.tBegin);
  INC (bt.NCall);
END check;



PROCEDURE TaskFrame (t:Oberon.Task);
BEGIN
  WITH t:BTask DO
    IF wait(t) THEN RETURN END;
    check(t);
    t.proc;
  END;
END TaskFrame;



PROCEDURE Install* (BTaskProc:BProc; Name:ARRAY OF CHAR;
                    Dt,tTotMax,tCalcMax,NCallMax: LONGINT): BTask;
(** Install BTaskProc as background procedure. Meaning of parameters see above. *)
VAR bt: BTask;
BEGIN
  NEW(bt);
  bt.proc := BTaskProc;
  bt.tNext:= Oberon.Time();   (* immediately *)
  bt.Dt   := Dt;
  IF tTotMax<=0  THEN bt.tStop:= MAX(LONGINT)
                 ELSE bt.tStop:= bt.tNext + tTotMax
  END;
  bt.tCalc:= 0;
  IF tCalcMax<=0 THEN bt.tCalcM:= MAX(LONGINT)
                 ELSE bt.tCalcM:= tCalcMax
  END;
  bt.NCall:= 0;
  IF NCallMax<=0 THEN bt.NCallM:= MAX(LONGINT)
                 ELSE bt.NCallM:= NCallMax
  END;
  COPY (Name, bt.name);
  bt.handle:= TaskFrame;
  bt.safe  := FALSE;
  Oberon.BInstall (bt);
  IF bt.name[0]#0X THEN
    Out.String("task '");   Out.String(bt.name);
    Out.String("' installed");   Out.Ln;
  END;
  RETURN bt;
END Install;



PROCEDURE Measure;
(* Count main event loop cycles during tmeas [ms] and store the result in count.
   Call CycleInd once each tmeas. Here the client can define the display method. *)
BEGIN
  INC(n);
  IF Oberon.Time()-tlast>=tmeas THEN   
  (* 1000 ticks/s *)
    count:= n;   CycleInd;   n:=0;   tlast:= Oberon.Time();
  END;
END Measure;


PROCEDURE InitMeasure*;
(**Install background task for main event loop cycle frequency measurement.
   Result: count cycles during tmeas [ms]. Call CycleInd once each tmeas. Here
   the client can define the display method. *)
BEGIN
  NEW(measbt);   n:=0;   tlast:= Oberon.Time();
  measbt:= Install (Measure, "Event Loop Frequency", 0, 0, 0, 0);
END InitMeasure;


PROCEDURE FinitMeasure*;
(**De-install main event loop cycle frequency measurement. *)
BEGIN   Remove(measbt)
END FinitMeasure;


PROCEDURE NoIndicator;
BEGIN
END NoIndicator;


PROCEDURE SimpleIndicator;
BEGIN   Out.Int (LONG(count)*1000 DIV tmeas, 4);   Out.Ln;
END SimpleIndicator;


PROCEDURE CycleFreq*;
BEGIN
  Out.Ln;   Out.String("Main event loop cycle frequency measurement.");   Out.Ln;
  Out.String("To be stopped by BTasks.FinitMeasure.");   Out.Ln;
  Out.String("Cycles per second:");   Out.Ln;
  tmeas:= 1000;   CycleInd:= SimpleIndicator;
  InitMeasure;
END CycleFreq;
  


BEGIN
  CycleInd:= NoIndicator;
END BTasks.


BTasks.CycleFreq;
BTasks.FinitMeasure;