MODULE RAWrite;  (* pjm *)

(* Write and verify a file to a 1.44Mb diskette *)

IMPORT IO, FIO, Lib, SYSTEM;
FROM Storage IMPORT ALLOCATE;

CONST
  TrackSize = 18;
  BufSize = TrackSize*512;

PROCEDURE AllocBuffer(VAR buf: ADDRESS);
BEGIN
  REPEAT
    ALLOCATE(buf, BufSize)
  UNTIL Seg(buf^) DIV 1000H = (Seg(buf^)+(BufSize+15) DIV 16-1) DIV 1000H
  (* buffer does not cross a 64k boundary *)
END AllocBuffer;

PROCEDURE Read(f: FIO.File;  buf: ADDRESS;  VAR numsec: CARDINAL);
VAR miss: CARDINAL;
BEGIN
  numsec := FIO.RdBin(f, buf^, BufSize);
  IF numsec MOD 512 # 0 THEN
    miss := 512-(numsec MOD 512);
    Lib.Fill(Lib.AddAddr(ADR(buf^), numsec-(numsec MOD 512)), miss, 0)
  END;
  numsec := (numsec+511) DIV 512
END Read;

PROCEDURE Write(dst: CHAR;  buf: ADDRESS;  pos, numsec, gran: CARDINAL): BOOLEAN;
VAR r: SYSTEM.Registers;  OK: BOOLEAN;  try, ofs: CARDINAL;  drive: SHORTCARD;
BEGIN
  drive := VAL(SHORTCARD, ORD(dst)-ORD('A'));
  IF drive > 3 THEN HALT END;
  IF gran > numsec THEN gran := numsec END;
  ofs := 0;
  REPEAT
    try := 3;
    REPEAT
      r.AH := 3;  r.AL := VAL(SHORTCARD, gran);  (* number of sectors to write *)
      r.CH := VAL(SHORTCARD, pos DIV TrackSize DIV 2);  (* track number *)
      r.CL := 1 + VAL(SHORTCARD, pos MOD TrackSize);  (* sector number *)
      r.DH := VAL(SHORTCARD, pos DIV TrackSize MOD 2);  (* head number *)
      r.DL := drive;  r.ES := Seg(buf^);  r.BX := Ofs(buf^)+ofs;
      Lib.Intr(r, 13H);
      OK := NOT (0 IN r.Flags);
      IF NOT OK THEN
        r.AH := 0;  r.DL := drive;
        Lib.Intr(r, 13H)
      END;
      DEC(try)
    UNTIL OK OR (try = 0);
    IF NOT OK THEN RETURN FALSE END;
    INC(pos, gran);  INC(ofs, gran*512);  DEC(numsec, gran)
  UNTIL numsec = 0;
  RETURN TRUE
END Write;

PROCEDURE Verify(dst: CHAR;  buf1, buf2: ADDRESS;  pos, numsec, gran: CARDINAL): BOOLEAN;
VAR r: SYSTEM.Registers;  OK: BOOLEAN;  try, ofs: CARDINAL;  drive: SHORTCARD;
BEGIN
  drive := VAL(SHORTCARD, ORD(dst)-ORD('A'));
  IF drive > 3 THEN HALT END;
  IF gran > numsec THEN gran := numsec END;
  ofs := 0;
  REPEAT
    try := 3;
    REPEAT
      r.AH := 2;  r.AL := VAL(SHORTCARD, gran);  (* number of sectors to read *)
      r.CH := VAL(SHORTCARD, pos DIV TrackSize DIV 2);  (* track number *)
      r.CL := 1 + VAL(SHORTCARD, pos MOD TrackSize);  (* sector number *)
      r.DH := VAL(SHORTCARD, pos DIV TrackSize MOD 2);  (* head number *)
      r.DL := drive;  r.ES := Seg(buf2^);  r.BX := Ofs(buf2^)+ofs;
      Lib.Intr(r, 13H);
      OK := NOT (0 IN r.Flags);
      IF NOT OK THEN
        r.AH := 0;  r.DL := drive;
        Lib.Intr(r, 13H)
      END;
      DEC(try)
    UNTIL OK OR (try = 0);
    IF NOT OK THEN RETURN FALSE END;
    INC(pos, gran);  INC(ofs, gran*512);  DEC(numsec, gran)
  UNTIL numsec = 0;
  RETURN Lib.Compare(ADR(buf1^), ADR(buf2^), ofs) = ofs
END Verify;

PROCEDURE Do;
VAR
  buf1, buf2: ADDRESS;  src: ARRAY [0..64] OF CHAR;  dst, yn: CHAR;
  f: FIO.File;  gran, pos, numsec: CARDINAL;  OK: BOOLEAN;
  total: LONGCARD;
BEGIN
  IO.WrStr("Native Oberon RAWrite 31.10.97 (http://www.oberon.ethz.ch/native/)");  IO.WrLn;
  IO.WrStr("Write a diskette image to a 1.44Mb diskette.  May be distributed freely.");  IO.WrLn;
  IO.WrLn;
  AllocBuffer(buf1);  AllocBuffer(buf2);
  IO.WrStr("Diskette image file: ");  IO.RdStr(src);  IO.WrLn;
  IF src[0] = 0C THEN RETURN END;
  REPEAT
    IO.WrStr("Drive to write to (A or B): ");  dst := IO.RdKey();
    dst := CAP(dst);
    IF dst = CHR(27) THEN RETURN END;
    IO.WrChar(dst);  IO.WrLn
  UNTIL (dst = "A") OR (dst = "B");
  IO.WrLn;
    (* write *)
  gran := TrackSize;  FIO.IOcheck := TRUE;
  pos := 0;  f := FIO.OpenRead(src);
  total := (FIO.Size(f)+511) DIV 512;
  LOOP
    Read(f, buf1, numsec);
    IF numsec = 0 THEN EXIT END;
    WHILE NOT Write(dst, buf1, pos, numsec, gran) DO
      IF gran = 1 THEN
        IO.WrLn;  IO.WrStr("Error: Can not write to diskette ");
        IO.WrChar(dst);  IO.WrLn;
        HALT
      ELSE
        gran := 1;  IO.WrLn;
        IO.WrStr("Switching to single-sector mode");  IO.WrLn
      END
    END;
    INC(pos, numsec);
    IO.WrChar(CHR(13));  IO.WrLngCard(VAL(LONGCARD, pos)*100 DIV total, 3);
    IO.WrStr("% written")
  END;
  IO.WrLn;  IO.WrLn;
  FIO.Close(f);
  REPEAT
    IO.WrStr("Verify the written diskette? (Y or N): ");  yn := IO.RdKey();
    yn := CAP(yn);
    IF yn = CHR(27) THEN RETURN END;
    IO.WrChar(yn);  IO.WrLn;
    IF yn = "N" THEN RETURN END
  UNTIL yn = "Y";
  IO.WrLn;
    (* verify *)
  gran := TrackSize;
  pos := 0;  f := FIO.OpenRead(src);
  total := (FIO.Size(f)+511) DIV 512;
  LOOP
    Read(f, buf1, numsec);
    IF numsec = 0 THEN EXIT END;
    REPEAT
      OK := Verify(dst, buf1, buf2, pos, numsec, gran);
      IF NOT OK THEN
        IF gran = 1 THEN
          IO.WrLn;  IO.WrStr("Error: Verify error on diskette ");
          IO.WrChar(dst);  IO.WrLn;
          HALT
        ELSE
          gran := 1;  IO.WrLn;
          IO.WrStr("Switching to single-sector mode");  IO.WrLn
        END
      END
    UNTIL OK;
    INC(pos, numsec);
    IO.WrChar(CHR(13));  IO.WrLngCard(VAL(LONGCARD, pos)*100 DIV total, 3);
    IO.WrStr("% verified")
  END;
  IO.WrLn;
  FIO.Close(f)
END Do;

BEGIN
  Do
END RAWrite.
