MODULE noBoot;  (* bes, 05 May 96 - 12 May 96 *)

  (* based on Gneiss bootstrap loader for DOS (small model) *)
  (* written by P.J. Muller *)

(* 23.01.97 pjm release 2.2 *)
(* 27.01.97 pjm kernel above 1Mb *)
(* 27.01.97 pjm removed check for Init *)
(* 28.01.97 pjm fixed errors *)
(* 12.02.97 pjm machine core rewritten *)
(* 02.04.97 pjm APM disable option *)
(* 14.08.97 pjm clear ebp for kernel, usage simpler *)
(* 15.10.97 pjm shift support *)
(* 10.11.97 pjm removed cpu check - Lib.CpuId does not always work *)

IMPORT
  Boot, BiosIO, IO, FIO, Lib, Str, SYSTEM;

CONST
  ProgramName = "noboot";
  Version = "2.2 (10.11.97)";
  Max = 512;
  DisableIRQs = TRUE;

TYPE
  Byte = SHORTCARD;  (*  8 bits *)
  Word = CARDINAL;  (* 16 bits *)
  DWord = LONGCARD;  (* 32 bits *)
  Inline2 = ARRAY [0..1] OF SHORTCARD;
  Inline4 = ARRAY [0..3] OF SHORTCARD;
  Inline12 = ARRAY [0..11] OF SHORTCARD;
  Inline25H = ARRAY [0..24H] OF SHORTCARD;
  (*# save, call(reg_param=>(ax),near_call=>off) *)
  XMSProc = PROCEDURE(CARDINAL): CARDINAL;
  (*# restore *)

VAR
  Program: ARRAY [0..128] OF CHAR;
  verbose, noreset, trace, XMS, disableNMI, apm: BOOLEAN;
  kernel, params: ARRAY [0..128] OF CHAR;
  scrCode: ARRAY [0..256] OF CHAR;
  stackSeg, stackTop, bootCode: CARDINAL;
  kernSeg, kernSize: CARDINAL;
  extMemSize, baseMemSize: DWord;
  lowMem, highMem: CARDINAL;
  screenSeg: CARDINAL;
  callXMS: XMSProc;
  table: POINTER stackSeg TO ARRAY [0..Max-1] OF DWord;
  tableIndex: CARDINAL;
  env: ARRAY [0..2*Max] OF CHAR;
  envSize: CARDINAL;
  kpar: ARRAY [0..1] OF LONGCARD;
  KernelBase: LONGCARD;

  (*** Messages ***)

  PROCEDURE Intro;
  BEGIN
    IO.WrStr("Native Oberon Bootstrap Loader, "); IO.WrStr(Version); IO.WrLn;
    IO.WrStr("Written by Pieter Muller & Bruno Essmann"); IO.WrLn; IO.WrLn;
  END Intro;

  PROCEDURE Abort (str1, str2: ARRAY OF CHAR);
  BEGIN
    IO.WrStr(Program); IO.WrStr(": "); IO.WrStr(str1);
    IO.WrStr(" "); IO.WrStr(str2); IO.WrLn;
    HALT
  END Abort;

  PROCEDURE Usage;
  BEGIN
    IO.WrStr("usage: "); IO.WrStr(Program);
    IO.WrStr(" [-hptv] kernel params"); IO.WrLn;
    IO.WrStr(" -p      don't reset processor"); IO.WrLn;
    IO.WrStr(" -t      trace mode (no boot)"); IO.WrLn;
    IO.WrStr(" -v      verbose message output"); IO.WrLn;
    IO.WrStr(" kernel  kernel image file name"); IO.WrLn;
    IO.WrStr(" params  parameter file name"); IO.WrLn;
    HALT
  END Usage;

  PROCEDURE Status (str1, str2: ARRAY OF CHAR);
  BEGIN
    IF verbose THEN IO.WrStr(str1); IO.WrStr(str2); IO.WrLn END
  END Status;

  PROCEDURE Progress (ch : CHAR);
    VAR p: POINTER screenSeg TO CARDINAL;
  BEGIN
    IF NOT trace THEN
      p := SYSTEM.NearADDRESS(0);
      (*# save,check(nil_ptr=>off) *)
      p^ := 4F00H + ORD(ch)
      (*# restore *)
    END
  END Progress;


  (*** Flags ***)

  PROCEDURE CheckFlags;
  CONST ShiftSet = BiosIO.KBFlagSet{BiosIO.RShift,BiosIO.LShift,BiosIO.Ctrl,
    BiosIO.Alt,BiosIO.Scroll};
  VAR edit: ARRAY [0..128] OF CHAR;  res: CARDINAL;
  BEGIN
    IF BiosIO.KBFlags() * ShiftSet # BiosIO.KBFlagSet{} THEN
      Lib.EnvironmentFind("EDITOR", edit);
      IF edit[0] = 0C THEN edit := "edit" END;
      Str.Append(edit, " ");  Str.Append(edit, params);
      res := Lib.ExecCmd(edit);
      IF res = MAX(CARDINAL) THEN
        Abort("Can not execute:", edit)
      END
    END
  END CheckFlags;

  (*** Parameters ***)

  PROCEDURE LoadParameters;
    VAR f: FIO.File; str, name, value: ARRAY [0..255] OF CHAR;
      line: CARDINAL;

    PROCEDURE Error (str: ARRAY OF CHAR);
    BEGIN
      IO.WrStr(params); IO.WrStr("', line "); IO.WrCard(line, 1);
      IO.WrStr(": "); IO.WrStr(str); IO.WrLn; HALT
    END Error;

    PROCEDURE IncEnv;
    BEGIN
      INC(envSize);
      IF envSize = 2*Max THEN Error("out of environment space") END
    END IncEnv;

    PROCEDURE Read;
      VAR i: CARDINAL; ok: BOOLEAN;
    BEGIN
      REPEAT
        INC(line); FIO.RdStr(f, str);
        i := 0; WHILE (i < HIGH(str)) AND (str[i] # CHR(0)) DO INC(i) END;
        IF str[i] # CHR(0) THEN Error("line too long") END;
      UNTIL ((str[0] # "#") AND (str[0] # CHR(0)) OR FIO.EOF)
    END Read;

    PROCEDURE Assign;
      VAR i: CARDINAL;
    BEGIN
      i := 0;
      WHILE (name[i] # CHR(0)) DO env[envSize] := name[i]; IncEnv; INC(i) END;
      env[envSize] := CHR(0); IncEnv; i := 0;
      WHILE (value[i] # CHR(0)) DO env[envSize] := value[i]; IncEnv; INC(i) END;
      env[envSize] := CHR(0); IncEnv
    END Assign;

    PROCEDURE Split;
      VAR quote: BOOLEAN; i, k: CARDINAL;
    BEGIN
      quote := FALSE;
      i := 0; k := 0;
      WHILE (str[i] # CHR(0)) AND (str[i] # '=') DO
        IF str[i] # ' ' THEN name[k] := str[i]; INC(k) END;
        INC(i)
      END;
      name[k] := CHR(0);
      IF str[i] # '=' THEN Error("malformed line, = missing") END;
      INC(i); WHILE str[i] = " " DO INC(i) END;
      IF str[i] = '"' THEN quote := TRUE; INC(i); END;
      IF str[i] = CHR(0) THEN Error("malformed line, value missing") END;
      k := 0;
      WHILE (str[i] # CHR(0)) AND (NOT quote OR (str[i] # '"')) DO
        IF quote OR (str[i] # " ") THEN value[k] := str[i]; INC(k) END;
        IF str[i] = ' ' THEN str[i] := CHR(0); DEC(i) END;
        INC(i)
      END;
      IF quote THEN
        IF str[i] # '"' THEN Error('malformed line, closing " missing') END;
        WHILE (str[i] # CHR(0)) AND (str[i] # "+") DO INC(i) END;
        IF str[i] = '+' THEN
          WHILE (str[i] # CHR(0)) AND (str[i] # '"') DO INC(i) END;
          IF str[i] = CHR(0) THEN Error('malformed line, " missing after +') END;
          INC(i); IF str[i] = CHR(0) THEN Error("malformed line, missing value after +") END;
          WHILE (str[i] # CHR(0)) AND (str[i] # '"') DO
            value[k] := str[i]; INC(k); INC(i)
          END;
          IF str[i] = CHR(0) THEN Error('malformed line, closing " missing after +') END
        END
      END;
      value[k] := CHR(0);
      IF Str.Compare(name, "Init") = 0 THEN
        Str.Copy(scrCode, value)
      ELSIF Str.Compare(name, "APM") = 0 THEN
        IF Str.Compare(value, "0") = 0 THEN
          apm := FALSE;  IO.WrStr("APM disabled");  IO.WrLn
        END
      ELSIF Str.Compare(name, "KernelBase") = 0 THEN
        k := Str.Length(value);
        IF (k # 0) AND (value[k-1] = "H") THEN
          value[k-1] := 0C;
          k := 16
        ELSE
          k := 10
        END;
        KernelBase := Str.StrToCard(value, k, quote);
        IF NOT quote THEN Error('Bad KernelBase value') END;
        IO.WrStr("KernelBase=");  IO.WrLngHex(KernelBase, 8);  IO.WrLn
      ELSE
        Assign
      END
    END Split;

    PROCEDURE Padd;
    BEGIN
      env[envSize] := CHR(0); IncEnv;
      WHILE (envSize MOD 4) # 0 DO
        env[envSize] := CHR(0); IncEnv
      END
    END Padd;

  BEGIN
    Status(params, " reading...");
    FIO.IOcheck := FALSE;
    f := FIO.OpenRead(params);
    IF f = MAX(CARDINAL) THEN Abort(params, "could not be opened") END;
    line := 0; envSize := 0;
    WHILE NOT FIO.EOF DO
      Read;
      IF NOT FIO.EOF THEN Split END
    END;
    Padd;
    FIO.Close(f)
  END LoadParameters;

  PROCEDURE InitParameters;
    VAR par: ARRAY [0..128] OF CHAR; i,k: CARDINAL; flags: BOOLEAN;
  BEGIN
    noreset := FALSE; verbose := FALSE;  apm := TRUE;
    KernelBase := 1000H;  (* default *)
    Lib.ParamStr(par, 0);
    IF par[0] = CHR(0) THEN
      Program := ProgramName
    ELSE
      k := 0; WHILE (par[k] # CHR(0)) AND (par[k] # '.') DO INC(k) END;
      WHILE (k > 0) AND (par[k-1] # '\') DO DEC(k) END;
      i := 0; WHILE (par[k] # CHR(0)) AND (par[k] # '.') DO Program[i] := par[k]; INC(k); INC(i) END;
      Str.Lows(Program)
    END;
    IF Lib.ParamCount() = 0 THEN Intro; Usage END;
    k := 0; i := 1; flags := TRUE;
    WHILE flags DO
      IF (k = 0) OR (par[k] = CHR(0)) THEN
        Lib.ParamStr(par, i); INC(i); k := 0;
        IF (par[k] # "-") AND (par[k] # "/") THEN
          flags := FALSE
        ELSE
          INC(k)
        END
      END;
      IF flags THEN
        CASE par[k] OF
          | "p", "P": noreset := TRUE
          | "t", "T": trace := TRUE; verbose := TRUE
          | "v", "V": verbose := TRUE
        ELSE
          Intro; IO.WrStr(Program); IO.WrStr(": unknown option -");
          IO.WrChar(par[k]); IO.WrLn; Usage
        END;
        INC(k)
      END
    END;
    IF par[0] = CHR(0) THEN
      Intro; IO.WrStr(Program); IO.WrStr(": kernel missing"); IO.WrLn;
      Usage
    END;
    k := 0; WHILE par[k] # CHR(0) DO kernel[k] := par[k]; INC(k) END;
    kernel[k] := CHR(0);
    Lib.ParamStr(par, i);
    IF par[0] = CHR(0) THEN
      Intro; IO.WrStr(Program); IO.WrStr(": params missing"); IO.WrLn;
      Usage
    END;
    k := 0; WHILE par[k] # CHR(0) DO params[k] := par[k]; INC(k) END;
    params[k] := CHR(0);
    IF trace THEN Status("trace mode enabled", "") END;
    IF noreset THEN Status("processor reset disabled", "") END;
    IF verbose AND (trace OR noreset) THEN IO.WrLn END;
    CheckFlags;
    LoadParameters
  END InitParameters;


  (*** CMOS ***)

  PROCEDURE CMOSWrite (Addr: SHORTCARD; Value: SHORTCARD); (* AT Tech. Ref. p-5-81 *)
    VAR i: BOOLEAN;
  BEGIN
    i := 9 IN BITSET(SYSTEM.GetFlags());
    SYSTEM.DI;
    SYSTEM.Out(070H, Addr MOD 080H + 080H);  (* CMOS address + no NMI *)
    SYSTEM.Out(071H, Value);  (* Value *)
    IF disableNMI THEN SYSTEM.Out(070H, 08DH) ELSE SYSTEM.Out(070H, 00DH) END;
    IF i THEN SYSTEM.EI END
  END CMOSWrite;

  PROCEDURE CMOSRead (Addr: SHORTCARD; VAR Value: SHORTCARD);
    VAR i: BOOLEAN;
  BEGIN
    i := 9 IN BITSET(SYSTEM.GetFlags());
    SYSTEM.DI;
    SYSTEM.Out(070H, Addr MOD 080H + 080H);  (* CMOS address + no NMI *)
    Value := SYSTEM.In(071H);  (* Value *)
    IF disableNMI THEN SYSTEM.Out(070H, 08DH) ELSE SYSTEM.Out(070H, 00DH) END;
    IF i THEN SYSTEM.EI END
  END CMOSRead;


  (*** Globals ***)

  PROCEDURE InitGlobals;
    VAR sR, xR: SYSTEM.Registers; ch: CHAR; v1, v2 : SHORTCARD;
      (*cpu: Lib.CpuRec;*)
  BEGIN
    (* check screen *)
    sR.AH := 15;
    Lib.Intr(sR, 10H);
    IF sR.AL = 7 THEN screenSeg := 0B000H ELSE screenSeg := 0B800H END;
(*
    (* check cpu *)
    IF NOT nocpu THEN
      Lib.CpuId(cpu);
      IF cpu.cpu = Lib.cpu_Unknown THEN cpu.cpu := Lib.cpu_80386 END; (* pentium *)
      IF cpu.cpu # Lib.cpu_80386 THEN
        IO.WrStr(Program); IO.WrStr(": native oberon requires an 80386 or compatible machine"); IO.WrLn;
        IO.WrStr("        use option -c to disable the cpu check, -h for help"); IO.WrLn;
        HALT
      END
    END;
*)
    (* check xms *)
    xR.AX := 04300H;
    Lib.Intr(xR, 02FH);
    XMS := (xR.AL = 080H);
    IF XMS THEN
      IF trace THEN IO.WrStr("XMS detected"); IO.WrLn END;
      xR.AX := 04310H;
      Lib.Intr(xR, 02FH);
      callXMS := XMSProc([xR.ES:xR.BX]);
    END;
    CMOSRead(17H, v1);  CMOSRead(18H, v2);
    extMemSize := (VAL(LONGCARD, v2)*256 + VAL(LONGCARD, v1))*1024;
    CMOSRead(15H, v1);  CMOSRead(16H, v2);
    baseMemSize := (VAL(LONGCARD, v2)*256 + VAL(LONGCARD, v1))*1024;
    IF trace THEN
      IO.WrCard(VAL(CARDINAL, baseMemSize DIV 1024), 1);
      IO.WrStr(" kbytes low memory detected"); IO.WrLn;
      IO.WrCard(VAL(CARDINAL, extMemSize DIV 1024), 1);
      IO.WrStr(" kbytes extended memory detected"); IO.WrLn; IO.WrLn
    END
  END InitGlobals;


  (*** Strings ***)

  PROCEDURE StrToUNum (String: ARRAY OF CHAR; Base: CARDINAL; VAR Number: LONGCARD): BOOLEAN;
    VAR AllOk: BOOLEAN; StrIdx: CARDINAL; TmpChr: CHAR; Digit, MaxDIV: LONGCARD;
  BEGIN
    IF (2 <= Base) AND (Base <= 36) THEN
      Number := 0; StrIdx := 0;
      MaxDIV := MAX(LONGCARD) DIV VAL(LONGCARD,Base);
      AllOk := TRUE;
      IF String[0] = '+' THEN
        INC(StrIdx)
      ELSIF String[0] = 0C THEN
        AllOk := FALSE
      END;
      LOOP
        IF (StrIdx > HIGH(String)) OR (String[StrIdx] = 00C) THEN
          EXIT
        ELSE
          TmpChr := CAP(String[StrIdx]);
          IF ('0' <= TmpChr) AND (TmpChr <= '9') THEN
            Digit := VAL(LONGCARD, ORD(TmpChr)-ORD('0'))
          ELSIF ('A' <= TmpChr) AND (TmpChr <= 'Z') THEN
            Digit := VAL(LONGCARD,ORD(TmpChr)-ORD('A')+10)
          ELSE
            AllOk := FALSE;
            EXIT
          END;
          IF (Digit < VAL(LONGCARD,Base)) AND ((MaxDIV > Number) OR
             (MAX(LONGCARD) DIV Number >= VAL(LONGCARD,Base)) AND
             (MAX(LONGCARD)-Number*VAL(LONGCARD,Base) >= Digit)) THEN
            Number := VAL(LONGCARD,Base)*Number+Digit
          ELSE
            AllOk := FALSE;
            EXIT
          END
        END;
        INC(StrIdx)
      END
    ELSE
      AllOk := FALSE
    END;
    RETURN AllOk
  END StrToUNum;

  PROCEDURE StrToCard (String: ARRAY OF CHAR; VAR Card: CARDINAL): BOOLEAN;
    VAR LCard: LONGCARD; AllOk: BOOLEAN;
  BEGIN
    AllOk := StrToUNum(String,10,LCard);
    IF AllOk THEN
      IF LCard <= VAL(LONGCARD,MAX(CARDINAL)) THEN
        Card := VAL(CARDINAL,LCard)
      ELSE
        AllOk := FALSE
      END
    END;
    RETURN AllOk
  END StrToCard;


  (*** Memory ***)

  PROCEDURE InitMemory;
    VAR R: SYSTEM.Registers; size: CARDINAL;
  BEGIN
    (* allocate largest available memory *)
    R.AH := 48H; R.BX := 0FFFFH; Lib.Dos(R);
    (* now BX = free memory available (paragraphs) *)
    size := R.BX; R.AH := 48H; Lib.Dos(R);
    IF 0 IN R.Flags THEN Abort("[ InitMemory ]", "DOS allocate failure") END;
    IF R.AX < 100H THEN  (* 1000H DIV 16 *)
      lowMem := 100H;
      DEC(size, 100H-R.AX)
    ELSE
      lowMem := R.AX
    END;
    highMem := lowMem + size;
    (* round to page size (4k) *)
    IF lowMem MOD 256 # 0 THEN INC(lowMem, 256-lowMem MOD 256) END;
    IF highMem MOD 256 # 0 THEN DEC(highMem, highMem MOD 256) END;
    IF highMem > 0A000H THEN highMem := 0A000H END;  (* QEMM? *)
    size := highMem-lowMem;
    IF trace THEN
      IO.WrStr("Bootstrap heap: ");
      IO.WrLngHex(VAL(LONGCARD, lowMem)*16, 5); IO.WrStr("-");
      IO.WrLngHex(VAL(LONGCARD, highMem)*16-1, 5); IO.WrStr(" (");
      IO.WrCard(size DIV 64, 1); IO.WrStr(" kbytes)"); IO.WrLn; IO.WrLn
    END
  END InitMemory;

  PROCEDURE Allocate(VAR oSeg: CARDINAL; iSize: CARDINAL);
  BEGIN  (* Allocate iSize paragraphs of memory and return segment oSeg *)
    IF lowMem + iSize > highMem THEN Abort("[ Allocate ]", "Out of low memory") END;
    DEC(highMem, iSize); oSeg := highMem
  END Allocate;


  (*** Boot Table ***)

  PROCEDURE NewEntry(type, recsize, v1, v2: DWord; VAR i: CARDINAL);
  BEGIN  (* Allocate table entry (table[i]) *)
    IF recsize MOD 4 # 0 THEN Abort("[ NewEntry ]", "bad record size") END;
    IF tableIndex+VAL(CARDINAL, recsize) > Max THEN Abort("[ NewEntry ]", "out of boot table space") END;
    i := tableIndex; INC(tableIndex, VAL(CARDINAL, recsize) DIV 4);
    table^[i] := type; table^[i+1] := recsize;
    table^[i+2] := v1; table^[i+3] := v2
  END NewEntry;

  PROCEDURE ShowTable;
    VAR showenv, i, k: CARDINAL; type: DWord;
      str: ARRAY [0..255] OF CHAR; ep: POINTER stackSeg TO ARRAY [0..2*Max] OF CHAR;

    PROCEDURE Range(addr, size: DWord);
    BEGIN
      IF addr # 0 THEN
        IO.WrLngHex(addr, 6); IO.WrStr(" - ");  IO.WrLngHex(addr+size-1, 6);
        IO.WrStr(" ("); IO.WrCard(VAL(CARDINAL, size DIV 1024), 5); IO.WrStr(" kbytes) ")
      END
    END Range;

    PROCEDURE ReadEnv;
      VAR k: CARDINAL;
    BEGIN
      k := 0;
      WHILE ep^[i] # CHR(0) DO str[k] := ep^[i]; INC(i); INC(k) END;
      INC(i); str[k] := "="; INC(k);
      WHILE ep^[i] # CHR(0) DO str[k] := ep^[i]; INC(i); INC(k) END;
      INC(i); str[k] := CHR(0);
    END ReadEnv;

  BEGIN  (* Show boot table *)
    IO.WrStr("Boot table:"); IO.WrLn;
    showenv := 0; i := 0;
    WHILE table^[i] # MAX(DWord) DO
      type := table^[i];
      IO.WrCard(i*4, 4);
      CASE VAL(CARDINAL, type) OF
        3:
          IO.WrStr(" [ 3] Boot memory  "); Range(table^[i+2], table^[i+3])
        |4:
          IO.WrStr(" [ 4] Free memory  "); Range(table^[i+2], table^[i+3])
        |5:
          IO.WrStr(" [ 5] Hard disk ");
          IO.WrCard(VAL(CARDINAL, table^[i+2]), 1); IO.WrStr("  ")
        |8:
          IO.WrStr(" [ 8] Environment  "); showenv := i
      ELSE
        IO.WrStr(" ["); IO.WrCard(VAL(CARDINAL, type), 1);
        IO.WrStr("] Type unknown")
      END (* CASE *);
      IO.WrLn;
      INC(i, VAL(CARDINAL, table^[i+1]) DIV 4)
    END;
    IO.WrCard(i*4, 4); IO.WrStr(" [-1] End of table"); IO.WrLn; IO.WrLn;
    IF showenv # 0 THEN
      IO.WrStr("Environment:"); IO.WrLn;
      ep := SYSTEM.ADR(table^[showenv+2]);
      i := 0;
      REPEAT
        ReadEnv; IO.WrStr(str); IO.WrLn
      UNTIL ep^[i] = CHR(0);
      IO.WrLn
    END
  END ShowTable;

  PROCEDURE InitTable;
    VAR seg, s, o, i: CARDINAL; v: SHORTCARD;
      Int: POINTER 0 TO ARRAY [0..255] OF RECORD ofs, seg: CARDINAL END;
  BEGIN
    (* boot memory *)
    Allocate(stackSeg, 4096 DIV 16);  stackTop := 800H;
    bootCode := 0;
    table := SYSTEM.NearADDRESS(stackTop);  tableIndex := 0;
    NewEntry(3, 16, VAL(DWord, stackSeg)*16, 4096, i);
    (* free memory *)
    IF extMemSize # 0 THEN
      NewEntry(4, 16, 100000H, extMemSize, i)
    END;
    (* fixed disk parameters *)
    CMOSRead(12H, v);
    (*#save, check(nil_ptr=>off)*)
    Int := SYSTEM.NearADDRESS(0H);
    IF v DIV 10H # 0 THEN
      s := Int^[41H].seg;  o := Int^[41H].ofs;
      NewEntry(5, 28, 0, 0, i);  (* HD 0 *)
      Lib.FarMove([s:o], SYSTEM.FarADR(table^[i+3]), 16)
    END;
    IF v MOD 10H # 0 THEN
      s := Int^[46H].seg;  o := Int^[46H].ofs;
      NewEntry(5, 28, 1, 0, i);  (* HD 1 *)
      Lib.FarMove([s:o], SYSTEM.FarADR(table^[i+3]), 16)
    END;
    (*#restore*)
    (* environment *)
    NewEntry(8, VAL(DWord, envSize+8), 0, 0, i);
    Lib.FarMove(SYSTEM.FarADR(env), SYSTEM.FarADR(table^[i+2]), envSize);
    (* end of table *)
    NewEntry(MAX(DWord), 4, 0, 0, i);
    IF trace THEN ShowTable END
  END InitTable;


  (*** File I/O ***)

  PROCEDURE Open (name: ARRAY OF CHAR): CARDINAL;
    VAR R: SYSTEM.Registers;
  BEGIN
    R.AX := 3D00H; R.DS := SYSTEM.Seg(name);
    R.DX := SYSTEM.Ofs(name); Lib.Dos(R);
    IF 0 IN R.Flags THEN Abort(name, "could not be opened") END;
    RETURN R.AX
  END Open;

  PROCEDURE Close (file: CARDINAL);
    VAR R: SYSTEM.Registers;
  BEGIN
    R.AH := 3EH; R.BX := file; Lib.Dos(R);
    IF 0 IN R.Flags THEN Abort("[ Close ]", "could not close file") END
  END Close;

  PROCEDURE FileSize (file: CARDINAL): LONGCARD;
    VAR Old1, Old2: CARDINAL; Pos: LONGCARD; R: SYSTEM.Registers;
  BEGIN
    (* Get current position *)
    R.AX := 4201H; R.BX := file; R.CX := 0; R.DX := 0; Lib.Dos(R);
    IF 0 IN R.Flags THEN Abort("[ FileSize ]", "could not read current file position") END;
    Old1 := R.AX; Old2 := R.DX;
    (* Get length *)
    R.AX := 4202H; R.BX := file; R.CX := 0; R.DX := 0; Lib.Dos(R);
    IF 0 IN R.Flags THEN Abort("[ FileSize ]", "could not read file length") END;
    Pos := (VAL(LONGCARD,R.DX)*10000H) + VAL(LONGCARD,R.AX);
    (* Restore position *)
    R.AX := 4200H; R.BX := file; R.CX := Old1; R.DX := Old2; Lib.Dos(R);
    IF 0 IN R.Flags THEN Abort("[ FileSize ]", "could not restore file position") END;
    RETURN Pos
  END FileSize;

  PROCEDURE Read(file: CARDINAL; buf: SYSTEM.FarADDRESS; len: CARDINAL): CARDINAL;
    VAR R: SYSTEM.Registers;
  BEGIN
    R.AH := 3FH; R.BX := file; R.CX := len;
    R.DS := SYSTEM.Seg(buf^); R.DX := SYSTEM.Ofs(buf^); Lib.Dos(R);
    IF 0 IN R.Flags THEN Abort("[ Read ]", "could not read file") END;
    RETURN R.AX
  END Read;

  PROCEDURE ReadFile(filename: ARRAY OF CHAR; VAR seg: CARDINAL; VAR fsize: LONGCARD);
    VAR f, size: CARDINAL; buf: SYSTEM.FarADDRESS; filesize: LONGCARD;
  BEGIN  (* read a file at segment seg with size fsize (multiple of 4k) *)
    Status(filename, " reading...");
    f := Open(filename);
    filesize := FileSize(f);
    IF filesize MOD 4096 = 0 THEN fsize := filesize
    ELSE fsize := filesize + 4096 - filesize MOD 4096
    END;
    Allocate(seg, VAL(CARDINAL, fsize DIV 16));
    buf := [seg:0];
    WHILE filesize # 0 DO
      size := Read(f, buf, MAX(CARDINAL));
      Lib.IncFarAddr(buf, size);
      DEC(filesize, VAL(LONGCARD, size))
    END;
    Close(f)
  END ReadFile;


  (*** Kernel ***)

  PROCEDURE InitKernel;
    VAR seg: CARDINAL; fsize : LONGCARD;

  BEGIN
    ReadFile(kernel, seg, fsize);
    kernSeg := seg; kernSize := VAL(CARDINAL, fsize DIV 16);
    (*IO.WrLngHex(LONGCARD(kernSeg)*16, 8);
    IO.WrLngHex(LONGCARD(kernSize)*16, 9);  IO.WrLn*)
  END InitKernel;


  (*** Screen ***)

(*
  (*# save, call(reg_param=>(ax, bx), inline=>on) *)
  INLINE PROCEDURE CallInit(adr: SYSTEM.FarADDRESS) =
    Inline12(55H, 53H, 50H, 8BH, 0ECH, 0FFH, 5EH, 0H, 83H, 0C4H, 4H, 5DH);
    (* push bp; push bx; push ax; mov bp, sp; call [dword bp]; add sp, 4; pop bp *)
  (*# restore *)
*)

  (*# save, call(reg_param=>(ax,bx,cx,dx,si,di), inline=>on) *)
  INLINE PROCEDURE CallInit(
      ax,bx,cx,dx,si,di,es,ds: CARDINAL;
      adr: SYSTEM.FarADDRESS;
      kpar: SYSTEM.FarADDRESS) =
    Inline25H(55H, 8BH, 0ECH, 1EH, 6, 8EH, 46H, 0CH, 8EH, 5EH, 0AH, 0FCH, 55H,
      0FFH, 5EH, 6, 5DH, 0C5H, 76H, 2, 89H, 4, 89H, 5CH, 2, 89H, 4CH, 4,
      89H, 54H, 6, 7, 1FH, 5DH, 83H, 0C4H, 0CH);
  (*# restore *)
(*
      7	    0000  55				 push bp
      8	    0001  8B EC				 mov bp,sp
     10	    0003  1E				 push ds
     11	    0004  06				 push es
     13	    0005  8E 46	0C			 mov es,[bp+12]	  	 ; es:0	= 1k scratch segment
     14	    0008  8E 5E	0A			 mov ds,[bp+10]
     16	    000B  FC				 cld
     17	    000C  55				 push bp
     18	    000D  FF 5E	06			 call [dword bp+6]
     19	    0010  5D				 pop bp
     21	    0011  C5 76	02			 lds si,[bp+2]
     22	    0014  89 04				 mov [si],ax
     23	    0016  89 5C	02			 mov [si+2],bx
     24	    0019  89 4C	04			 mov [si+4],cx
     25	    001C  89 54	06			 mov [si+6],dx
     27	    001F  07				 pop es
     28	    0020  1F				 pop ds
     29	    0021  5D				 pop bp
     31	    0022  83 C4	0C			 add sp,12
*)

  PROCEDURE InitScreen;
    VAR i, scratch: CARDINAL; tmp: CHAR;
  BEGIN (* initialize screen according to Init environment *)
    i := 0;
    WHILE scrCode[2*i] # CHR(0) DO
      tmp := CAP(scrCode[2*i]);
      IF ('0' <= tmp) AND (tmp <= '9') THEN scrCode[i] := VAL(CHAR, (ORD(tmp)-ORD('0'))*16)
      ELSIF ('A' <= tmp) AND (tmp <= 'Z') THEN scrCode[i] := VAL(CHAR, (ORD(tmp)-ORD('A')+10)*16)
      ELSE Abort("[ InitScreen ]", "Wrong format of Init environment")
      END;
      IF scrCode[2*i+1] = CHR(0) THEN Abort("[ InitScreen ]", "Wrong format of Init environment") END;
      tmp := CAP(scrCode[2*i+1]);
      IF ('0' <= tmp) AND (tmp <= '9') THEN scrCode[i] := scrCode[i] + VAL(CHAR, ORD(tmp)-ORD('0'))
      ELSIF ('A' <= tmp) AND (tmp <= 'Z') THEN scrCode[i] := scrCode[i] + VAL(CHAR, ORD(tmp)-ORD('A')+10)
      ELSE Abort("[ InitScreen ]", "Wrong format of Init environment")
      END;
      INC(i)
    END;
    scrCode[i] := CHR(0CBH); (* retf *)
    (*CallInit(SYSTEM.FarADR(scrCode[0]))*)
    Allocate(scratch, 1024 DIV 16);
    CallInit(0,0,0,0,0,0,scratch,0,SYSTEM.FarADR(scrCode[0]),
      SYSTEM.FarADR(kpar[0]))
  END InitScreen;


  (*** Floppy ***)

  PROCEDURE StopFloppy;
  BEGIN  (* stop the floppy motor *)
    SYSTEM.Out(03F2H, 00CH)
  END StopFloppy;

  (*** A20 ***)

  PROCEDURE UnmaskA20;

    PROCEDURE Error;
    BEGIN
      LOOP END
    END Error;

    PROCEDURE Empty8042;
      VAR x: CARDINAL;
    BEGIN
      x := MAX(CARDINAL);
      WHILE (x # 0) AND (1 IN BITSET(SYSTEM.In(064H))) DO DEC(x) END;
      IF x = 0 THEN Error END
    END Empty8042;

  BEGIN  (* UnmaskA20 - From AT Tech. Ref. p 5-172 and XMS Spec. 2.0  *)
    Progress('A');
    IF XMS THEN
      IF callXMS(0300H) # 1 THEN Error END
    ELSE
      Empty8042; SYSTEM.Out(064H, 0D1H);
      Empty8042; SYSTEM.Out(060H, 0DFH);
      Empty8042
    END;
    Progress('2')
  END UnmaskA20;

  (*** Relocation stuff ***)

  (*# save, call(reg_param=>(si,di),inline=>on) *)
  INLINE PROCEDURE RelocateJmp (seg, ofs: CARDINAL) =
    Inline4(056H,057H,0CBH,090H);  (* push si; push di; retf; nop *)
  (*# restore *)

  (*# save, call(reg_param=>(),inline=>on) *)
  INLINE PROCEDURE Hlt () =
    Inline2(0F4H,090H);  (* hlt; nop *)
  (*# restore *)

(*
  PROCEDURE Push (v: CARDINAL);
    VAR p: POINTER stackSeg TO CARDINAL;
  BEGIN
    DEC(stackTop, 2);
    p := SYSTEM.NearADDRESS(stackTop);
    p^ := v
  END Push;

  PROCEDURE Relocate;
    VAR RomInit: POINTER 040H TO ARRAY [0..1] OF CARDINAL;
      Int: POINTER 0 TO ARRAY [0..255] OF RECORD ofs, seg: CARDINAL END;
      s, o, x : CARDINAL; p, t : LONGCARD;
  BEGIN
    Progress('R');
    (* Set up the stack *)
    IF KernelBase > 0FFFFFH THEN
      Push(CARDINAL(KernelBase DIV 10000H));
      Push(CARDINAL(KernelBase MOD 10000H))
    END;
    Push(00CFH); Push(9200H); Push(0); Push(0FFFFH);  (* kernel data *)
    Push(00CFH); Push(9A00H); Push(0); Push(0FFFFH);  (* kernel code *)
    Push(0); Push(0); Push(0); Push(0);  (* null descriptor *)
    p := VAL(LONGCARD, stackSeg)*16 + VAL(LONGCARD, stackTop);
    Push(2*8);  (* kernel data segment selector *)
    Push(1*8);  (* kernel code segment selector *)
    Push(CARDINAL(KernelBase DIV 10000H));
    Push(CARDINAL(KernelBase MOD 10000H));  (* kernel entry point *)
    Push(CARDINAL(kpar[1] DIV 10000H));
    Push(CARDINAL(kpar[1] MOD 10000H));
    Push(CARDINAL(kpar[0] DIV 10000H));
    Push(CARDINAL(kpar[0] MOD 10000H));
    t := VAL(LONGCARD, stackSeg)*16 + VAL(LONGCARD, CARDINAL(table));
    Push(VAL(CARDINAL, t DIV 10000H)); Push(VAL(CARDINAL, t MOD 10000H)); Push(0);
    Push(VAL(CARDINAL, p DIV 10000H));  Push(VAL(CARDINAL, p MOD 10000H)); Push(3*8-1);
    Push(screenSeg); Push(kernSize); Push(kernSeg);
    IF KernelBase > 0FFFFFH THEN
      Push(0FFFFH)
    ELSE
      Push(CARDINAL(KernelBase DIV 16))
    END;
    (* Patch the boot code *)
    (*# save,data(const_assign=>on) *)
    Boot.Code[1] := SHORTCARD(stackSeg MOD 100H);
    Boot.Code[2] := SHORTCARD(stackSeg DIV 100H);
    Boot.Code[4] := SHORTCARD(stackTop MOD 100H);
    Boot.Code[5] := SHORTCARD(stackTop DIV 100H);
    (*# restore *)
    (* Move boot code up to LoaderSeg *)
    Lib.FarMove(SYSTEM.FarADR(Boot.Code), [stackSeg:bootCode], SIZE(Boot.Code));
    Progress('C');
    SYSTEM.DI;  (* Disable interrupts *)
    IF DisableIRQs THEN
      SYSTEM.Out(021H, 0FFH);  (* Mask off IRQ 0-7 *)
      SYSTEM.Out(0A1H, 0FFH);  (* Mask off IRQ 8-15 (AT only) *)
      FOR x := 0 TO 15 DO
        SYSTEM.Out(020H, 020H);  (* EOI *)
        Lib.Delay(10)
      END
    END;
    IF noreset THEN
      RelocateJmp(stackSeg, bootCode)
    ELSE
      (* Reset the CPU to get out of virtual mode *)
      RomInit := SYSTEM.NearADDRESS(067H);
      RomInit^[0] := bootCode;  (* offset to jmp to after reset *)
      RomInit^[1] := stackSeg;  (* segment *)
      (* HL - 15 Aug 95: Changed CMOSWrite from 15,5 to 15,10 - dont know why *)
      CMOSWrite(15, 10);  (* Write shutdown status byte *)
      Progress('D');
      SYSTEM.Out(064H, 0FEH);  (* shut down *)
      LOOP Hlt (* to work with cache *) END  (* wait for end *)
    END
  END Relocate;
*)

  PROCEDURE Push (v: LONGCARD);
    VAR p: POINTER stackSeg TO LONGCARD;
  BEGIN
    DEC(stackTop, 4);
    p := SYSTEM.NearADDRESS(stackTop);
    p^ := v
  END Push;

  PROCEDURE Linear(seg, ofs: CARDINAL): LONGCARD;
  BEGIN
    RETURN VAL(LONGCARD, seg)*16 + VAL(LONGCARD, ofs)
  END Linear;

  PROCEDURE Relocate;
    VAR RomInit: POINTER 040H TO ARRAY [0..1] OF CARDINAL;
      Int: POINTER 0 TO ARRAY [0..255] OF RECORD ofs, seg: CARDINAL END;
      s, o, x : CARDINAL; p, t : LONGCARD;
  BEGIN
    Progress('R');
    (* Set up the stack *)
    IF apm THEN Push(1) ELSE Push(0) END;
    Push(Linear(stackSeg, CARDINAL(table)));
    Push(kpar[1]);  Push(kpar[0]);
    Push(KernelBase);  (* entry *)
    Push(VAL(LONGCARD, kernSize)*16 DIV 4);
    Push(KernelBase);  (* dest *)
    Push(Linear(kernSeg, 0));

    (* Patch the boot code *)
    (*# save,data(const_assign=>on) *)
    Boot.Code[1] := SHORTCARD(stackSeg MOD 100H);
    Boot.Code[2] := SHORTCARD(stackSeg DIV 100H);
    Boot.Code[4] := SHORTCARD(stackTop MOD 100H);
    Boot.Code[5] := SHORTCARD(stackTop DIV 100H);
    (*# restore *)

    UnmaskA20;

    (* Move boot code up to LoaderSeg *)
    Lib.FarMove(SYSTEM.FarADR(Boot.Code), [stackSeg:bootCode], SIZE(Boot.Code));

    Progress('C');
    SYSTEM.DI;  (* Disable interrupts *)
    IF DisableIRQs THEN
      SYSTEM.Out(021H, 0FFH);  (* Mask off IRQ 0-7 *)
      SYSTEM.Out(0A1H, 0FFH);  (* Mask off IRQ 8-15 (AT only) *)
      FOR x := 0 TO 15 DO
        SYSTEM.Out(020H, 020H);  (* EOI *)
        Lib.Delay(10)
      END
    END;
    IF noreset THEN
      RelocateJmp(stackSeg, bootCode)
    ELSE
      (* Reset the CPU to get out of virtual mode *)
      RomInit := SYSTEM.NearADDRESS(067H);
      RomInit^[0] := bootCode;  (* offset to jmp to after reset *)
      RomInit^[1] := stackSeg;  (* segment *)
      (* HL - 15 Aug 95: Changed CMOSWrite from 15,5 to 15,10 *)
      CMOSWrite(15, 10);  (* Write shutdown status byte *)
      Progress('D');
      SYSTEM.Out(064H, 0FEH);  (* shut down *)
      LOOP Hlt (* to work with cache *) END  (* wait for end *)
    END
  END Relocate;

  (*** Main Program ***)

BEGIN
  InitParameters; InitGlobals; InitMemory; InitTable; InitKernel;
  IF trace THEN Status(Program, " done"); HALT END;
  StopFloppy; InitScreen;
  Status("entering protected mode...", ""); Progress('E');
  SYSTEM.DI;  (* Switch off interrupts *)
  Relocate
END noBoot.
