TextDocs.NewDoc     TF   CColor     Flat  Locked  Controls  Org 	   BIER`   b        3         Oberon12.Scn.Fnt     Syntax12.Scn.Fnt                         Oberon12i.Scn.Fnt     Oberon10i.Scn.Fnt                                       Oberon10.Scn.Fnt                                                                                                  +                                                   /                
       -                                                                       (*
TBox: Set of tools.

Copyright (C) 2001 Grard Meunier

This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License  for more details.

You should have received a copy of the GNU General Public License along with this program; if not, write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*)

MODULE TBoxDiffEq;
	
	
	
	(** Numerical integrator of sets of differential equations; Runge-Kutta method with adaptive stepsize control. Cf. Numerical Recipes, The Art of Scientific Computing. Integrates sets of equations of the form:
dyi/dx= fi(x, y1,..., yi,..., yn) *)
	
	IMPORT
		
		M:= Math;
	
	TYPE
		
		Vecteur*= POINTER TO ARRAY OF REAL; (** Array of real values. *)
		
		EquaDiff*= POINTER TO RECORD (** Set of differential equations dyi/dx= fi(x, y1,..., yi,..., yn). *)
		END;
	
	PROCEDURE Add (v1, v2: Vecteur): Vecteur;
		
		VAR
			
			n, i: LONGINT;
			v: Vecteur;
		
		BEGIN (*Add*)
			ASSERT((v1# NIL) & (v2# NIL));
			n:= LEN(v1);
			ASSERT(n= LEN(v2));
			NEW(v, n);
			FOR i:= 0 TO n- 1 DO
				v[i]:= v1[i]+ v2[i];
			END;
			RETURN v;
		END Add;
	
	PROCEDURE Mul (r: REAL; v: Vecteur): Vecteur;
		
		VAR
			
			i, n: LONGINT;
			u: Vecteur;
		
		BEGIN (*Mul*)
			ASSERT(v# NIL);
			n:= LEN(v);
			NEW(u, n);
			FOR i:= 0 TO n- 1 DO
				u[i]:= r* v[i];
			END;
			RETURN u;
		END Mul;
	
	(** Returns the values fi(x, y1,..., yi,..., yn) of the derivatives of the searched functions y, relative to the variable x in the set eq of differential equations. *)
	PROCEDURE (eq: EquaDiff) Derivs-(x: REAL; y: Vecteur): Vecteur;
		
		BEGIN (*Derivs*)
			HALT(100);
		END Derivs;
	
	PROCEDURE (eq: EquaDiff) Rkck (y, dydx: Vecteur; x,  h: REAL; VAR yerr: Vecteur): Vecteur;
		
		CONST
			
			a2= 1./ 5.; a3= 3./ 10.; a4= 3./ 5.; a5= 1.; a6= 7./ 8.;
			b21= 1./ 5.;
			b31= 3./ 40.; b32= 9./ 40.;
			b41= 3./ 10.; b42= -9./ 10.; b43= 6./ 5.;
			b51= -11./ 54.; b52= 5./ 2.; b53= -70./ 27.; b54= 35./ 27.;
			b61= 1631./ 55296.; b62= 175./ 512.; b63= 575./ 13824.; b64= 44275./ 110592.; b65= 253./ 4096.;
			c1= 37./ 378.; c3= 250./ 621.; c4= 125./ 594.; c6= 512./ 1771.;
			dc1= c1- 2825./ 27648.; dc3= c3- 18575./ 48384.; dc4= c4- 13525./ 55296.; dc5= -277./ 14336.; dc6= c6- 1./ 4.;
		
		VAR
						
			ak2, ak3, ak4, ak5, ak6: Vecteur;
		
		BEGIN (*Rkck*)
			ASSERT((y# NIL) & (dydx# NIL));
			ASSERT(LEN(y)= LEN(dydx));
			ak2:= eq.Derivs(x+ a2* h, Add(y, Mul(h* b21, dydx)));
			ak3:= eq.Derivs(x+ a3* h, Add(y, Mul(h, Add(Mul(b31, dydx), Mul(b32, ak2)))));
			ak4:= eq.Derivs(x+ a4* h, Add(y, Mul(h, Add(Mul(b41, dydx), Add(Mul(b42, ak2), Mul(b43, ak3))))));
			ak5:= eq.Derivs(x+ a5* h, Add(y, Mul(h, Add(Mul(b51, dydx), Add(Mul(b52, ak2), Add(Mul(b53, ak3), Mul(b54, ak4)))))));
			ak6:= eq.Derivs(x+ a6* h, Add(y, Mul(h, Add(Mul(b61, dydx), Add(Mul(b62, ak2), Add(Mul(b63, ak3), Add(Mul(b64, ak4), Mul(b65, ak5))))))));
			yerr:= Mul(h, Add(Mul(dc1, dydx), Add(Mul(dc3, ak3), Add(Mul(dc4, ak4), Add(Mul(dc5, ak5), Mul(dc6, ak6))))));
			RETURN Add(y, Mul(h, Add(Mul(c1, dydx), Add(Mul(c3, ak3), Add(Mul(c4, ak4), Mul(c6, ak6))))));
		END Rkck;
	
	PROCEDURE (eq: EquaDiff) Rkqs (y, dydx, yscal: Vecteur; VAR x, h: REAL; eps: REAL): Vecteur;
		
		CONST
			
			safety= 0.9;
			pgrow= -0.2;
			pschrnk= -0.25;
			errcon= 1.89E-4;
		
		VAR
			
			yres, yerr: Vecteur;
			errmax, r: REAL;
			i, n: LONGINT;
		
		BEGIN (*Rkqs*)
			ASSERT((y# NIL) & (dydx# NIL) & (yscal# NIL));
			n:= LEN(y);
			ASSERT((n= LEN(dydx)) & (n= LEN(yscal)));
			LOOP
				yres:= eq.Rkck(y, dydx, x, h, yerr);
				errmax:= 0.;
				FOR i:= 0 TO n- 1 DO
					r:= ABS(yerr[i]/ yscal[i]);
					IF errmax< r THEN
						errmax:= r;
					END;
				END;
				errmax:= errmax/ eps;
				IF errmax> 1. THEN
					h:= safety* h* M.exp(M.ln(errmax)* pschrnk);
					r:= 0.1* h;
					IF ABS(h)< ABS(r) THEN
						h:= r;
					END;
				ELSE
					x:= x+ h;
					IF errmax> errcon THEN
						h:= safety* h* M.exp(M.ln(errmax)* pgrow);
					ELSE
						h:= 5.* h;
					END;
					RETURN yres;
				END;
			END;
		END Rkqs;
	
	(** Integrates the y starting values of the functions of the set eq of differential equations from x to x + h with accuracy eps. On ouput, x and y are updated. *)
	PROCEDURE (eq: EquaDiff) Step*(VAR y: Vecteur; VAR x: REAL; h, eps: REAL);
		
		CONST
			
			tiny= 1.E-30;
		
		VAR
			
			n, i: LONGINT;
			dydx, yscal: Vecteur;
			x1, x2: REAL;
		
		BEGIN (*Step*)
			ASSERT(y# NIL, 20);
			n:= LEN(y);
			NEW(yscal, n);
			x1:= x; x2:= x+ h;
			LOOP
				dydx:= eq.Derivs(x, y);
				ASSERT(dydx# NIL, 60);
				ASSERT(LEN(dydx)= LEN(y), 61);
				FOR i:= 0 TO n- 1 DO
					yscal[i]:= ABS(y[i])+ ABS(h* dydx[i])+ tiny;
				END;
				IF (x+ h- x2)* (x+ h- x1)> 0. THEN
					h:= x2- x;
				END;
				y:= eq.Rkqs(y, dydx, yscal, x, h, eps);
				IF (x- x2)* (x2- x1)>= 0. THEN
					EXIT;
				END;
			END;
		END Step;
	
	END TBoxDiffEq.
BIER  :         "         d      d
     C  <       g 
     C  Syntax10.Scn.Fnt 28.03.2002  23:55:51  TextGadgets.NewStyleProc TimeStamps.New  