TextDocs.NewDoc     F   CColor     Flat  Locked  Controls  Org !   BIER`   b        3  1   Syntax12.Scn.Fnt             (*
TBox: Set of tools.

Copyright (C) 2001 Grard Meunier

This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License  for more details.

You should have received a copy of the GNU General Public License along with this program; if not, write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*)

MODULE TBoxBArbre;
	
	 
	
	(** Manages a disk file like a heap, allocating and freeing it easily, with the possibility of indexing by BTrees. *)
	
	IMPORT
		
		A:= TBoxAvl, Strings;
	
	CONST
		
		nbPagesMin= 256; (* Minimal number of allocated pages. *)
		
		bNil*= 0; (** Nil pointer in a database. *)
		
		valOct= 100H;
		TSI= SIZE(LONGINT);
		TSB= SIZE(BOOLEAN);
		TSY= SIZE(SHORTINT);
		TSC= SIZE(CHAR);
	
	TYPE
		
		PSHORTINT*= POINTER TO ARRAY OF SHORTINT;
		
		Fabrique*= POINTER TO RECORD (** Factory for files and databases. *)
		END;
		
		Fichier*= POINTER TO RECORD (** File. *)
		END;
		
		BaseDon*= POINTER TO RECORD (A.ANYPTR) (** Database. *)
			ref: Fichier; (* File containing the database. *)
			racine, (* Root of the leftist tree of free clusters. *)
			fin-, (** Length of file. *)
			max, (* Size of the largest free cluster. *)
			nbPlaces-, (** Number of fixed places in the database. *)
			nbPages: LONGINT; (* Number of allocated pages. *)
			pages: A.Tree; (* Buffer structured as tree of pages. *)
			pagesAnneau: Page; (* Buffer structured as ring of pages. *)
			indAnneau: Index; (* Ring of the indexes of the database. *)
		END;
		
		GesCles*= POINTER TO RECORD (** Key manager. *)
			f: DataFab;
		END;
		
		Cha*= POINTER TO ARRAY OF CHAR;
		
		String*= POINTER TO RECORD (Data)
			c*: Cha;
		END;
		
		StringFab*= POINTER TO RECORD (DataFab)
		END;
		
		GesClesString*= POINTER TO RECORD (GesCles) (** Key manager for String. *)
		END;
		
		Index*= POINTER TO RECORD (** Index, structured as a btree. *)
			suivI, precI: Index;
			baseI: BaseDon;
			gestion: GesCles;
			refI, 
			racI, 
			chaine, 
			pCour, 
			taCle, 
			taCles, 
			haut: LONGINT;
		END;
		
		Comp*= SHORTINT; (** Result of a comparaison. *)
		
	CONST
		
		(** Result of a comparaison. *)
		inf*= 0; (** Less than. *)
		ega*= 1; (** Equal. *)
		sup*= 2; (** Greater than. *)
		
		garde1= 1548793025;
		garde2= 2015876309;
	
	TYPE
		
		CBaseTete= POINTER TO RECORD (Data) (* Header of a cluster. *)
			taille: LONGINT; (* Size of the used part of the cluster. *)
		END;
		
		CBaseTeteOcc= POINTER TO RECORD (CBaseTete) (* Header of a reserved cluster. *)
		END;
		
		CBaseTeteOccFab= POINTER TO RECORD (DataFab)
		END;
		
		CBaseTeteLib= POINTER TO RECORD (CBaseTete) (* Header of a free cluster. *)
			(* Structure of leftist tree. Cf. Knuth, The art of computer programming, vol. 3, ch. 5.2.3 and exercises 32 & 35. *)
			pere, 
			gauche, 
			droite: LONGINT;
			distG, 
			distD: SHORTINT;
		END;
		
		CBaseTeteLibFab= POINTER TO RECORD (DataFab)
		END;
		
		CBaseQueue= POINTER TO RECORD (Data) (* Tail of a cluster. *)
		END;
		
		CBaseQueueOcc= POINTER TO RECORD (CBaseQueue) (* Tail of a reserved cluster. *)
		END;
		
		CBaseQueueOccFab= POINTER TO RECORD (DataFab)
		END;
		
		CBaseQueueLib= POINTER TO RECORD (CBaseQueue) (* Tail of a free cluster. *)
			 taille: LONGINT; (* Total size of the cluster. *)
		 END;
		
		CBaseQueueLibFab= POINTER TO RECORD (DataFab)
		END;
	
	CONST
		
		taCBaseTeteOcc= TSB+ TSI;
		taCBaseQueueOcc= TSB;
		taCBaseTeteLib= TSB+ 4*TSI+ 2* TSY;
		taCBaseQueueLib= TSI+ TSB;
		baseTete= 4* TSI+ 2* TSB;
	
	TYPE
		
		Data*= POINTER TO RECORD (** Content of a page, used to store data or a key. *)
		END;
		
		DataFab*= POINTER TO RECORD (** Factory of data. *)
		END;
		
		GesData*= POINTER TO RECORD
			base: BaseDon;
			fab: DataFab;
		END;
		
		Page= POINTER TO RECORD (A.Elem)
			suivant, precedent: Page;
			pageP: Data;
			posP, 
			tailP: LONGINT;
			ecr: BOOLEAN;
		END;
	
	CONST
		
		taSect= 512;
		taPageMax= 2*taSect;
	
	TYPE
		
		Element= RECORD
			ptr,
			finC: LONGINT;
		END;
	
	CONST
		
		taElement= 2* TSI;
		
		minEl= (taPageMax- 2* TSI) DIV (2* taElement)- 1;
	
	TYPE
		
		PageI= POINTER TO RECORD (Data)
			cles,
			nbEl: LONGINT;
			elems: ARRAY 2* (minEl+ 1) OF Element;
		END;
		
		PageIFab= POINTER TO RECORD (DataFab)
		END;
		
		Cles= PSHORTINT;
		
		ClesCont= POINTER TO RECORD (Data)
			c: Cles;
		END;
		
		ClesFab= POINTER TO RECORD (DataFab)
		END;
		
		Chaine= POINTER TO RECORD (Data)
			suivant,
			precedent, 
			posDon: LONGINT;
			clef: Cles;
		END;
		
		ChaineFab= POINTER TO RECORD (DataFab)
		END;
		
		Info= POINTER TO RECORD (Data)
			iRac, 
			iCha,
			iTaC, 
			iH: LONGINT;
		END;
		
		InfoFab=POINTER TO RECORD (DataFab)
		END;
	
	CONST
		
		taPageRec= 2* TSI+ 2* (minEl+ 1)* taElement;
		taChaine= 4* TSI;
		taInfo= 6* TSI;
	
	TYPE
		
		Reader*= RECORD
			len-,	(** Length of the input stream of the reader. *)
			pos-: LONGINT;	(** Position of the reader in the input stream. *)
			ref: Fichier;
			s: PSHORTINT;
		END;
		
		Writer*= RECORD
			ref: Fichier;
			s: A.Tree;
			p: LONGINT;
		END;
		
		Char= POINTER TO RECORD (A.Elem)
			c: SHORTINT;
		END;
	
	VAR
		
		cTLib: CBaseTeteLibFab;
		cTOcc: CBaseTeteOccFab;
		cQLib: CBaseQueueLibFab;
		cQOcc: CBaseQueueOccFab;
		
		pif: PageIFab;
		clf: ClesFab;
		chf: ChaineFab;
		inff: InfoFab;
		
		cNIL: Cles;
		
	(** Comparison method. s1 and s2 are the keys to be compared. The result can be inf, ega or sup. GesCles.CompP must induce a total order on the set of keys. *)
	PROCEDURE (g: GesCles) CompP-(s1, s2: Data): Comp;
		
		BEGIN (*CompP*)
			HALT(100);
		END CompP;
	
	(** Creates a prefix of a key. On input cle1 and cle2 are two keys, with cle1< cle2. On output, cle2 may be modified, but cannot be enlarged, so that it becomes a prefix of its previous value. A prefix Pref(key1, key2) of  the key key2 in relation to the key key1 is defined by:
		1) GesCles.Pref(key1, key2) is a key, which can be compared to other keys by the mean of the method GesCles.CompP;
		2) Pref(key1, key2) is the shortest key with key1< Pref(key1, key2) <= key2;
		3) key1 may have a nul length and, in this case, mut be considered less than any other key.
Prefixes are useful if keys are long or if their lengths vary much. In this case, if GesCles.PrefP is instantiated, database is shorter and searches are faster. *)
	PROCEDURE (g: GesCles) PrefP-(cle1: Data; VAR cle2: Data);
		
		BEGIN (*PrefP*)
		END PrefP;
	
	(** Comparison method of two Strings. Uses the lexical order. *)
	PROCEDURE (g: GesClesString) CompP-(cle1, cle2: Data): Comp;
		
		VAR
			
			i: LONGINT;
		
		BEGIN (*CompP*)
			WITH cle1: String DO
				WITH cle2: String DO
					i:= 0;
					WHILE (cle1.c[i]= cle2.c[i]) & (cle1.c[i]# 0X) DO
						INC(i);
					END;
					IF cle1.c[i]< cle2.c[i] THEN
						RETURN inf;
					ELSIF cle1.c[i]> cle2.c[i] THEN
						RETURN sup;
					ELSE
						RETURN ega;
					END;
				END;
			END;
		END CompP;
	
	(** Comparison method of two Strings. Uses the lexical order. *)
	PROCEDURE (g: GesClesString) PrefP-(cle1: Data; VAR cle2: Data);
		
		VAR
			
			l, l2: LONGINT;
			cle: String;
		
		BEGIN (*PrefP*)
			WITH cle1: String DO
				WITH cle2: String DO
					l2:= Strings.Length(cle2.c^);
					NEW(cle); NEW(cle.c, l2+ 1);
					l:= 0;
					cle.c[l]:= 0X;
					WHILE (l<= l2) & ~((g.CompP(cle1, cle)= inf) & (g.CompP(cle, cle2)<= ega)) DO
						cle.c[l]:= cle2.c[l];
						INC(l);
						cle.c[l]:= 0X;
					END;
				END;
			END;
			cle2:= cle;
		END PrefP;
	
	(** Creates a file with name nF, with the help of fab. Does not open this file. *)
	PROCEDURE (fab: Fabrique) Cree-(nF: ARRAY OF CHAR);
		
		BEGIN (*Cree*)
			HALT(100);
		END Cree;
	
	(** Opens a file with name nF, with the help of fab, and returns this file. Returns NIL if the file does not exist. *)
	PROCEDURE (fab: Fabrique) Ouvre-(nF: ARRAY OF CHAR): Fichier;
		
		BEGIN (*Ouvre*)
			HALT(100);
		END Ouvre;
	
	(** Closes the file ref. *)
	PROCEDURE (ref: Fichier) Ferme-;
		
		BEGIN (*Ferme*)
			HALT(100);
		END Ferme;
	
	(** Flushes the file ref on disk. *)
	PROCEDURE (ref: Fichier) Flush-;
		
		BEGIN (*Flush*)
			HALT(100);
		END Flush;
	
	(** Sets the reading position of the file ref to pos; the firt position is 0. *)
	PROCEDURE (ref: Fichier) PosLis-(pos: LONGINT);
		
		BEGIN (*PosLis*)
			HALT(100);
		END PosLis;
	
	(** Reads the array a from the file ref at reading position. *)
	PROCEDURE (ref: Fichier) Lis-(VAR a: ARRAY OF SHORTINT);
		
		BEGIN (*Lis*)
			HALT(100);
		END Lis;
	
	(** Sets the writing position of the file ref to pos; the firt position is 0. *)
	PROCEDURE (ref: Fichier) PosEcris-(pos: LONGINT);
		
		BEGIN (*PosEcris*)
			HALT(100);
		END PosEcris;
	
	(** Writes the array a to the file ref at writing position. *)
	PROCEDURE (ref: Fichier) Ecris-(VAR a: ARRAY OF SHORTINT);
		
		BEGIN (*Ecris*)
			HALT(100);
		END Ecris;
	
	(** Returns in fin the length of the file ref. *)
	PROCEDURE (ref: Fichier) Fin-(VAR fin: LONGINT);
		
		BEGIN (*Fin*)
			HALT(100);
		END Fin;
	
	(** Truncates the file ref at the length fin. *)
	PROCEDURE (ref: Fichier) Tronque-(fin: LONGINT);
		
		BEGIN (*Tronque*)
			HALT(100);
		END Tronque;
	
	(** Reads and returns a real from the array a at position pos. *)
	PROCEDURE (ref: Fichier) BytesToReal-(VAR a: ARRAY OF SHORTINT; VAR pos: LONGINT): REAL;
		
		BEGIN (*BytesToReal*)
			HALT(100);
		END BytesToReal;
	
	(** Reads and returns a longreal from the array a at position pos. *)
	PROCEDURE (ref: Fichier) BytesToLReal-(VAR a: ARRAY OF SHORTINT; VAR pos: LONGINT): LONGREAL;
		
		BEGIN (*BytesToLReal*)
			HALT(100);
		END BytesToLReal;
	
	(** Returns a pointer to an array of byte coding for the real r. *)
	PROCEDURE (ref: Fichier) RealToBytes-(r: REAL): PSHORTINT;
		
		BEGIN (*RealToBytes*)
			HALT(100);
		END RealToBytes;
	
	(** Returns a pointer to an array of byte coding for the longreal r. *)
	PROCEDURE (ref: Fichier) LRealToBytes-(r: LONGREAL): PSHORTINT;
		
		BEGIN (*LRealToBytes*)
			HALT(100);
		END LRealToBytes;
	
	PROCEDURE (VAR r: Reader) InitReader (ref: Fichier; a: PSHORTINT; pos: LONGINT);
		
		BEGIN (*InitReader*)
			r.ref:= ref;
			r.s:= a;
			r.pos:= pos;
			r.len:= LEN(a);
		END InitReader;
	
	(** Reads a byte from the input stream. *)
	PROCEDURE (VAR r: Reader) InByte*(): SHORTINT;
		
		VAR
			
			c: SHORTINT;
		
		BEGIN (*InByte*)
			ASSERT(r.pos< r.len, 20);
			c:= r.s[r.pos];
			INC(r.pos);
			RETURN c;
		END InByte;
	
	(** Reads an array of bytes from the input stream. *)
	PROCEDURE (VAR r: Reader) InBytes*(VAR b: ARRAY OF SHORTINT);
		
		VAR
			
			i: LONGINT;
		
		BEGIN (*InBytes*)
			ASSERT(r.len- r.pos>= LEN(b));
			FOR i:= 0 TO LEN(b)- 1 DO
				b[i]:= r.s[r.pos];
				INC(r.pos);
			END;
		END InBytes;
	
	(** Reads a boolean from the input stream. *)
	PROCEDURE (VAR r: Reader) InBool*(): BOOLEAN;
		
		VAR
			
			b: BOOLEAN;
		
		BEGIN (*InBool*)
			ASSERT(r.pos< r.len, 20);
			b:= r.s[r.pos]# 0;
			INC(r.pos);
			RETURN b;
		END InBool;
	
	(** Reads a char from the input stream. *)
	PROCEDURE (VAR r: Reader) InChar*(): CHAR;
		
		VAR
			
			i, n, c: LONGINT;
		
		BEGIN (*InChar*)
			ASSERT(r.pos+ TSC<= r.len, 20);
			n:= 0;
			FOR i:= r.pos+ TSC- 1 TO r.pos BY -1 DO
				c:= r.s[i];
				IF c< 0 THEN
					INC(c, valOct);
				END;
				n:= n* valOct+ c;
			END;
			INC(r.pos, TSC);
			RETURN CHR(n);
		END InChar;
	
	(** Reads an array of chars from the input stream. Reading stops when a 0X has been read, or when LEN(c)- 1 chars have been read, or when the end of the input stream has been reached. *)
	PROCEDURE (VAR r: Reader) InChars*(VAR c: ARRAY OF CHAR);
		
		VAR
			
			i, n, m: LONGINT;
		
		BEGIN (*InChars*)
			i:= 0; n:= r.len; m:= LEN(c)- 1;
			LOOP
				IF (r.pos= n) OR (i= m) THEN
					c[i]:= 0X;
					EXIT;
				END;
				c[i]:= r.InChar();
				IF c[i]= 0X THEN
					EXIT;
				END;
				INC(i);
			END;
		END InChars;
	
	(** Reads an integer from the input stream. *)
	PROCEDURE (VAR r: Reader) InSInt*(): INTEGER;
		
		VAR
			
			i: LONGINT;
			n, c: INTEGER;
		
		BEGIN (*InSInt*)
			ASSERT(r.pos+ SIZE(INTEGER)<= r.len, 20);
			n:= 0;
			FOR i:= r.pos+ SIZE(INTEGER)- 1 TO r.pos BY -1 DO
				c:= r.s[i];
				IF c< 0 THEN
					INC(c, valOct);
				END;
				n:= SHORT(n* valOct+ c);
			END;
			INC(r.pos, SIZE(INTEGER));
			RETURN n;
		END InSInt;
	
	(** Reads a longint from the input stream. *)
	PROCEDURE (VAR r: Reader) InInt*(): LONGINT;
		
		VAR
			
			i, n, c: LONGINT;
		
		BEGIN (*InInt*)
			ASSERT(r.pos+ TSI<= r.len, 20);
			n:= 0;
			FOR i:= r.pos+ TSI- 1 TO r.pos BY -1 DO
				c:= r.s[i];
				IF c< 0 THEN
					INC(c, valOct);
				END;
				n:= n* valOct+ c;
			END;
			INC(r.pos, TSI);
			RETURN n;
		END InInt;
	
	(** Reads a longreal from the input stream. Uses Fichier.BytesToSReal. *)
	PROCEDURE (VAR r: Reader) InLReal*(): LONGREAL;
		
		BEGIN (*InLReal*)
			RETURN r.ref.BytesToLReal(r.s^, r.pos);
		END InLReal;
	
	(** Reads a real from the input stream. Uses Fichier.BytesToReal. *)
	PROCEDURE (VAR r: Reader) InReal*(): REAL;
		
		BEGIN (*InReal*)
			RETURN r.ref.BytesToReal(r.s^, r.pos);
		END InReal;
	
	(** Reads a set from the input stream. *)
	PROCEDURE (VAR r: Reader) InSet*(): SET;
		
		VAR
			
			s: SET;
			i, j, k, c: LONGINT;
		
		BEGIN (*InSet*)
			ASSERT(r.pos+ SIZE(SET)<= r.len, 20);
			s:= {}; k:= 0;
			FOR i:= r.pos TO r.pos+ SIZE(SET)- 1 DO
				c:= r.s[i];
				IF c< 0 THEN
					INC(c, valOct);
				END;
				FOR j:= 1 TO TSY DO
					IF ODD(c) THEN
						INCL(s, k);
					END;
					c:= c DIV 2;
					INC(k);
				END;
			END;
			INC(r.pos, SIZE(SET));
			RETURN s;
		END InSet;
	
	PROCEDURE (VAR w: Writer) InitWriter (ref: Fichier);
		
		BEGIN (*InitWriter*)
			w.ref:= ref;
			A.New(w.s);
			w.p:= 0;
		END InitWriter;
	
	PROCEDURE (VAR w: Writer) Write (): PSHORTINT;
		
		VAR
			
			a: PSHORTINT;
			e: A.Elem;
			i: LONGINT;
		
		BEGIN (*Write*)
			NEW(a, w.s.NumberOfElems());
			e:= w.s.Next(NIL);
			i:= 0;
			WHILE e# NIL DO
				a[i]:= e(Char).c;
				e:= w.s.Next(e);
				INC(i);
			END;
			RETURN a;
		END Write;
	
	(** Writes a byte to the output stream. *)
	PROCEDURE (VAR w: Writer) OutByte*(c: SHORTINT);
		
		VAR
			
			ch: Char;
		
		BEGIN (*OutByte*)
			NEW(ch);
			w.s.Append(ch);
			ch.c:= c;
			INC(w.p);
		END OutByte;
	
	(** Writes an array of bytes to the output stream. *)
	PROCEDURE (VAR w: Writer) OutBytes*(VAR b: ARRAY OF SHORTINT);
		
		VAR
			
			i: LONGINT;
		
		BEGIN (*OutBytes*)
			FOR i:= 0 TO LEN(b)- 1 DO
				w.OutByte(b[i]);
			END;
		END OutBytes;
	
	(** Writes a boolean to the output stream. *)
	PROCEDURE (VAR w: Writer) OutBool*(b: BOOLEAN);
		
		BEGIN (*OutBool*)
			IF b THEN
				w.OutByte(1);
			ELSE
				w.OutByte(0);
			END;
		END OutBool;
	
	(** Writes a char to the output stream. *)
	PROCEDURE (VAR w: Writer) OutChar*(c: CHAR);
		
		VAR
			
			i, n: LONGINT;
		
		BEGIN (*OutChar*)
			n:= ORD(c);
			FOR i:= 1 TO TSC DO
				w.OutByte(SHORT(SHORT(n MOD valOct)));
				n:= n DIV valOct;
			END;
		END OutChar;
	
	(** Writes an array of chars to the output stream, up to, and including, the first 0X char. *)
	PROCEDURE (VAR w: Writer) OutChars*(VAR c: ARRAY OF CHAR);
		
		VAR
			
			i: LONGINT;
		
		BEGIN (*OutChars*)
			i:= -1;
			REPEAT
				INC(i);
				w.OutChar(c[i]);
			UNTIL c[i]= 0X;
		END OutChars;
	
	(** Writes an integer to the output stream. *)
	PROCEDURE (VAR w: Writer) OutSInt*(n: INTEGER);
		
		VAR
			
			i: LONGINT;
		
		BEGIN (*OutSInt*)
			FOR i:= 1 TO SIZE(INTEGER) DO
				w.OutByte(SHORT(n MOD valOct));
				n:= SHORT(n DIV valOct);
			END;
		END OutSInt;
	
	(** Writes a longint to the output stream. *)
	PROCEDURE (VAR w: Writer) OutInt*(n: LONGINT);
		
		VAR
			
			i: LONGINT;
		
		BEGIN (*OutInt*)
			FOR i:= 1 TO TSI DO
				w.OutByte(SHORT(SHORT(n MOD valOct)));
				n:= n DIV valOct;
			END;
		END OutInt;
	
	(** Writes a longreal to the output stream. Uses Fichier.SRealToBytes. *)
	PROCEDURE (VAR w: Writer) OutLReal*(r: LONGREAL);
		
		VAR
			
			p: PSHORTINT;
		
		BEGIN (*OutLReal*)
			p:= w.ref.LRealToBytes(r);
			w.OutBytes(p^);
		END OutLReal;
	
	(** Writes a real to the output stream. Uses Fichier.RealToBytes. *)
	PROCEDURE (VAR w: Writer) OutReal*(r: REAL);
		
		VAR
			
			p: PSHORTINT;
		
		BEGIN (*OutReal*)
			p:= w.ref.RealToBytes(r);
			w.OutBytes(p^);
		END OutReal;
	
	(** Writes a set to the output stream. *)
	PROCEDURE (VAR w: Writer) OutSet*(s: SET);
		
		VAR
			
			i, j, n: LONGINT;
		
		BEGIN (*OutSet*)
			FOR i:= 0 TO 8*(SIZE(SET)- 1) BY 8 DO
				n:= 0;
				FOR j:= 7 TO 0 BY -1 DO
					n:= 2* n;
					IF i+ j IN s THEN
						INC(n);
					END;
				END;
				w.OutByte(SHORT(SHORT(n)));
			END;
		END OutSet;
	
	PROCEDURE (ref: Fichier) EcrisInt (n: LONGINT);
		
		VAR
			
			w: Writer;
			p: PSHORTINT;
		
		BEGIN (*EcrisInt*)
			w.InitWriter(ref);
			w.OutInt(n);
			p:= w.Write();
			ref.Ecris(p^);
		END EcrisInt;
	
	PROCEDURE (ref: Fichier) EcrisBool (b: BOOLEAN);
		
		VAR
			
			a: ARRAY 1 OF SHORTINT;
		
		BEGIN (*EcrisBool*)
			IF b THEN
				a[0]:= 1;
			ELSE
				a[0]:= 0;
			END;
			ref.Ecris(a);
		END EcrisBool;
	
	(** Creates data of length ta (bytes), with the help of f. *)
	PROCEDURE (f: DataFab) New-(ta: LONGINT): Data;
		
		BEGIN (*New*)
			HALT(100);
		END New;
	
	(** Writes data d with the help of the writer w. *)
	PROCEDURE (d: Data) Write-(VAR w: Writer);
		
		BEGIN (*Write*)
			HALT(100);
		END Write;
	
	(** Reads data d with the help of the reader r. *)
	PROCEDURE (d: Data) Read-(VAR r: Reader);
		
		BEGIN (*Read*)
			HALT(100);
		END Read;
	
	(** Creates data containing an array of char of length l (chars) and ta= 2* l (bytes), with the help of f. *)
	PROCEDURE (f: StringFab) New-(ta: LONGINT): Data;
		
		VAR
			
			s: String;
		
		BEGIN (*New*)
			ASSERT(ta MOD SIZE(CHAR)= 0, 20);
			NEW(s); NEW(s.c, ta DIV SIZE(CHAR)+ 1);
			RETURN s;
		END New;
	
	(** Writes data s containing an array of char with the help of the writer w. *)
	PROCEDURE (s: String) Write-(VAR w: Writer);
		
		BEGIN (*Write*)
			w.OutChars(s.c^);
		END Write;
	
	(** Reads data s containing an array of char with the help of the reader r. *)
	PROCEDURE (s: String) Read-(VAR r: Reader);
		
		BEGIN (*Read*)
			r.InChars(s.c^);
		END Read;
	
	(** Creates and returns a new manager of the data created by fab in the database base. *)
	PROCEDURE (base: BaseDon) GesDataCree*(fab: DataFab): GesData;
		
		VAR
			
			g: GesData;
		
		BEGIN (*GesDataCree*)
			ASSERT(fab# NIL, 20);
			NEW(g);
			g.base:= base;
			g.fab:= fab;
			RETURN g;
		END GesDataCree;
	
	PROCEDURE (f: CBaseTeteLibFab) New (ta: LONGINT): Data;
		
		VAR
			
			p: CBaseTeteLib;
		
		BEGIN (*New*)
			ASSERT(ta= taCBaseTeteLib);
			NEW(p);
			RETURN p;
		END New;
	
	PROCEDURE (t: CBaseTeteLib) Write (VAR w: Writer);
		
		BEGIN (*Write*)
			w.OutBool(TRUE);
			w.OutInt(t.taille);
			w.OutInt(t.pere);
			w.OutInt(t.gauche);
			w.OutInt(t.droite);
			w.OutByte(t.distG);
			w.OutByte(t.distD);
		END Write;
	
	PROCEDURE (t: CBaseTeteLib) Read (VAR r: Reader);
		
		BEGIN (*Read*)
			ASSERT(r.InBool());
			t.taille:= r.InInt();
			t.pere:= r.InInt();
			t.gauche:= r.InInt();
			t.droite:= r.InInt();
			t.distG:= r.InByte();
			t.distD:= r.InByte();
		END Read;
	
	PROCEDURE (f: CBaseQueueLibFab) New (ta: LONGINT): Data;
		
		VAR
			
			p: CBaseQueueLib;
		
		BEGIN (*New*)
			ASSERT(ta= taCBaseQueueLib);
			NEW(p);
			RETURN p;
		END New;
	
	PROCEDURE (q: CBaseQueueLib) Write (VAR w: Writer);
		
		BEGIN (*Write*)
			w.OutInt(q.taille);
			w.OutBool(TRUE);
		END Write;
	
	PROCEDURE (q: CBaseQueueLib) Read (VAR r: Reader);
		
		BEGIN (*Read*)
			q.taille:= r.InInt();
			ASSERT(r.InBool());
		END Read;
	
	PROCEDURE (f: CBaseTeteOccFab) New (ta: LONGINT): Data;
		
		VAR
			
			p: CBaseTeteOcc;
		
		BEGIN (*New*)
			ASSERT(ta= taCBaseTeteOcc);
			NEW(p);
			RETURN p;
		END New;
	
	PROCEDURE (t: CBaseTeteOcc) Write (VAR w: Writer);
		
		BEGIN (*Write*)
			w.OutBool(FALSE);
			w.OutInt(t.taille);
		END Write;
	
	PROCEDURE (t: CBaseTeteOcc) Read (VAR r: Reader);
		
		BEGIN (*Read*)
			ASSERT(~r.InBool());
			t.taille:= r.InInt();
		END Read;
	
	PROCEDURE (f: CBaseQueueOccFab) New (ta: LONGINT): Data;
		
		VAR
			
			p: CBaseQueueOcc;
		
		BEGIN (*New*)
			ASSERT(ta= taCBaseQueueOcc);
			NEW(p);
			RETURN p;
		END New;
	
	PROCEDURE (t: CBaseQueueOcc) Write (VAR w: Writer);
		
		BEGIN (*Write*)
			w.OutBool(FALSE);
		END Write;
	
	PROCEDURE (q: CBaseQueueOcc) Read (VAR r: Reader);
		
		BEGIN (*Read*)
			ASSERT(~r.InBool());
		END Read;
	
	PROCEDURE (ref: Fichier) LisInt (): LONGINT;
		
		VAR
			
			a: PSHORTINT;
			r: Reader;
		
		BEGIN (*LisInt*)
			NEW(a, TSI);
			ref.Lis(a^);
			r.InitReader(ref, a, 0);
			RETURN r.InInt();
		END LisInt;
	
	PROCEDURE (ref: Fichier) LisBool (): BOOLEAN;
		
		VAR
			
			a: ARRAY TSY OF SHORTINT;
		
		BEGIN (*LisBool*)
			ref.Lis(a);
			RETURN a[0]# 0;
		END LisBool;
	
	PROCEDURE (base: BaseDon) LisBase (ptr: LONGINT; VAR a: ARRAY OF SHORTINT);
		
		BEGIN (*LisBase*)
			base.ref.PosLis(ptr);
			base.ref.Lis(a);
		END LisBase;
	
	PROCEDURE (base: BaseDon) EcrisBase (ptr: LONGINT; VAR a: ARRAY OF SHORTINT);
		
		BEGIN (*EcrisBase*)
			base.ref.PosEcris(ptr);
			base.ref.Ecris(a);
		END EcrisBase;
	
	PROCEDURE (p1: Page) Compare (p2: A.Elem): SHORTINT;
		
		BEGIN (*Compare*)
			WITH p2: Page DO
				IF p1.posP< p2.posP THEN
					RETURN A.inf;
				ELSIF p1.posP> p2.posP THEN
					RETURN A.sup;
				ELSE
					RETURN A.ega;
				END;
			END;
		END Compare;
	
	PROCEDURE (base: BaseDon) TrouvePage (pos: LONGINT; VAR p: Page): BOOLEAN;
		
		VAR
			
			n: LONGINT;
			q: A.Elem;
		
		BEGIN (*TrouvePage*)
			NEW(p);
			p.posP:= pos;
			q:= p;
			IF base.pages.Search(q, n) THEN
				p:= q(Page);
				RETURN TRUE;
			END;
			RETURN FALSE;
		END TrouvePage;
	
	PROCEDURE (base: BaseDon) DeplacePage (p: Page);
		
		BEGIN (*DeplacePage*)
			p.suivant.precedent:= p.precedent;
			p.precedent.suivant:= p.suivant;
			p.suivant:= base.pagesAnneau.suivant;
			p.precedent:= base.pagesAnneau;
			base.pagesAnneau.suivant.precedent:= p;
			base.pagesAnneau.suivant:= p;
		END DeplacePage;
	
	PROCEDURE (base: BaseDon) CreePage (taPage, pos: LONGINT): Page;
		
		VAR
			
			p: Page;
			q: A.Elem;
			b: BOOLEAN;
			n: LONGINT;
			w: Writer;
			pS: PSHORTINT;
		
		BEGIN (*CreePage*)
			IF base.pages.NumberOfElems()< base.nbPages THEN
				NEW(p);
				p.suivant:= base.pagesAnneau.suivant;
				p.precedent:= base.pagesAnneau;
				base.pagesAnneau.suivant.precedent:= p;
				base.pagesAnneau.suivant:= p
			ELSE
				p:= base.pagesAnneau.precedent;
				base.DeplacePage(p);
				IF p.ecr THEN
					n:= p.posP;
					IF p.pageP IS CBaseQueue THEN
						DEC(n, p.tailP- 1);
					END;
					w.InitWriter(base.ref);
					p.pageP.Write(w);
					pS:= w.Write();
					base.EcrisBase(n, pS^);
				END;
				b:= base.pages.Delete(p);
				ASSERT(b);
			END;
			p.ecr:= FALSE;
			p.posP:= pos;
			p.pageP:= NIL;
			p.tailP:= taPage;
			q:= p;
			b:= base.pages.SearchIns(q, n);
			ASSERT(~b);
			RETURN p;
		END CreePage;
	
	PROCEDURE (base: BaseDon) NewPage (taPage: LONGINT; fab: DataFab; pos: LONGINT): Data;
		
		VAR
			
			p: Page;
		
		BEGIN (*NewPage*)
			p:= base.CreePage(taPage, pos);
			p.pageP:= fab.New(taPage);
			RETURN p.pageP;
		END NewPage;
	
	PROCEDURE (base: BaseDon) ErasePage (pos: LONGINT);
		
		VAR
			
			p: Page;
			b: BOOLEAN;
		
		BEGIN (*ErasePage*)
			IF base.TrouvePage(pos, p) THEN
				p.suivant.precedent:= p.precedent;
				p.precedent.suivant:= p.suivant;
				b:= base.pages.Delete(p);
				ASSERT(b);
			END;
		END ErasePage;
	
	PROCEDURE (base: BaseDon) SelectPageSys (pos, taPage: LONGINT): Page;
		
		VAR
			
			p: Page;
		
		BEGIN (*SelectPageSys*)
			IF base.TrouvePage(pos, p) THEN
				base.DeplacePage(p);
			ELSE
				p:= base.CreePage(taPage, pos);
			END;
			RETURN p;
		END SelectPageSys;
	
	PROCEDURE (base: BaseDon) LisPageSys (pos, taPage: LONGINT; fab: DataFab): Data;
		
		VAR
			
			p: Page;
			a: PSHORTINT;
			r: Reader;
		
		BEGIN (*LisPageSys*)
			p:= base.SelectPageSys(pos, taPage);
			IF p.pageP= NIL THEN
				NEW(a, p.tailP);
				p.pageP:= fab.New(p.tailP);
				IF p.pageP IS CBaseQueue THEN
					DEC(pos, p.tailP- 1);
				END;
				base.LisBase(pos, a^);
				r.InitReader(base.ref, a, 0);
				p.pageP.Read(r);
			END;
			RETURN p.pageP;
		END LisPageSys;
	
	PROCEDURE (base: BaseDon) EcrisPage (pos: LONGINT);
		
		VAR
			
			p: Page;
		
		BEGIN (*EcrisPage*)
			ASSERT(base.TrouvePage(pos, p) & (p.pageP# NIL), 100);
			p.ecr:= TRUE;
		END EcrisPage;
	
	PROCEDURE (base: BaseDon) TeteLibre (pos: LONGINT; VAR cont: CBaseTete): BOOLEAN;
		
		VAR
			
			p: Page;
			b: BOOLEAN;
			d: Data;
		
		BEGIN (*TeteLibre*)
			IF base.TrouvePage(pos, p) THEN
				cont:= p.pageP(CBaseTete);
				RETURN cont IS CBaseTeteLib;
			ELSE
				base.ref.PosLis(pos);
				b:= base.ref.LisBool();
				IF b THEN
					d:= base.LisPageSys(pos, taCBaseTeteLib, cTLib);
				ELSE
					d:= base.LisPageSys(pos, taCBaseTeteOcc, cTOcc);
				END;
				cont:= d(CBaseTete);
				RETURN b;
			END;
		END TeteLibre;
	
	PROCEDURE (base: BaseDon) QueueLibre (pos: LONGINT; VAR cont: CBaseQueue): BOOLEAN;
		
		VAR
			
			p: Page;
			b: BOOLEAN;
			d: Data;
		
		BEGIN (*QueueLibre*)
			IF base.TrouvePage(pos, p) THEN
				cont:= p.pageP(CBaseQueue);
				RETURN cont IS CBaseQueueLib;
			ELSE
				base.ref.PosLis(pos);
				b:= base.ref.LisBool();
				IF b THEN
					d:= base.LisPageSys(pos, taCBaseQueueLib, cQLib);
				ELSE
					d:= base.LisPageSys(pos, taCBaseQueueOcc, cQOcc);
				END;
				cont:= d(CBaseQueue);
				RETURN b;
			END;
		END QueueLibre;
	
	PROCEDURE (p: Page) Do (base: A.ANYPTR);
		
		VAR
			
			n: LONGINT;
			w: Writer;
			pS: PSHORTINT;
		
		BEGIN (*Do*)
			IF p.ecr THEN
				p.ecr:= FALSE;
				n:= p.posP;
				IF p.pageP IS CBaseQueue THEN
					DEC(n, p.tailP- 1);
				END;
				w.InitWriter(base(BaseDon).ref);
				p.pageP.Write(w);
				pS:= w.Write();
				base(BaseDon).EcrisBase(n, pS^);
			END;
		END Do;
	
	PROCEDURE (base: BaseDon) MAJPages;
		
		BEGIN (*MAJPages*)
			base.pages.WalkThrough(base);
		END MAJPages;
	
	(** Creates a new file of name nF, with the help of fab, and a database inside this file, with nbPlaces fixed places. Does not open this database. Fixed places are locations where can be recorded integers, that is data pointers. *)
	PROCEDURE (fab: Fabrique) CreeBase*(nF: ARRAY OF CHAR; nbPlaces: LONGINT);
		
		VAR
			
			ref: Fichier;
			i: LONGINT;
		
		BEGIN (*CreeBase*)
			fab.Cree(nF);
			ref:= fab.Ouvre(nF);
			ref.PosEcris(0);
			ref.EcrisInt(garde1);
			ref.EcrisInt(bNil);
			ref.EcrisInt(garde2);
			ref.EcrisInt(nbPlaces);
			FOR i:= 1 TO nbPlaces DO
				ref.EcrisInt(bNil);
			END;
			ref.EcrisBool(FALSE);
			ref.EcrisBool(FALSE);
			ref.Ferme;
		END CreeBase;
	
	(** Opens and returns, with the help of fab, the database created in the file of name nF. nbPages is the maximal number of allocated buffer pages. Returns NIL if the file does not exit. *)
	PROCEDURE (fab: Fabrique) OuvreBase*(nF: ARRAY OF CHAR; nbPages: LONGINT): BaseDon;
		
		VAR
			
			base: BaseDon;
			d: Data;
		
		BEGIN (*OuvreBase*)
			NEW(base);
			base.ref:= fab.Ouvre(nF);
			IF base.ref= NIL THEN
				RETURN NIL;
			END;
			base.ref.PosLis(0);
			ASSERT(base.ref.LisInt()= garde1, 20);
			base.racine:= base.ref.LisInt();
			ASSERT(base.ref.LisInt()= garde2, 20);
			base.nbPlaces:= base.ref.LisInt();
			base.ref.PosLis((base.nbPlaces+ 4)*TSI);
			ASSERT(~base.ref.LisBool(), 21);
			base.ref.PosEcris((base.nbPlaces+ 4)*TSI);
			base.ref.EcrisBool(TRUE);
			base.ref.Fin(base.fin);
			IF nbPages< nbPagesMin THEN
				nbPages:= nbPagesMin;
			END;
			base.nbPages:= nbPages;
			A.New(base.pages);
			NEW(base.pagesAnneau);
			base.pagesAnneau.suivant:= base.pagesAnneau;
			base.pagesAnneau.precedent:= base.pagesAnneau;
			NEW(base.indAnneau);
			base.indAnneau.suivI:= base.indAnneau;
			base.indAnneau.precI:= base.indAnneau;
			IF base.racine= bNil THEN
				base.max:= 0;
			ELSE
				d:= base.LisPageSys(base.racine, taCBaseTeteLib,cTLib);
				base.max:= d(CBaseTeteLib).taille;
			END;
			RETURN base;
		END OuvreBase;
	
	(** Closes, with the help of fab, the database created in the file of name nF, if it exists and is open. It is a rescue procedure. Use it only in case of an accidentally kept open database. *)
	PROCEDURE (fab: Fabrique) FermeBase*(nF: ARRAY OF CHAR);
		
		VAR
			
			ref: Fichier;
			nbPlaces: LONGINT;
		
		BEGIN (*FermeBase*)
			ref:= fab.Ouvre(nF);
			IF ref# NIL THEN
				ref.PosLis(0);
				ASSERT(ref.LisInt()= garde1, 20);
				ref.PosLis(2* TSI);
				ASSERT(ref.LisInt()= garde2, 20);
				nbPlaces:= ref.LisInt();
				ref.PosEcris((nbPlaces+ 4)*TSI);
				ref.EcrisBool(FALSE);
				ref.Ferme;
			END;
		END FermeBase;
	
	(** Updates the database base on disk. *)
	PROCEDURE (base: BaseDon) MAJBase*;
		
		BEGIN (*MAJBase*)
			ASSERT(base.ref# NIL, 20);
			base.MAJPages;
			base.ref.PosEcris(TSI);
			base.ref.EcrisInt(base.racine);
			base.ref.Flush;
		END MAJBase;
	
	(** Closes the index ind. *)
	PROCEDURE (ind: Index) Ferme*;
		
		BEGIN (*Ferme*)
			ASSERT(ind.baseI# NIL, 20);
			ind.precI.suivI:= ind.suivI;
			ind.suivI.precI:= ind.precI;
			ind.baseI:= NIL;
		END Ferme;
	
	(** Closes the database base. *)
	PROCEDURE (base: BaseDon) FermeBase*;
		
		VAR
			
			ind: Index;
		
		BEGIN (*FermeBase*)
			ASSERT(base.ref# NIL, 20);
			base.MAJPages;
			base.ref.PosEcris(TSI);
			base.ref.EcrisInt(base.racine);
			base.ref.PosEcris((base.nbPlaces+ 4)*TSI);
			base.ref.EcrisBool(FALSE);
			base.ref.Ferme;
			base.ref:= NIL;
			base.pages:= NIL;
			base.pagesAnneau:= NIL;
			ind:= base.indAnneau.suivI;
			WHILE ind# base.indAnneau DO
				ind.Ferme;
				ind:= ind.suivI;
			END;
			base.indAnneau:= NIL;
		END FermeBase;
	
	PROCEDURE (base: BaseDon) LisBaseLong (ptr: LONGINT): LONGINT;
		
		BEGIN (*LisBaseLong*)
			base.ref.PosLis(ptr- TSI);
			RETURN base.ref.LisInt();
		END LisBaseLong;
	
	(** Joins the two leftist trees of roots pointed by rac1 and rac2, and returns a pointer to the new root in racine. base is the database. papa is a pointer to the father of the root. cont1 and cont2 are the roots of the two trees. On return, dist contains the field distD of the new root + 1. *)
	PROCEDURE (base: BaseDon) Melange (papa, rac1, rac2: LONGINT; cont1, cont2: CBaseTeteLib; VAR racine: LONGINT; VAR dist: SHORTINT);
		
		VAR
			
			cont, aCont: CBaseTeteLib;
			tmpA: LONGINT;
			tmpB: SHORTINT;
			d: Data;
		
		BEGIN (*Melange*)
			IF rac1= bNil THEN
				racine:= rac2;
				IF rac2= bNil THEN
					dist:= 1;
				ELSE
					dist:= cont2.distD; INC(dist);
					IF cont2.pere# papa THEN
						cont2.pere:= papa;
						base. EcrisPage(rac2);
					END;
				END;
			ELSIF rac2= bNil THEN
				racine:= rac1;
				dist:= cont1.distD; INC(dist);
				IF cont1.pere# papa THEN
					cont1.pere:= papa;
					base. EcrisPage(rac1);
				END;
			ELSIF (cont1.taille> cont2.taille) OR (cont1.taille= cont2.taille) & (rac1< rac2) THEN
				racine:= rac1;
				NEW(aCont);
				aCont^:= cont1^;
				cont1.pere:= papa;
				IF cont1.droite# bNil THEN
					d:= base.LisPageSys(cont1.droite, taCBaseTeteLib, cTLib);
					cont:= d(CBaseTeteLib);
				END;
				base.Melange(rac1, cont1.droite, rac2, cont, cont2, cont1.droite, cont1.distD);
				IF cont1.distG< cont1.distD THEN
					tmpA:= cont1.gauche; cont1.gauche:= cont1.droite; cont1.droite:= tmpA;
					tmpB:= cont1.distG; cont1.distG:= cont1.distD; cont1.distD:= tmpB;
				END;
				dist:= cont1.distD; INC(dist);
				IF (cont1.pere# aCont.pere) OR (cont1.gauche# aCont.gauche) OR (cont1.droite# aCont.droite) OR (cont1.distG# aCont.distG) OR (cont1.distD# aCont.distD) THEN
					base. EcrisPage(rac1);
				END;
			ELSE
				racine:= rac2;
				NEW(aCont);
				aCont^:= cont2^;
				cont2.pere:= papa;
				IF cont2.droite# bNil THEN
					d:= base.LisPageSys(cont2.droite, taCBaseTeteLib, cTLib);
					cont:= d(CBaseTeteLib);
				END;
				base.Melange(rac2, rac1, cont2.droite, cont1, cont, cont2.droite, cont2.distD);
				IF cont2.distG< cont2.distD THEN
					tmpA:= cont2.gauche; cont2.gauche:= cont2.droite; cont2.droite:= tmpA;
					tmpB:= cont2.distG; cont2.distG:= cont2.distD; cont2.distD:= tmpB;
				END;
				dist:= cont2.distD; INC(dist);
				IF (cont2.pere# aCont.pere) OR (cont2.gauche# aCont.gauche) OR (cont2.droite# aCont.droite) OR (cont2.distG# aCont.distG) OR (cont2.distD# aCont.distD) THEN
					base. EcrisPage(rac2);
				END;
			END;
		END Melange;
	
	(** Inserts the free header cluster cont , pointed by ptr, in the leftist tree of root pointed by racine. If the root is modified, racine points to the new one. base is the database. If the size of the greatest cluster is modified, max returns the new value. *)
	PROCEDURE (base: BaseDon) InsLibBase (ptr: LONGINT; cont: CBaseTeteLib; VAR racine, max: LONGINT);
		
		VAR
			
			rac: LONGINT;
			dist: SHORTINT;
			cont2: CBaseTeteLib;
			d: Data;
		
		BEGIN (*InsLibBase*)
			cont.pere:= bNil;
			cont.gauche:= bNil;
			cont.droite:= bNil;
			cont.distG:= 1;
			cont.distD:= 1;
			rac:= racine;
			IF racine= bNil THEN
				base.EcrisPage(ptr);
			ELSE
				d:= base.LisPageSys(racine, taCBaseTeteLib, cTLib);
				cont2:= d(CBaseTeteLib);
			END;
			base.Melange(bNil, racine, ptr, cont2, cont, racine, dist);
			IF racine= ptr THEN
				max:= cont.taille;
			ELSIF racine# rac THEN
				d:= base.LisPageSys(racine, taCBaseTeteLib, cTLib);
				max:= d(CBaseTeteLib).taille;
			END;
		END InsLibBase;
	
	(** Removes the free header cluster cont, pointed by ptr, from the leftist tree of root racine. If the root is modified, racine points to the new one. base is the database. If the size of the greatest cluster is modified, max returns the new value. *)
	PROCEDURE (base: BaseDon) SupLibBase (ptr: LONGINT; cont: CBaseTeteLib; VAR racine, max: LONGINT);
		
		VAR
			
			contG, contD: CBaseTeteLib;
			fils, nPtr: LONGINT;
			dist, aDist: SHORTINT;
			d: Data;
		
		BEGIN (*SupLibBase*)
			IF cont.gauche# bNil THEN
				d:= base.LisPageSys(cont.gauche, taCBaseTeteLib, cTLib);
				contG:= d(CBaseTeteLib);
			END;
			IF cont.droite# bNil THEN
				d:= base.LisPageSys(cont.droite, taCBaseTeteLib, cTLib);
				contD:= d(CBaseTeteLib);
			END;
			base.Melange(cont.pere, cont.gauche, cont.droite, contG, contD, fils, dist);
			nPtr:= cont.pere;
			IF nPtr= bNil THEN
				racine:= fils;
				IF racine= bNil THEN
					max:= 0;
				ELSE
					d:= base.LisPageSys(racine, taCBaseTeteLib, cTLib);
					cont:= d(CBaseTeteLib);
					max:= cont.taille;
				END;
			ELSE
				LOOP
					d:= base.LisPageSys(nPtr, taCBaseTeteLib, cTLib);
					cont:= d(CBaseTeteLib);
					aDist:= cont.distD;
					IF cont.gauche= ptr THEN
						cont.gauche:= fils;
						cont.distG:= dist;
					ELSE
						cont.droite:= fils;
						cont.distD:= dist;
					END;
					IF cont.distD> cont.distG THEN
						fils:= cont.gauche; cont.gauche:= cont.droite; cont.droite:= fils;
						dist:= cont.distG; cont.distG:= cont.distD; cont.distD:= dist;
					END;
					base.EcrisPage(nPtr);
					IF cont.distD= aDist THEN
						EXIT;
					END;
					dist:= cont.distD; INC(dist);
					ptr:= nPtr;
					fils:= nPtr;
					nPtr:= cont.pere;
					IF nPtr= bNil THEN
						EXIT;
					END;
				END;
			END;
		END SupLibBase;
	
	PROCEDURE (base: BaseDon) ReserveBase (taille: LONGINT): LONGINT;
		
		VAR
			
			ta, taReste, ptr: LONGINT;
			cont: CBaseTeteLib;
			contQ: CBaseQueueLib;
			contTO: CBaseTeteOcc;
			contQO: CBaseQueueOcc;
			d: Data;
		
		BEGIN (*ReserveBase*)
			ta:= ((taille+ taCBaseTeteOcc+ taCBaseQueueOcc- 1) DIV (taCBaseTeteLib+ taCBaseQueueLib)+ 1)*(taCBaseTeteLib+ taCBaseQueueLib);
			IF base.max< ta THEN
				ptr:= base.fin;
				INC(base.fin, ta);
			ELSE
				taReste:= base.max- ta;
				ptr:= base.racine;
				d:= base.LisPageSys(ptr, taCBaseTeteLib, cTLib);
				cont:= d(CBaseTeteLib);
				base.SupLibBase(ptr, cont, base.racine, base.max);
				base.ErasePage(ptr);
				IF taReste= 0 THEN
					base.ErasePage(ptr+ ta -1);
				ELSE
					d:= base.LisPageSys(ptr+ ta+ taReste- 1, taCBaseQueueLib, cQLib);
					contQ:= d(CBaseQueueLib);
					contQ.taille:= taReste;
					base.EcrisPage(ptr+ ta+ taReste- 1);
					d:= base.NewPage(taCBaseTeteLib, cTLib, ptr+ ta);
					cont:= d(CBaseTeteLib);
					cont.taille:= taReste;
					base.InsLibBase(ptr+ ta, cont, base.racine, base.max);
				END;
			END;
			d:= base.NewPage(taCBaseTeteOcc, cTOcc, ptr);
			contTO:= d(CBaseTeteOcc);
			contTO.taille:= taille;
			base.EcrisPage(ptr);
			d:= base.NewPage(taCBaseQueueOcc, cQOcc, ptr+ ta- 1);
			contQO:= d(CBaseQueueOcc);
			base.EcrisPage(ptr+ ta- 1);
			INC(ptr, taCBaseTeteOcc);
			RETURN ptr;
		END ReserveBase;
	
	PROCEDURE (base: BaseDon) EffaceBase (ptr: LONGINT);
		
		VAR
			
			q, ta: LONGINT;
			contTL: CBaseTeteLib;
			contQL: CBaseQueueLib;
			contTO: CBaseTeteOcc;
			contT: CBaseTete;
			contQ: CBaseQueue;
			yAQueue: BOOLEAN;
			d: Data;
		
		BEGIN (*EffaceBase*)
			ASSERT(ptr>= baseTete+ taCBaseTeteOcc+ base.nbPlaces* TSI);
			DEC(ptr, taCBaseTeteOcc);
			d:= base.LisPageSys(ptr, taCBaseTeteOcc, cTOcc);
			contTO:= d(CBaseTeteOcc);
			ta:= ((contTO.taille+ taCBaseTeteOcc+ taCBaseQueueOcc- 1) DIV (taCBaseTeteLib+ taCBaseQueueLib)+ 1)*(taCBaseTeteLib+ taCBaseQueueLib);
			ASSERT(ptr+ ta<= base.fin);
			base.ErasePage(ptr);
			q:= ptr+ ta;
			base.ErasePage(q- 1);
			yAQueue:= FALSE;
			IF q= base.fin THEN
				base.fin:= ptr;
				base.ref.Tronque(base.fin);
			ELSIF base.TeteLibre(q, contT) THEN
				contTL:= contT(CBaseTeteLib);
				INC(ta, contTL.taille);
				base.SupLibBase(q, contTL, base.racine, base.max);
				base.ErasePage(q);
				yAQueue:= TRUE;
			END;
			q:= ptr- 1;
			IF base.QueueLibre(q, contQ) THEN
				contQL:= contQ(CBaseQueueLib);
				base.ErasePage(q);
				q:= ptr- contQL.taille;
				d:= base.LisPageSys(q, taCBaseTeteLib, cTLib);
				contTL:= d(CBaseTeteLib);
				base.SupLibBase(q, contTL, base.racine, base.max);
				IF ptr= base.fin THEN
					base.ErasePage(q);
					base.ErasePage(ptr+ ta- 1);
					base.fin:= q;
					base.ref.Tronque(base.fin);
				ELSE
					INC(contTL.taille, ta);
					IF yAQueue THEN
						d:= base.LisPageSys(q+ contTL.taille- 1, taCBaseQueueLib, cQLib);
					ELSE
						d:= base.NewPage(taCBaseQueueLib, cQLib, q+ contTL.taille- 1);
					END;
					contQL:= d(CBaseQueueLib);
					contQL.taille:= contTL.taille;
					base.EcrisPage(q+ contTL.taille- 1);
					base.InsLibBase(q, contTL, base.racine, base.max);
				END;
			ELSIF ptr# base.fin THEN
				d:= base.NewPage(taCBaseTeteLib, cTLib, ptr);
				contTL:= d(CBaseTeteLib);
				contTL.taille:= ta;
				IF yAQueue THEN
					d:= base.LisPageSys(ptr+ ta- 1, taCBaseQueueLib, cQLib);
				ELSE
					d:= base.NewPage(taCBaseQueueLib, cQLib, ptr+ ta- 1);
				END;
				contQL:= d(CBaseQueueLib);
				contQL.taille:= ta;
				base.EcrisPage(ptr+ ta- 1);
				base.InsLibBase(ptr, contTL, base.racine, base.max);
			END;
		END EffaceBase;
	
	(** Reads and returns the content of the fixed place place of the database base. The first place has number 0. *)
	PROCEDURE (base: BaseDon) LisPlace*(place: LONGINT): LONGINT;
		
		BEGIN (*LisPlace*)
			ASSERT(base.ref# NIL, 20);
			ASSERT((place>= 0) & (place< base.nbPlaces), 21);
			base.ref.PosLis((place+ 4)* TSI);
			RETURN base.ref.LisInt();
		END LisPlace;
	
	(** Writes val in the fixed place place of the database base. The first place has number 0. *)
	PROCEDURE (base: BaseDon) EcrisPlace*(place: LONGINT; val: LONGINT);
		
		BEGIN (*EcrisPlace*)
			ASSERT(base.ref# NIL, 20);
			ASSERT((place>= 0) & (place< base.nbPlaces), 21);
			base.ref.PosEcris((place+ 4)*TSI);
			base.ref.EcrisInt(val);
		END EcrisPlace;
	
	PROCEDURE (base: BaseDon) NouvPage (taPage: LONGINT; fab: DataFab; VAR pos: LONGINT): Data;
		
		BEGIN (*NouvPage*)
			pos:= base.ReserveBase(taPage);
			RETURN base.NewPage(taPage, fab, pos);
		END NouvPage;
	
	PROCEDURE (base: BaseDon) EffacePage (pos: LONGINT);
		
		BEGIN (*EffacePage*)
			base.ErasePage(pos);
			base.EffaceBase(pos);
		END EffacePage;
	
	PROCEDURE (base: BaseDon) SelectPage (pos: LONGINT): Page;
		
		VAR
			
			p: Page;
		
		BEGIN (*SelectPage*)
			IF base.TrouvePage(pos, p) THEN
				base.DeplacePage(p);
			ELSE
				p:= base.CreePage(base.LisBaseLong(pos), pos);
			END;
			RETURN p;
		END SelectPage;
	
	PROCEDURE (base: BaseDon) LisPageLong (pos: LONGINT): LONGINT;
		
		VAR
			
			p: Page;
		
		BEGIN (*LisPageLong*)
			p:= base.SelectPage(pos);
			RETURN p.tailP;
		END LisPageLong;
	
	PROCEDURE (base: BaseDon) LisPage (pos: LONGINT; fab: DataFab): Data;
		
		VAR
			
			p: Page;
			a: PSHORTINT;
			r: Reader;
		
		BEGIN (*LisPage*)
			p:= base.SelectPage(pos);
			IF p.pageP= NIL THEN
				NEW(a, p.tailP);
				p.pageP:= fab.New(p.tailP);
				base.LisBase(pos, a^);
				r.InitReader(base.ref, a, 0);
				p.pageP.Read(r);
			END;
			RETURN p.pageP;
		END LisPage;
	
	(** Reads data d in database base at position ptr. *)
	PROCEDURE (d: GesData) LisDon*(ptr: LONGINT): Data;
	
		VAR
			
			r: Reader;
			w: Writer;
			a: PSHORTINT;
			pa: Data;
		
		BEGIN (*LisDon*)
			ASSERT(d.base.ref# NIL, 20);
			w.InitWriter(d.base.ref);
			pa:= d.base.LisPage(ptr, d.fab);
			pa.Write(w);
			a:= w.Write();
			r.InitReader(d.base.ref, a, 0);
			pa:= d.fab.New(LEN(a));
			pa.Read(r);
			RETURN pa;
		END LisDon;
	
	(** Allocates a cluster of size taille (bytes), for the data managed by d, in its database, and returns its position. taille must be the size of the production of the method Data.Write of the data managed by d. *)
	PROCEDURE (d: GesData) ReserveDon*(taille: LONGINT): LONGINT;
		
		VAR
			
			pos: LONGINT;
			pa: Data;
		
		BEGIN (*ReserveDon*)
			ASSERT(d.base.ref# NIL, 20);
			pa:= d.base.NouvPage(taille, d.fab, pos);
			RETURN pos;
		END ReserveDon;
	
	(** Writes data, managed by d, at the position ptr in the database of d. *)
	PROCEDURE (d: GesData) EcrisDon*(ptr: LONGINT; data: Data);
		
		VAR
			
			r: Reader;
			w: Writer;
		
		BEGIN (*EcrisDon*)
			ASSERT(d.base.ref# NIL, 20);
			w.InitWriter(d.base.ref);
			data.Write(w);
			r.InitReader(d.base.ref, w.Write(), 0);
			data:= d.base.LisPage(ptr, d.fab);
			data.Read(r);
			d.base.EcrisPage(ptr);
		END EcrisDon;
	
	(** Erases data managed by d at position ptr in the database of d. *)
	PROCEDURE (d: GesData) EffaceDon*(ptr: LONGINT);
		
		BEGIN (*EffaceDon*)
			ASSERT(d.base.ref# NIL, 20);
			d.base.EffacePage(ptr);
		END EffaceDon;
	
	PROCEDURE (pa: PageI) Write (VAR w: Writer);
		
		VAR
			
			i: LONGINT;
		
		BEGIN (*Write*)
			w.OutInt(pa.cles);
			w.OutInt(pa.nbEl);
			FOR i:= 0 TO 2* (minEl+ 1)- 1 DO
				w.OutInt(pa.elems[i].ptr);
				w.OutInt(pa.elems[i].finC);
			END;
		END Write;
	
	PROCEDURE (pa: PageI) Read (VAR r: Reader);
		
		VAR
			
			i: LONGINT;
		
		BEGIN (*Read*)
			pa.cles:= r.InInt();
			pa.nbEl:= r.InInt();
			FOR i:= 0 TO 2* (minEl+ 1)- 1 DO
				pa.elems[i].ptr:= r.InInt();
				pa.elems[i].finC:= r.InInt();
			END;
		END Read;
	
	PROCEDURE (f: PageIFab)  New (ta: LONGINT): Data;
		
		VAR
			
			pa: PageI;
		
		BEGIN (*New*)
			ASSERT(ta= taPageRec);
			NEW(pa);
			RETURN pa;
		END New;
	
	PROCEDURE (c: ClesCont) Write (VAR w: Writer);
		
		BEGIN (*Write*)
			w.OutBytes(c.c^);
		END Write;
	
	PROCEDURE (c: ClesCont) Read (VAR r: Reader);
		
		BEGIN (*Read*)
			NEW(c.c, r.len);
			r.InBytes(c.c^);
		END Read;
	
	PROCEDURE (f: ClesFab)  New (ta: LONGINT): Data;
		
		VAR
			
			pa: ClesCont;
		
		BEGIN (*New*)
			ASSERT(ta> 0);
			NEW(pa);
			NEW(pa.c, ta);
			RETURN pa;
		END New;
	
	PROCEDURE (c: Chaine) Write (VAR w: Writer);
		
		BEGIN (*Write*)
			w.OutInt(c.suivant);
			w.OutInt(c.precedent);
			w.OutInt(c.posDon);
			IF c.clef= cNIL THEN
				w.OutInt(0);
			ELSE
				w.OutInt(LEN(c.clef));
				w.OutBytes(c.clef^);
			END;
		END Write;
	
	PROCEDURE (c: Chaine) Read (VAR r: Reader);
		
		VAR
			
			n: LONGINT;
		
		BEGIN (*Read*)
			c.suivant:= r.InInt();
			c.precedent:= r.InInt();
			c.posDon:= r.InInt();
			n:= r.InInt();
			IF n= 0 THEN
				c.clef:= cNIL;
			ELSE
				NEW(c.clef, n);
				r.InBytes(c.clef^);
			END;
		END Read;
	
	PROCEDURE (f: ChaineFab) New (ta: LONGINT): Data;
		
		VAR
			
			pa: Chaine;
		
		BEGIN (*New*)
			NEW(pa);
			ta:= ta- taChaine;
			IF ta= 0 THEN
				pa.clef:= cNIL;
			ELSE
				NEW(pa.clef, ta);
			END;
			RETURN pa;
		END New;
	
	PROCEDURE (i: Info) Write (VAR w: Writer);
		
		BEGIN (*Write*)
			w.OutInt(garde1);
			w.OutInt(i.iRac);
			w.OutInt(i.iCha);
			w.OutInt(i.iTaC);
			w.OutInt(i.iH);
			w.OutInt(garde2);
		END Write;
	
	PROCEDURE (i: Info) Read (VAR r: Reader);
		
		BEGIN (*Read*)
			ASSERT(r.InInt()= garde1);
			i.iRac:= r.InInt();
			i.iCha:= r.InInt();
			i.iTaC:= r.InInt();
			i.iH:= r.InInt();
			ASSERT(r.InInt()= garde2);
		END Read;
	
	PROCEDURE (f: InfoFab)  New (ta: LONGINT): Data;
		
		VAR
			
			pa: Info;
		
		BEGIN (*New*)
			ASSERT(ta= taInfo);
			NEW(pa);
			RETURN pa;
		END New;
	
	(** Creates a new index in the database base and returns its reference, but does not open it. tailleCles is the size of keys. If keys have a fixed size, put this size in tailleCles. If the size of keys does not vary much, put the greatest size in tailleCles. If the greatest size of keys is unknown, or if this size vary much, or if you want to use prefixes, put zero in tailleCles: you'll have to fix the size of each key later. *)
	PROCEDURE (base: BaseDon) IndCree*(tailleCles: LONGINT): LONGINT;
		
		VAR
			
			pos: LONGINT;
			p: Info;
			c: Chaine;
			d: Data;
		
		BEGIN (*IndCree*)
			ASSERT(base.ref# NIL, 20);
			ASSERT(tailleCles>= 0, 21);
			d:= base.NouvPage(taInfo, inff, pos);
			p:= d(Info);
			p.iTaC:= tailleCles;
			p.iH:= 0;
			d:= base.NouvPage(taChaine, chf, p.iCha);
			c:= d(Chaine);
			p.iRac:= p.iCha;
			c.suivant:= p.iCha;
			c.precedent:= p.iCha;
			c.posDon:= bNil;
			base.EcrisPage(p.iCha);
			base.EcrisPage(pos);
			RETURN pos;
		END IndCree;
	
	(** Opens and returns an index created with the reference ref by BaseDon.IndCree in the database base. g is the key manager of the index and f is the factory of its keys. The current position of the index is reset. *)
	PROCEDURE (base: BaseDon) IndOuvre*(ref: LONGINT; g: GesCles; f: DataFab): Index;
		
		VAR
			
			ind: Index;
			p: Info;
			d: Data;
		
		BEGIN (*IndOuvre*)
			ASSERT(base.ref# NIL, 20);
			ASSERT(g# NIL, 21);
			ASSERT(f# NIL, 22);
			d:= base.LisPage(ref, inff);
			p:= d(Info);
			NEW(ind);
			ind.racI:= p.iRac;
			ind.chaine:= p.iCha;
			ind.taCle:= p.iTaC;
			ind.haut:= p.iH;
			ind.suivI:= base.indAnneau.suivI;
			ind.precI:= base.indAnneau;
			base.indAnneau.suivI.precI:= ind;
			base.indAnneau.suivI:= ind;
			ind.baseI:= base;
			ind.refI:= ref;
			g.f:= f;
			ind.gestion:= g;
			ind.pCour:= ind.chaine;
			IF ind.taCle=0 THEN
				ind.taCles:= taSect;
			ELSE
				ind.taCles:= (2* minEl+ 1)* ind.taCle;
			END;
			RETURN ind;
		END IndOuvre;
	
	(** Deletes the index of position ref in the database base; this index must have been closed. *)
	PROCEDURE (base: BaseDon) IndDetruit*(ref: LONGINT);
		
		VAR
			
			p: Info;
			ind: Index;
			a: ARRAY taInfo OF SHORTINT;
			i: LONGINT;
			d: Data;
		
		PROCEDURE FDet (pos, h: LONGINT);
			
			VAR
				
				i: LONGINT;
				pa: PageI;
				d: Data;
			
			BEGIN (*FDet*)
				IF h>0 THEN
					d:=base.LisPage(pos, pif);
					pa:=d(PageI);
					base.EffacePage(pa.cles);
					FOR i:= 0 TO pa.nbEl DO
						FDet(pa.elems[i].ptr, h- 1);
					END;
				END;
				base.EffacePage(pos);
			END FDet;
		
		BEGIN (*IndDetruit*)
			ASSERT(base.ref# NIL, 20);
			ind:= base.indAnneau.suivI;
			WHILE (ind# base.indAnneau) & (ind.refI# ref) DO
				ind:= ind.suivI;
			END;
			ASSERT(ind= base.indAnneau, 21);
			d:= base.LisPage(ref, inff);
			p:= d(Info);
			FDet(p.iRac, p.iH);
			FOR i:= 0 TO taInfo- 1 DO
				a[i]:= 0;
			END;
			base.EcrisBase(ref, a);
			base.EffacePage(ref);
		END IndDetruit;
	
	PROCEDURE (g: GesCles) Prefixe (ref: Fichier; c1, c2: Data; VAR p: Cles; VAR l: LONGINT);
		
		VAR
			
			w: Writer;
		
		BEGIN (*Prefixe*)
			g.PrefP(c1, c2);
			w.InitWriter(ref); c2.Write(w);
			p:= w.Write();
			l:= LEN(p);
		END Prefixe;
	
	PROCEDURE (ind: Index) MAJRac;
		
		VAR
			
			p: Info;
			d: Data;
		
		BEGIN (*MAJRac*)
			d:= ind.baseI.LisPage(ind.refI, inff);
			p:= d(Info);
			p.iRac:= ind.racI;
			p.iH:= ind.haut;
			ind.baseI.EcrisPage(ind.refI);
		END MAJRac;
	
	PROCEDURE (ind: Index) AjusteTailleCles (p: PageI; taEnPlus: LONGINT);

		VAR
			
			l, l1, l2, i: LONGINT;
			nCles: LONGINT;
			cl: ClesCont;
			nCl: Cles;
			d: Data;
		
		BEGIN (*AjusteTailleCles*)
			l1:= ind.baseI.LisPageLong(p.cles);
			l:= p.elems[p.nbEl].finC;
			l2:= l+ taEnPlus;
			IF l2> l1 THEN
				REPEAT
					INC(l1, taSect);
				UNTIL l1>= l2;
				d:= ind.baseI.NouvPage(l1, clf, nCles);
				nCl:= d(ClesCont).c;
				d:= ind.baseI.LisPage(p.cles, clf);
				cl:= d(ClesCont);
				FOR i:= 0 TO l- 1 DO
					nCl[i]:= cl.c[i];
				END;
				ind.baseI.EffacePage(p.cles);
				p.cles:= nCles;
				ind.baseI.EcrisPage(p.cles);
			END;
		END AjusteTailleCles;
	
	PROCEDURE (ind: Index) TransInter (pa1, pa2: PageI; src, dst, nb: LONGINT);
		
		VAR
			
			cl1, cl2: ClesCont;
			lC, i, diff, u, v: LONGINT;
			d: Data;
		
		BEGIN (*TransInter*)
			lC:= pa1.elems[src+ nb- 1].finC- pa1.elems[src- 1].finC;
			ind.AjusteTailleCles(pa2, lC);
			d:= ind.baseI.LisPage(pa1.cles, clf);
			cl1:= d(ClesCont);
			d:= ind.baseI.LisPage(pa2.cles, clf);
			cl2:= d(ClesCont);
			FOR i:= pa2.elems[pa2.nbEl].finC- 1 TO pa2.elems[dst- 1].finC BY -1 DO
				cl2.c[i+ lC]:= cl2.c[i];
			END;
			u:= pa2.elems[dst- 1].finC; v:= pa1.elems[src- 1].finC;
			FOR i:= 0 TO lC- 1 DO
				cl2.c[i+ u]:= cl1.c[i+ v];
			END;
			diff:= pa1.elems[src+ nb- 1].finC- pa1.elems[src- 1].finC;
			FOR i:= pa1.elems[src+ nb- 1].finC TO pa1.elems[pa1.nbEl].finC- 1 DO
				cl1.c[i- diff]:= cl1.c[i];
			END;
			FOR i:= dst TO pa2.nbEl DO
				INC(pa2.elems[i].finC, lC);
			END;
			IF dst<= pa2.nbEl THEN
				FOR i:= pa2.nbEl TO dst BY -1 DO
					pa2.elems[i+ nb]:= pa2.elems[i];
				END;
			END;
			diff:= pa2.elems[dst- 1].finC- pa1.elems[src- 1].finC;
			FOR i:= src TO src+ nb- 1 DO
				INC(pa1.elems[i].finC, diff);
			END;
			FOR i:= 0 TO nb- 1 DO
				pa2.elems[dst+ i]:= pa1.elems[src+ i];
			END;
			FOR i:= src+ nb TO pa1.nbEl DO
				DEC(pa1.elems[i].finC, lC);
			END;
			IF src+ nb<= pa1.nbEl THEN
				FOR i:= src+ nb TO pa1.nbEl DO
					pa1.elems[i- nb]:= pa1.elems[i];
				END;
			END;
			INC(pa2.nbEl, nb);
			DEC(pa1.nbEl, nb);
		END TransInter;
	
	PROCEDURE (ind: Index) TransIn (pa: PageI; el: LONGINT; VAR c: Cles; lC, dst: LONGINT);
		
		VAR
			
			i: LONGINT;
			cl: ClesCont;
			d: Data;
		
		BEGIN (*TransIn*)
			ind.AjusteTailleCles(pa, lC);
			d:= ind.baseI.LisPage(pa.cles, clf);
			cl:= d(ClesCont);
			FOR i:= pa.elems[pa.nbEl].finC- 1 TO pa.elems[dst- 1].finC BY -1 DO
				cl.c[i+ lC]:= cl.c[i];
			END;
			FOR i:= 0 TO lC- 1 DO
				cl.c[pa.elems[dst- 1].finC+ i]:= c[i];
			END;
			FOR i:= dst TO pa.nbEl DO
				INC(pa.elems[i].finC, lC);
			END;
			IF dst<=pa.nbEl THEN
				FOR i:= pa.nbEl TO dst BY -1 DO
					pa.elems[i+ 1]:= pa.elems[i];
				END;
			END;
			pa.elems[dst].ptr:= el;
			pa.elems[dst].finC:= pa.elems[dst- 1].finC+ lC;
			INC(pa.nbEl);
		END TransIn;
	
	PROCEDURE (ind: Index) TransOut (pa: PageI; src: LONGINT; VAR el: LONGINT; VAR c: Cles; VAR lC: LONGINT);
		
		VAR
			
			cl: ClesCont;
			i: LONGINT;
			d: Data;
		
		BEGIN (*TransOut*)
			el:= pa.elems[src].ptr;
			d:= ind.baseI.LisPage(pa.cles, clf);
			cl:= d(ClesCont);
			lC:= pa.elems[src].finC- pa.elems[src- 1].finC;
			NEW(c, lC);
			FOR i:= 0 TO lC- 1 DO
				c[i]:= cl.c[pa.elems[src- 1].finC+ i];
			END;
			FOR i:= pa.elems[src].finC TO pa.elems[pa.nbEl].finC- 1 DO
				cl.c[i- lC]:= cl.c[i];
			END;
			FOR i:= src+ 1 TO pa.nbEl DO
				DEC(pa.elems[i].finC, lC);
			END;
			IF src< pa.nbEl THEN
				FOR i:= src TO pa.nbEl- 1 DO
					pa.elems[i]:= pa.elems[i+ 1]
				END;
			END;
			DEC(pa.nbEl);
		END TransOut;
	
	(** Seeks in the index ind the key cle of length tailleCle and inserts it if it is not there already. The result indicates if the key was found. Fixes the current position of the index on the key found or inserted. *)
	PROCEDURE (ind: Index) RechIns*(cle: Data): BOOLEAN;
		
		VAR
			
			inc, trouve: BOOLEAN;
			taC, el, p, lC: LONGINT;
			page: PageI;
			cl: ClesCont;
			c, cleA: Cles;
			w: Writer;
			d: Data;
		
		PROCEDURE Cherche (p, pere: LONGINT; numPere, h: LONGINT; VAR el: LONGINT; VAR lC: LONGINT; VAR c: Cles);
			
			VAR
				
				ch, nCh: Chaine;
				comp: Comp;
				pa, pa1, pa2, paP: PageI;
				cRet: Cles;
				cl: ClesCont;
				g, d, i, lCRet, n, m, elRet, nEl: LONGINT;
				data: Data;
				r: Reader;
				dd: Data;
			
			BEGIN (*Cherche*)
				IF h=0 THEN
					dd:= ind.baseI.LisPage(p, chf);
					ch:= dd(Chaine);
					r.InitReader(ind.baseI.ref, ch.clef, 0);
					IF p= ind.chaine THEN
						data:= ind.gestion.f.New(0);
						comp:= sup;
					ELSE
						n:= ind.baseI.LisPageLong(p)- taChaine;
						data:= ind.gestion.f.New(n);
						data.Read(r);
						comp:= ind.gestion.CompP(cle, data);
					END;
					trouve:= comp=ega;
					inc:= ~trouve;
					IF inc THEN
						dd:= ind.baseI.NouvPage(taChaine+ taC, chf, el);
						nCh:= dd(Chaine);
						ind.pCour:= el;
						ind.baseI.EcrisPage(el);
						nCh.posDon:= bNil;
						ASSERT(LEN(nCh.clef)= LEN(cleA));
						nCh.clef:= cleA;
						IF comp= inf THEN
							nCh.suivant:= p;
							nCh.precedent:= ch.precedent;
							dd:= ind.baseI.LisPage(ch.precedent, chf);
							nCh:= dd(Chaine);
							nCh.suivant:= el;
							ind.baseI.EcrisPage(ch.precedent);
							ch.precedent:= el;
							IF pere# bNil THEN
								dd:= ind.baseI.LisPage(pere, pif);
								pa:= dd(PageI);
								pa.elems[numPere].ptr:= el;
								ind.baseI.EcrisPage(pere);
							END;
							el:= p;
							ind.gestion.Prefixe(ind.baseI.ref, cle, data, c, lC);
						ELSE
							nCh.suivant:= ch.suivant;
							nCh.precedent:= p;
							dd:= ind.baseI.LisPage(ch.suivant, chf);
							nCh:= dd(Chaine);
							nCh.precedent:= el;
							ind.baseI.EcrisPage(ch.suivant);
							ch.suivant:= el;
							ind.gestion.Prefixe(ind.baseI.ref, data, cle, c, lC);
						END;
						ind.baseI.EcrisPage(p);
					ELSE
						ind.pCour:= p;
					END;
				ELSE
					dd:= ind.baseI.LisPage(p, pif);
					pa:= dd(PageI);
					dd:= ind.baseI.LisPage(pa.cles, clf);
					cl:= dd(ClesCont);
					g:= 1; d:= pa.nbEl+ 1;
					WHILE g<d DO
						i:= (g+ d) DIV 2;
						data:= ind.gestion.f.New(pa.elems[i].finC- pa.elems[i- 1].finC);
						r.InitReader(ind.baseI.ref, cl.c, pa.elems[i- 1].finC);
						data.Read(r);
						IF ind.gestion.CompP(cle, data)= inf THEN
							d:= i;
						ELSE
							g:= i+ 1;
						END;
					END;
					DEC(d);
					Cherche(pa.elems[d].ptr, p, d, h- 1, elRet, lCRet, cRet);
					IF inc THEN
						dd:= ind.baseI.LisPage(p, pif);
						pa:= dd(PageI);
						ind.TransIn(pa, elRet, cRet, lCRet, d+ 1);
						IF pa.nbEl<= 2* minEl THEN
							inc:= FALSE;
						ELSE
							n:= 1;
							m:= 2*minEl+ 1;
							IF pere# bNil THEN
								dd:= ind.baseI.LisPage(pere, pif);
								paP:= dd(PageI);
								IF numPere< paP.nbEl THEN
									INC(n);
									dd:= ind.baseI.LisPage(paP.elems[numPere+ 1].ptr, pif);
									pa2:= dd(PageI);
									INC(m, pa2.nbEl);
								END;
								IF numPere> 0 THEN
									INC(n);
									dd:= ind.baseI.LisPage(paP.elems[numPere- 1].ptr, pif);
									pa1:= dd(PageI);
									INC(m, pa1.nbEl);
								END;
							END;
							IF m<= 2* n* minEl THEN
								inc:= FALSE;
								m:= (m+ 1) DIV n;
								IF (numPere> 0) & (m> pa1.nbEl) THEN
									ind.TransOut(paP, numPere, el, c, lC);
									ind.TransIn(pa1, pa.elems[0].ptr, c, lC, pa1.nbEl+ 1);
									IF m> pa1.nbEl THEN
										ind.TransInter(pa, pa1, 1, pa1.nbEl+ 1, m- pa1.nbEl);
									END;
									ind.TransOut(pa, 1, el, c, lC);
									ind.TransIn(paP, p, c, lC, numPere);
									pa.elems[0].ptr:= el;
									ind.baseI.EcrisPage(paP.elems[numPere- 1].ptr);
									ind.baseI.EcrisPage(pa1.cles);
								END;
								IF (numPere< paP.nbEl) & (m> pa2.nbEl) THEN
									ind.TransOut(paP, numPere+ 1, el, c, lC);
									ind.TransIn(pa2, pa2.elems[0].ptr, c, lC, 1);
									IF m> pa2.nbEl THEN
										ind.TransInter(pa, pa2, pa.nbEl+ pa2.nbEl- m+ 1, 1, m- pa2.nbEl);
									END;
									ind.TransOut(pa, pa.nbEl, nEl, c, lC);
									ind.TransIn(paP, el, c, lC, numPere+ 1);
									pa2.elems[0].ptr:= nEl;
									ind.baseI.EcrisPage(el);
									ind.baseI.EcrisPage(pa2.cles);
								END;
								IF (numPere> 0) & (m< pa1.nbEl) THEN
									ind.TransOut(paP, numPere, el, c, lC);
									ind.TransIn(pa, pa.elems[0].ptr, c, lC, 1);
									IF m+ 1< pa1.nbEl THEN
										ind.TransInter(pa1, pa, m+ 2, 1, pa1.nbEl- m- 1);
									END;
									ind.TransOut(pa1, m+ 1, el, c, lC);
									ind.TransIn(paP, p, c, lC, numPere);
									pa.elems[0].ptr:= el;
									ind.baseI.EcrisPage(paP.elems[numPere- 1].ptr);
									ind.baseI.EcrisPage(pa1.cles);
								END;
								IF (numPere< paP.nbEl) & (m< pa2.nbEl) THEN
									ind.TransOut(paP, numPere+ 1, el, c, lC);
									ind.TransIn(pa, pa2.elems[0].ptr, c, lC, pa.nbEl+ 1);
									IF m+ 1<pa2.nbEl THEN
										ind.TransInter(pa2, pa, 1, pa.nbEl+ 1, pa2.nbEl- m- 1);
									END;
									ind.TransOut(pa2, 1, nEl, c, lC);
									ind.TransIn(paP, el, c, lC, numPere+ 1);
									pa2.elems[0].ptr:= nEl;
									ind.baseI.EcrisPage(el);
									ind.baseI.EcrisPage(pa2.cles);
								END;
								ind.baseI.EcrisPage(pere);
								ind.baseI.EcrisPage(paP.cles);
							ELSE
								ind.TransOut(pa, minEl+ 1, nEl, c, lC);
								dd:= ind.baseI.NouvPage(taPageRec, pif, el);
								pa1:= dd(PageI);
								dd:= ind.baseI.NouvPage(ind.taCles, clf, pa1.cles);
								cl:= dd(ClesCont);
								pa1.nbEl:= 0;
								pa1.elems[0].ptr:= nEl;
								pa1.elems[0].finC:= 0;
								ind.TransInter(pa, pa1, minEl+ 1, 1, minEl);
								ind.baseI.EcrisPage(el);
								ind.baseI.EcrisPage(pa1.cles);
							END;
						END;
						ind.baseI.EcrisPage(p);
						ind.baseI.EcrisPage(pa.cles);
					END;
				END;
			END Cherche;
		
		BEGIN (*RechIns*)
			ASSERT(ind.baseI# NIL, 20);
			w.InitWriter(ind.baseI.ref); cle.Write(w);
			cleA:= w.Write();
			taC:= LEN(cleA);
			Cherche(ind.racI, bNil, 0, ind.haut, el, lC, c);
			IF inc THEN
				INC(ind.haut);
				p:= ind.racI;
				d:= ind.baseI.NouvPage(taPageRec, pif, ind.racI);
				page:= d(PageI);
				d:= ind.baseI.NouvPage(ind.taCles, clf, page.cles);
				cl:= d(ClesCont);
				page.nbEl:= 0;
				page.elems[0].ptr:= p;
				page.elems[0].finC:= 0;
				ind.TransIn(page, el, c, lC, 1);
				ind.baseI.EcrisPage(ind.racI);
				ind.baseI.EcrisPage(page.cles);
				ind.MAJRac;
			END;
			RETURN trouve;
		END RechIns;
	
	(** Seek in the index ind the key cle. The result indicates if the key was found. Fixes the current position of the index on the found key or that which is immediately above than the key sought in the event of unfruitful search. *)
	PROCEDURE (ind: Index) Cherche*(cle: Data): BOOLEAN;
		
		VAR
			
			h, g, d, i, p: LONGINT;
			pa: PageI;
			cl: ClesCont;
			ch: Chaine;
			comp: Comp;
			data: Data;
			r: Reader;
			dd: Data;
		
		BEGIN (*Cherche*)
			ASSERT(ind.baseI# NIL, 20);
			p:= ind.racI;
			FOR h:= 1 TO ind.haut DO
				dd:= ind.baseI.LisPage(p, pif);
				pa:= dd(PageI);
				dd:= ind.baseI.LisPage(pa.cles, clf);
				cl:= dd(ClesCont);
				g:= 1; d:= pa.nbEl+ 1;
				WHILE g<d DO
					i:= (g+ d) DIV 2;
					data:= ind.gestion.f.New(pa.elems[i].finC- pa.elems[i- 1].finC);
					r.InitReader(ind.baseI.ref, cl.c, pa.elems[i- 1].finC);
					data.Read(r);
					IF ind.gestion.CompP(cle, data)= inf THEN
						d:= i;
					ELSE
						g:= i+ 1;
					END;
				END;
				DEC(d);
				p:= pa.elems[d].ptr;
			END;
			dd:= ind.baseI.LisPage(p, chf);
			ch:= dd(Chaine);
			IF p= ind.chaine THEN
				comp:= sup;
			ELSE
				data:= ind.gestion.f.New(ind.baseI.LisPageLong(p)- taChaine);
				r.InitReader(ind.baseI.ref, ch.clef, 0);
				data.Read(r);
				comp:= ind.gestion.CompP(cle, data);
			END;
			IF comp= sup THEN
				ind.pCour:= ch.suivant;
			ELSE
				ind.pCour:= p;
			END;
			RETURN comp= ega;
		END Cherche;
	
	(** Erases from index ind the key cle. If the key cle does not belong to the index, ind.Efface does nothing. Modifies the current position of the index only if the key was found; in this case, the current position is reset. *)
	PROCEDURE (ind: Index) Efface*(cle: Data);
		
		VAR
			
			dec, trouve: BOOLEAN;
			pSuiv, numSuiv: LONGINT;
			
			pa: PageI;
			d: Data;
		
		PROCEDURE Eff (p, pere, numPere, h: LONGINT);
			
			VAR
				
				d: LONGINT;
				
				ch, chP, chS: Chaine;
				pa: PageI;
				cl: ClesCont;
				c, cE: Cles;
				lC, lCE, i, g, el: LONGINT;
				data1, data2: Data;
				r: Reader;
				dd: Data;
			
			PROCEDURE Corrige (p, num: LONGINT);
				
				VAR
					
					n, m, lC, q, el, nEl: LONGINT;
					paP, pa, pa1, pa2: PageI;
					c: Cles;
					d: Data;
				
				BEGIN (*Corrige*)
					n:= 1;
					m:= minEl- 1;
					d:= ind.baseI.LisPage(p, pif);
					paP:= d(PageI);
					IF num< paP.nbEl THEN
						INC(n);
						d:= ind.baseI.LisPage(paP.elems[num+ 1].ptr, pif);
						pa2:= d(PageI);
						INC(m, pa2.nbEl);
					END;
					IF num> 0 THEN
						INC(n);
						d:= ind.baseI.LisPage(paP.elems[num- 1].ptr, pif);
						pa1:= d(PageI);
						INC(m, pa1.nbEl);
					END;
					q:= paP.elems[num].ptr;
					d:= ind.baseI.LisPage(q, pif);
					pa:= d(PageI);
					IF m>= 2* (n- 1)* minEl THEN
						dec:= FALSE;
						m:= (m+ 1) DIV n;
						IF (num> 0) & (m< pa1.nbEl) THEN
							ind.TransOut(paP, num, el, c, lC);
							ind.TransIn(pa, pa.elems[0].ptr, c, lC, 1);
							IF m+ 1< pa1.nbEl THEN
								ind.TransInter(pa1, pa, m+ 2, 1, pa1.nbEl- m- 1);
							END;
							ind.TransOut(pa1, m+ 1, el, c, lC);
							ind.TransIn(paP, q, c, lC, num);
							pa.elems[0].ptr:= el;
							ind.baseI.EcrisPage(paP.elems[num- 1].ptr);
							ind.baseI.EcrisPage(pa1.cles);
						END;
						IF (num< paP.nbEl) & (m< pa2.nbEl) THEN
							ind.TransOut(paP, num+ 1, el, c, lC);
							ind.TransIn(pa, pa2.elems[0].ptr, c, lC, pa.nbEl+ 1);
							IF m+ 1< pa2.nbEl THEN
								ind.TransInter(pa2, pa, 1, pa.nbEl+ 1, pa2.nbEl- m- 1);
							END;
							ind.TransOut(pa2, 1, nEl, c, lC);
							ind.TransIn(paP, el, c, lC, num+ 1);
							pa2.elems[0].ptr:= nEl;
							ind.baseI.EcrisPage(el);
							ind.baseI.EcrisPage(pa2.cles);
						END;
						IF (num> 0) & (m> pa1.nbEl) THEN
							ind.TransOut(paP, num, el, c, lC);
							ind.TransIn(pa1, pa.elems[0].ptr, c, lC, pa1.nbEl+ 1);
							IF m> pa1.nbEl THEN
								ind.TransInter(pa, pa1, 1, pa1.nbEl+ 1, m- pa1.nbEl);
							END;
							ind.TransOut(pa, 1, el, c, lC);
							ind.TransIn(paP, q, c, lC, num);
							pa.elems[0].ptr:= el;
							ind.baseI.EcrisPage(paP.elems[num- 1].ptr);
							ind.baseI.EcrisPage(pa1.cles);
						END;
						IF (num< paP.nbEl) & (m> pa2.nbEl) THEN
							ind.TransOut(paP, num+ 1, el, c, lC);
							ind.TransIn(pa2, pa2.elems[0].ptr, c, lC, 1);
							IF m> pa2.nbEl THEN
								ind.TransInter(pa, pa2, pa.nbEl+ pa2.nbEl- m+ 1, 1, m- pa2.nbEl);
							END;
							ind.TransOut(pa, pa.nbEl, nEl, c, lC);
							ind.TransIn(paP, el, c, lC, num+ 1);
							pa2.elems[0].ptr:= nEl;
							ind.baseI.EcrisPage(el);
							ind.baseI.EcrisPage(pa2.cles);
						END;
						ind.baseI.EcrisPage(q);
						ind.baseI.EcrisPage(pa.cles);
					ELSE
						lC:= m+ 1;
						m:= lC DIV (n- 1);
						IF num= 0 THEN
							n:= m;
						ELSE
							n:= lC- m;
						END;
						IF (num< paP.nbEl) & (n> pa2.nbEl) THEN
							ind.TransOut(paP, num+ 1, el, c, lC);
							ind.TransIn(pa2, pa2.elems[0].ptr, c, lC, 1);
							IF n> pa2.nbEl THEN
								ind.TransInter(pa, pa2, pa.nbEl+ pa2.nbEl- n+ 1, 1, n- pa2.nbEl);
							END;
							IF pa.nbEl= 0 THEN
								pa2.elems[0].ptr:= pa.elems[0].ptr;
								paP.elems[num].ptr:= el;
							ELSE
								ind.TransOut(pa, pa.nbEl, nEl, c, lC);
								ind.TransIn(paP, el, c, lC, num+ 1);
								pa2.elems[0].ptr:= nEl;
							END;
							ind.baseI.EcrisPage(el);
							ind.baseI.EcrisPage(pa2.cles);
						END;
						IF (num> 0) & (m> pa1.nbEl) THEN
							ind.TransOut(paP, num, el, c, lC);
							ind.TransIn(pa1, pa.elems[0].ptr, c, lC, pa1.nbEl+ 1);
							IF m> pa1.nbEl THEN
								ind.TransInter(pa, pa1, 1, pa1.nbEl+ 1, m- pa1.nbEl);
							END;
							ind.baseI.EcrisPage(paP.elems[num- 1].ptr);
							ind.baseI.EcrisPage(pa1.cles);
						END;
						ind.baseI.EffacePage(pa.cles);
						ind.baseI.EffacePage(q);
						dec:= paP.nbEl<minEl;
					END;
					ind.baseI.EcrisPage(p);
					ind.baseI.EcrisPage(paP.cles);
				END Corrige;
			
			PROCEDURE Del (q, h: LONGINT);
				
				VAR
					
					pa: PageI;
					el, lC, lCE: LONGINT;
					c, cE: Cles;
					dd: Data;
				
				BEGIN (*Del*)
					dd:= ind.baseI.LisPage(q, pif);
					pa:= dd(PageI);
					IF h=1 THEN
						ind.TransOut(pa, pa.nbEl, el, c, lC);
						ind.baseI.EcrisPage(q);
						ind.baseI.EcrisPage(pa.cles);
						dec:= pa.nbEl< minEl;
						dd:= ind.baseI.LisPage(p, pif);
						pa:= dd(PageI);
						ind.TransOut(pa, d, el, cE, lCE);
						ind.TransIn(pa, el, c, lC, d);
						ind.baseI.EcrisPage(p);
						ind.baseI.EcrisPage(pa.cles);
					ELSE
						Del(pa.elems[pa.nbEl].ptr, h- 1);
						IF dec THEN
							Corrige(q, pa.nbEl);
						END;
					END;
				END Del;
			
			BEGIN (*Eff*)
				IF h= 0 THEN
					dec:= FALSE;
					IF p= ind.chaine THEN
						trouve:= FALSE;
					ELSE
						dd:= ind.baseI.LisPage(p, chf);
						ch:= dd(Chaine);
						data1:= ind.gestion.f.New(ind.baseI.LisPageLong(p)- taChaine);
						r.InitReader(ind.baseI.ref, ch.clef, 0);
						data1.Read(r);
						trouve:= ind.gestion.CompP(cle, data1)= ega;
					END;
					IF trouve THEN
						dd:= ind.baseI.LisPage(ch.suivant, chf);
						chS:= dd(Chaine);
						chS.precedent:= ch.precedent;
						ind.baseI.EcrisPage(ch.suivant);
						dd:= ind.baseI.LisPage(ch.precedent, chf);
						chP:= dd(Chaine);
						chP.suivant:= ch.suivant;
						ind.baseI.EcrisPage(ch.precedent);
						IF numPere=0 THEN
							dd:= ind.baseI.LisPage(pere, pif);
							pa:= dd(PageI);
							pa.elems[numPere].ptr:= ch.precedent;
							ind.baseI.EcrisPage(pere);
						END;
						IF pSuiv# bNil THEN
							data1:= ind.gestion.f.New(ind.baseI.LisPageLong(ch.precedent)- taChaine);
							IF ch.precedent# ind.chaine THEN
								r.InitReader(ind.baseI.ref, chP.clef, 0);
								data1.Read(r);
							END;
							data2:= ind.gestion.f.New(ind.baseI.LisPageLong(ch.suivant)- taChaine);
							IF ch.suivant# ind.chaine THEN
								r.InitReader(ind.baseI.ref, chS.clef, 0);
								data2.Read(r);
							END;
							ind.gestion.Prefixe(ind.baseI.ref, data1, data2, c, lC);
							dd:= ind.baseI.LisPage(pSuiv, pif);
							pa:= dd(PageI);
							lCE:= pa.elems[numSuiv].finC- pa.elems[numSuiv- 1].finC;
							ASSERT(lC<= lCE, 60);
							IF lC< lCE THEN
								ind.TransOut(pa, numSuiv, el, cE, lCE);
								ind.TransIn(pa, el, c, lC, numSuiv);
								ind.baseI.EcrisPage(pSuiv);
								ind.baseI.EcrisPage(pa.cles);
							END;
						END;
						IF ind.pCour= p THEN
							ind.pCour:= ind.chaine;
						END;
						ind.baseI.EffacePage(p);
					END;
				ELSE
					dd:= ind.baseI.LisPage(p, pif);
					pa:= dd(PageI);
					dd:= ind.baseI.LisPage(pa.cles, clf);
					cl:= dd(ClesCont);
					g:= 1; d:= pa.nbEl+ 1;
					WHILE g< d DO
						i:= (g+ d) DIV 2;
						data1:= ind.gestion.f.New(pa.elems[i].finC- pa.elems[i- 1].finC);
						r.InitReader(ind.baseI.ref, cl.c, pa.elems[i- 1].finC);
						data1.Read(r);
						IF ind.gestion.CompP(cle, data1)= inf THEN
							d:= i;
						ELSE
							g:= i+ 1;
						END;
					END;
					DEC(d);
					IF d< pa.nbEl THEN
						pSuiv:= p;
						numSuiv:= d+ 1;
					END;
					Eff(pa.elems[d].ptr, p, d, h- 1);
					IF trouve THEN
						IF d> 0 THEN
							dd:= ind.baseI.LisPage(p, pif);
							pa:= dd(PageI);
							IF h= 1 THEN
								ind.TransOut(pa, d, el, c, lC);
								dec:= pa.nbEl< minEl;
								ind.baseI.EcrisPage(p);
								ind.baseI.EcrisPage(pa.cles);
							ELSE
								Del(pa.elems[d- 1].ptr, h- 1);
								IF dec THEN
									Corrige(p, d- 1);
								END;
							END;
							trouve:= FALSE;
						END;
					ELSIF dec THEN
						Corrige(p, d);
					END;
				END;
			END Eff;
		
		BEGIN (*Efface*)
			ASSERT(ind.baseI# NIL, 20);
			pSuiv:= bNil;
			Eff(ind.racI, bNil, 0, ind.haut);
			IF dec THEN
				d:= ind.baseI.LisPage(ind.racI, pif);
				pa:= d(PageI);
				IF pa.nbEl= 0 THEN
					DEC(ind.haut);
					pSuiv:= ind.racI;
					ind.racI:= pa.elems[0].ptr;
					ind.baseI.EffacePage(pa.cles);
					ind.baseI.EffacePage(pSuiv);
					ind.MAJRac;
				END;
			END;
		END Efface;
	
	(** Resets the current position of the index ind. After this action, ind.YACour() returns FALSE. *)
	PROCEDURE (ind: Index) InitCour*;
		
		BEGIN (*InitCour*)
			ASSERT(ind.baseI# NIL, 20);
			ind.pCour:= ind.chaine;
		END InitCour;
	
	(** Tests if the index ind is positioned on a key (returns TRUE then) or if its current position is reset (FALSE returned). *)
	PROCEDURE (ind: Index) YACour*(): BOOLEAN;
		
		BEGIN (*YACour*)
			ASSERT(ind.baseI# NIL, 20);
			RETURN ind.pCour# ind.chaine;
		END YACour;
	
	(** Position the index ind on the next key. If the current position is reset, the index is positioned on the first key. If the index is positioned on the last key, its current position becomes reset. *)
	PROCEDURE (ind: Index) Suivant*;
		
		VAR
			
			ch: Chaine;
			d: Data;
		
		BEGIN (*Suivant*)
			ASSERT(ind.baseI# NIL, 20);
			d:= ind.baseI.LisPage(ind.pCour, chf);
			ch:= d(Chaine);
			ind.pCour:= ch.suivant;
		END Suivant;
	
	(** Position the index ind on the previous key. If the current position is reset, the index is positioned on the last key. If the index is positioned on the first key, its current position becomes reset. *)
	PROCEDURE (ind: Index) Precedent*;
		
		VAR
			
			ch: Chaine;
			d: Data;
		
		BEGIN (*Precedent*)
			ASSERT(ind.baseI# NIL, 20);
			d:= ind.baseI.LisPage(ind.pCour, chf);
			ch:= d(Chaine);
			ind.pCour:= ch.precedent;
		END Precedent;
	
	(** Returns in cle the value of the key in the current position of the index ind. If the current position of the index is reset, the value of cle is not modified. The parameter cle must be sufficiently large to contain the key, or the result is unforeseeable. *)
	PROCEDURE (ind: Index) CleCour*(): Data;
		
		VAR
			
			ch: Chaine;
			r: Reader;
			data: Data;
		
		BEGIN (*CleCour*)
			ASSERT(ind.baseI# NIL, 20);
			IF ind.pCour= ind.chaine THEN
				RETURN NIL;
			END;
			data:= ind.baseI.LisPage(ind.pCour, chf);
			ch:= data(Chaine);
			data:= ind.gestion.f.New(ind.baseI.LisPageLong(ind.pCour)- taChaine);
			r.InitReader(ind.baseI.ref, ch.clef, 0);
			data.Read(r);
			RETURN data;
		END CleCour;
	
	(** Reads, at the current position of the index ind, the integer associated data. When a new key is inserted, the data which is initially attached to it has the value bNil. The reset position of an index has, itself, a associated data, legible by Index.LisVal. *)
	PROCEDURE (ind: Index) LisVal*(): LONGINT;
		
		VAR
			
			ch: Chaine;
			d: Data;
		
		BEGIN (*LisVal*)
			ASSERT(ind.baseI# NIL, 20);
			d:= ind.baseI.LisPage(ind.pCour, chf);
			ch:= d(Chaine);
			RETURN ch.posDon;
		END LisVal;
	
	(** Writes on the current position of the index ind the value val of the associated integer data. When a new key is inserted, the data which is initially attached to it has the value bNil. The reset position of an index has, itself, a associated data, modifiable by Index.EcrisVal. *)
	PROCEDURE (ind: Index) EcrisVal*(val: LONGINT);
		
		VAR
			
			ch: Chaine;
			d: Data;
		
		BEGIN (*EcrisVal*)
			ASSERT(ind.baseI# NIL, 20);
			d:= ind.baseI.LisPage(ind.pCour, chf);
			ch:= d(Chaine);
			ch.posDon:= val;
			ind.baseI.EcrisPage(ind.pCour);
		END EcrisVal;
	
	BEGIN (*TBoxBArbre*)
		NEW(cTLib);
		NEW(cTOcc);
		NEW(cQLib);
		NEW(cQOcc);
		NEW(pif);
		NEW(clf);
		NEW(chf);
		NEW(inff);
		NEW(cNIL, 1);
	END TBoxBArbre.
BIER   r   <       g 
     C  Syntax10.Scn.Fnt 12.03.2002  00:26:14  TimeStamps.New  