TextDocs.NewDoc     lF   CColor    Flat  Locked  Controls  Org 6  BIER`   b        3    Oberon10.Scn.Fnt           D       2        (               "           J   l   D               B                B        Y    T&   _       2        $    ^       y            B        5        ^        P       n        A        3        }        B        ?        X    `                   O   e	   c    @   +    9           H                                <        9        _                                 T   _           M                        +       W                H       `    e    W        /                    B            0    C       W    B   J    H   F       o       Q       s    Z   `    j    h                                           
                   q    U          V         \            L           G            R)       
       
   G    	    (* ETH Oberon, Copyright 2001 ETH Zuerich Institut fuer Computersysteme, ETH Zentrum, CH-8092 Zuerich.
Refer to the "General ETH Oberon System Source License" contract available at: http://www.oberon.ethz.ch/ *)

 MODULE Kernel;	(** non-portable *)	(* rc/ard/nm/jm/jdv/pjm/rml*)

(** Module Kernel is responsible for memory allocation, garbage collection and other run-time support for the Oberon language, as well as interfacing to the underlying machine.  Using features exported from module Kernel may result in an unportable module, as not all Oberon Kernels are the same. This module is of little importance to most Oberon programmers. *)

IMPORT SYSTEM;

CONST
		(* heap/GC *)
	B = 32;	(* heap block size, must be a multiple of 32 *)
	N = 9;	(* number of free lists - 1 *)
	nil = 0;
	MarkBit = 0; ArrayBit = 1;  FreeBit = 2;  SubObjBit = 3; 
	mark = {MarkBit}; array = {ArrayBit};  free = {FreeBit};  subobj = {SubObjBit};
	ReserveSize = 65536-8;	(* bytes reserved for "Out of memory" trap handling *)
	AlwaysGC = TRUE;

		(* interrupt handling *)
	IRQ* = 32;	(** Interrupt  number for IRQ0, used with InstallIP *)	(* {IRQ MOD 8 = 0} *)
	IDTSize = 32+16;
	IntA0 = 020H;	IntA1 = 021H;	(* Interrupt Controller 1 *)
	IntB0 = 0A0H;	IntB1 = 0A1H;	(* Interrupt Controller 2 *)

	V86EnterInt = 28;  V86ExitInt = 29;	(* mirrored in module V86 *)
	
	TraceUnexpected = TRUE;	(* show trace info about unexpected interrupts *)
	IgnoreTrap15 = TRUE;	(* ignore spurious trap 15 generated by APIC *)
	
	MaxIntsShift = 2;	(* minimum value 1 *)
	MaxInts = ASH(1, MaxIntsShift);	(* max number of shared interrupts (including terminator) *)
	MaxIntsShift2 = 2+2;

		(* memory management *)
	KernelCodeSel = 1*8;	(* selector 1 in gdt, RPL 0 *)
	KernelStackSel = 2*8;	(* selector 2 in gdt, RPL 0 *)
	UserCodeSel = 3*8 + 3;	(* selector 3 in gdt, RPL 3 *)
	UserStackSel = 4*8 + 3;	(* selector 4 in gdt, RPL 3 *)
	DataSel = 4*8;	(* selector 4 in gdt, RPL 0 *)
	KernelTR = 5*8;	(* selector 5 in gdt, RPL 0 *)
		(* selectors 6..8 are used by APM *)
	PS = 4096;	(* page size in bytes*)
	PTEs = 1024;	(* number of page table/directory entries *)
	KernelStackSize = 16*1024;	(* {KernelStackSize MOD 4096 = 0} *)
	PageNotPresent = 0;	(* not present page *)
	NormalPage = 7;	(* user, present, r/w *)
	V86 = TRUE;	(* experimental V86 access *)
	VesaAdr = 0E0000000H;  VesaSize = 400000H;	(* where MapVesa=1 maps *)
	DefaultPageHeap = 16384;
	DefaultStackSize = 128*1024;
	MapCacheSize = 4;
	
	DisplayBase = 0B8000H;
	DisplayWidth = 80*2;	(* width of text display line in bytes *)
	DisplayHeight = 25;	(* number of text display lines *)
	
		(* timer *)
	TimeUnit* = 1000;	(** timer ticks per second, returned by GetTimer *)
	Rate = 1193180;	(* timer clock is 1.19318 MHz *)

	LittleEndian* = TRUE;	(** byte order of basic types *)
	LsbIs0* = TRUE;	(** bit order of SET type *)
	
TYPE
	ADDRESS = LONGINT;
	Name* = ARRAY 32 OF CHAR;	(* module name *)
	Proc* = PROCEDURE;
	Cmd* = RECORD
		name*: Name;
		adr*: ADDRESS;
	END;
	Module* = POINTER TO ModuleDesc;	(* module descriptor *)
	ExportPtr* = POINTER TO ExportDesc;
	ExportDesc* = RECORD
		fp*: LONGINT; adr*: LONGINT;
		nofExp*: INTEGER; 
		dsc*: POINTER TO ARRAY OF ExportDesc
	END;
	ArrayPtr* = POINTER TO ARRAY OF LONGINT;
	ModuleDesc* = RECORD	(* Note: if this is changed, also update the static Linker *)
		next*: Module;
		name*: Name;
		init*, trapped*: BOOLEAN;
		refcnt*, sb*: LONGINT;
		entries*: POINTER TO ARRAY OF ADDRESS;
		cmds*: POINTER TO ARRAY OF Cmd;
		ptrTab*, tdescs*: POINTER TO ARRAY OF ADDRESS;
		imports*: POINTER TO ARRAY OF Module;
		data*, code*, refs*: POINTER TO ARRAY OF CHAR;
		publics*, privates*: LONGINT;	(* for slim binaries *)
		nofimp*: INTEGER; import*: ArrayPtr;
		nofstrc*: INTEGER; struct*: ArrayPtr;
		nofreimp*: INTEGER; reimp*: ArrayPtr;
		export*: ExportDesc;
		term*: Proc
	END;

		(* type descriptors *)
	Tag = POINTER TO TypeDesc;
	TypeDesc = RECORD
		size: LONGINT;
		ptroff: LONGINT
	END;

		(* interrupt handling *)
	TrapHandler* = PROCEDURE (err, fp, pc, pf: LONGINT);

		(* heap/GC *)
	FreeBlockPtr = POINTER TO FreeBlock;
	FreeBlock = RECORD
		(* off-4 *) tag: Tag;
		(* off0 *) size: LONGINT;	(* field size aligned to 8-byte boundary, size MOD B = B-4 *)
		(* off4 *) next: ADDRESS
	END;
	BlockPtr = POINTER TO Block;
	Block = RECORD
		lastElemToMark, currElem, firstElem: BlockPtr
	END;
	Blockm4Ptr = POINTER TO Blockm4;
	Blockm4 = RECORD
		tag: Tag;
		lastElemToMark, currElem, firstElem: LONGINT
	END;
	InitPtr = POINTER TO RECORD tag: Tag; z0, z1, z2, z3, z4, z5, z6, z7: LONGINT END;
		(* the following type is used indirectly in InitHeap *)
	PtrElemDesc = RECORD a: SYSTEM.PTR END;	(* has same type descriptor as element of ARRAY OF POINTER *)
	
	Finalizer* = PROCEDURE (obj: SYSTEM.PTR);
	FinObj = POINTER TO FinObjNode;
	FinObjNode = RECORD
		next: FinObj;
		obj: LONGINT;
		marked: BOOLEAN;
		fin: Finalizer;
	END;

		(* memory management *)
	GateDescriptor = RECORD
		offsetBits0to15: INTEGER;
		selector: INTEGER;
		gateType: INTEGER;
		offsetBits16to31: INTEGER
	END;
	SegmentDescriptor = RECORD
		limit0to15: INTEGER;
		base0to15: INTEGER;
		base16to23: CHAR;
		accessByte: CHAR;
		granularityByte: CHAR;
		base24to31: CHAR
	END;
	TSSDesc = RECORD	(* 1, p. 485 and p. 612 for required fields *)
		Link, pad0: INTEGER;
		ESP0: LONGINT;
		ESS0, pad1: INTEGER;
		ESP1: LONGINT;
		ESS1, pad2: INTEGER;
		ESP2: LONGINT;
		ESS2, pad3: INTEGER;
		CR3: LONGINT;
		EIP: LONGINT;
		EFLAGS: SET;
		EAX, ECX, EDX, EBX, ESP, EBP, ESI, EDI: LONGINT;
		ES, pad4: INTEGER;
		CS, pad5: INTEGER;
		SS, pad6: INTEGER;
		DS, pad7: INTEGER;
		FS, pad8: INTEGER;
		GS, pad9: INTEGER;
		LDT, pad10: INTEGER;
		TaskAttributes: INTEGER;
		IOBitmapOffset: INTEGER;
		IOBitmap: ARRAY 65536 DIV 32+1 OF SET	(* only required if V86=TRUE *)
	END;
	IDT = ARRAY IDTSize OF GateDescriptor;
	GDT = ARRAY 9 OF SegmentDescriptor;
(*
	PageTablePtr = POINTER TO RECORD a:ARRAY PTEs OF LONGINT END; 
	PageDirectoryPtr = POINTER TO RECORD a:ARRAY PTEs OF LONGINT END; 
*)
	Vendor = ARRAY 13 OF CHAR;
	V86Regs = RECORD	(* 19 dwords, order important.  mirrored in module V86. *)
		EDI, ESI, EBP, fillerESP, EBX, EDX, ECX, EAX: LONGINT;
		int, errCode: LONGINT;
		EIP, CS: LONGINT;
		EFLAGS: SET;
		ESP, SS, ES, DS, FS, GS: LONGINT
	END;

	MilliTimer* = RECORD target: LONGINT END;

VAR
		(* exported variables *)
	version*: ARRAY 32 OF CHAR;
	modules* : Module;	(** list of modules, patched by Linker *)
	StackOrg* : ADDRESS;	(** highest address on stack (single-process system) *)
	bt*: ADDRESS;	(** boot table *)
	tspeed*: LONGINT;	(** <=0: no V24 tracing, >0: tspeed = speed of com output *)
	tbase*: INTEGER;	(** trace base port, 3F8H=com1, 2F8H=com2, etc. *)
	break* : BOOLEAN;	(** has ctrl-break been pressed? *)
	copro*: BOOLEAN;	(** is a coprocessor present? *)
	inGC*: BOOLEAN;	(** executing inside GC? *)
	EnableGC*, DisableGC*: Proc;	(** Enable or Disable the GC *)
	timer*: Proc;	(** for internal use *)
	runtime*: ARRAY 3 OF ADDRESS;	(** for internal use *)
	traceConsole*: BOOLEAN;	(** trace on console? *)
	shutdown*: LONGINT;	(** system shutdown code 0=running, 1=powering off, 2=rebooting *)
	tlpt: LONGINT;
	bioslpt: ARRAY 3 OF INTEGER;

		(* heap/GC *)
	memTop, heapTop: ADDRESS;
	firstBlock, endBlock: (*FreeBlockPtr*) ADDRESS;
	A: ARRAY N+1 OF (*FreeBlockPtr*) ADDRESS;
	reserve: BlockPtr;	(* reserved memory for out of memory trap *)
	sysres: InitPtr;	(* root for memory reserved by system *)
	initres: InitPtr;	(* root for init calls *)
	ptrElemTag: ADDRESS;
	candidates: ARRAY 1024 OF ADDRESS;	(* stack pointer candidates *)
	nofcand: INTEGER;
	firstTry: BOOLEAN;	(* used with GC & reserve *)
	GCstack: BOOLEAN;
	FinObjs: FinObj;
	traceheap: SET;
	dma0, dma1, dmafree: ADDRESS;
	GClevel: LONGINT;
	
		(* interrupt handling/memory management *)
	handler0, handler: TrapHandler;	(* trap handlers *)
	loop: Proc;	(* main loop *)
	idt: IDT;	(* interrupt descriptor table *)
	gdt: GDT;	(* global descriptor table *)
	ktss: TSSDesc;
	glue: ARRAY IDTSize OF ARRAY 12 OF CHAR;	(* code *)
	intHandler: ARRAY IDTSize OF ARRAY MaxInts OF Proc;	(* should not use pointers, as they are modified by GC *)
	intHandlerAdr: LONGINT;	(* assembler workaround *)
	defaultHandler: Proc;
	kernelpd, v86pd: ADDRESS;	(* address of page directory *)
	handlingtrap: BOOLEAN;
	oldcopro: BOOLEAN;	(* use old IRQ13 method of signalling copro errors *)
	trapCR: ARRAY 5 OF LONGINT;  trapDR: ARRAY 8 OF LONGINT;
	trapfpu: ARRAY 7 OF SET;
	mapPtr: LONGINT;	(* virtual address of current start of memory mapping area (grows downward) *)
	mapcache: ARRAY MapCacheSize OF RECORD physAdr, size, virtAdr: LONGINT END;
	vregadr: LONGINT;	(* V86 registers *)
	vframe: LONGINT;
	traceBufAdr, traceBufSize, traceHead, traceTail, traceMark: LONGINT;
	
		(* misc. *)
	second, minute, hour, day, month, year: INTEGER;	(* RT clock status *)
	clockmissed, clockmode, clockints: LONGINT;
	ticks: LONGINT;	(* timer ticks since startup *)
	configadr: ADDRESS;	(* address of config info *)
	pspeed: LONGINT;	(* previous trace speed *)
	pageheap, pageheap0, pageheap1: ADDRESS;
	displayPos: LONGINT;
	kpar: ARRAY 2 OF LONGINT;
	apmofs: ADDRESS;
	powersave: BOOLEAN;

		(* cpu *)
	cpuversion, cpufeatures: LONGINT;
	cpuvendor: Vendor;
	cpu: SHORTINT;

(** -- Low-level tracing support -- *)

PROCEDURE -GetFlags(): SET;
CODE {SYSTEM.i386}
	PUSHFD
	POP EAX
END GetFlags;

PROCEDURE -SetFlags(flags: SET);
CODE {SYSTEM.i386}
	POPFD
END SetFlags;

(* Copy4 - Copy "size4" dwords from "src" to "dst" *)

PROCEDURE -Copy4(src, dst, size4: LONGINT);
CODE {SYSTEM.i386}
	POP ECX
	POP EDI
	POP ESI
	CLD
	REP
	MOVSD
END Copy4;

(* Fill4 - Fill "size4" dwords at "dest" with "filler" *)

PROCEDURE Fill4(dst, size4, filler: LONGINT);
CODE {SYSTEM.i386}
	MOV EDI, dst[EBP]
	MOV ECX, size4[EBP]
	MOV EAX, filler[EBP]
	CLD
	REP STOSD
END Fill4;

(** WriteChar - Write a character to the trace output (not reentrant). *)

PROCEDURE WriteChar*(c: CHAR);
VAR status, flags: SET;  adr: LONGINT;
BEGIN
	flags := GetFlags();  SYSTEM.CLI();
	IF traceConsole THEN	(* screen tracing on *)
		IF c = 0DX THEN
			IF displayPos MOD DisplayWidth # 0 THEN DEC(displayPos, displayPos MOD DisplayWidth) END
		ELSIF c = 0AX THEN
			INC(displayPos, DisplayWidth)
		ELSE
			SYSTEM.PUT(DisplayBase+displayPos, SYSTEM.VAL(INTEGER, 0700H+ORD(c)));
			INC(displayPos, 2)
		END;
		IF displayPos >= DisplayWidth*DisplayHeight THEN
			DEC(displayPos, DisplayWidth);
			Copy4(DisplayBase+DisplayWidth, DisplayBase, (DisplayWidth*(DisplayHeight-1)) DIV 4);
			Fill4(DisplayBase+DisplayWidth*(DisplayHeight-1), DisplayWidth DIV 4, 07200720H)
		END;
			(* move cursor *)
		SYSTEM.PORTOUT(3D4H, 0EX);
		SYSTEM.PORTOUT(3D5H, CHR(displayPos DIV 2 DIV 100H));
		SYSTEM.PORTOUT(3D4H, 0FX);
		SYSTEM.PORTOUT(3D5H, CHR(displayPos DIV 2 MOD 100H))
	END;
	IF tspeed > 0 THEN	(* V24 tracing on *)
		IF tspeed # pspeed THEN	(* init *)
			SYSTEM.PORTOUT(tbase+3, 80X);	(* set speed *)
			SYSTEM.PORTOUT(tbase+1, CHR(ASH(115200 DIV tspeed, -8)));
			SYSTEM.PORTOUT(tbase, CHR(115200 DIV tspeed));
			SYSTEM.PORTOUT(tbase+3, 3X);	(* 8N1 *)
			SYSTEM.PORTOUT(tbase+4, 3X);	(* DTR & RTS on *)
			SYSTEM.PORTOUT(tbase+1, 0X);	(* interrupts off *)
			pspeed := tspeed
		END;
		REPEAT	(* wait until port is ready to accept a character *)
			SYSTEM.PORTIN(tbase+5, SYSTEM.VAL(CHAR, status))
		UNTIL status * {5,6} = {5,6};	(* THR & TSR empty *)
		SYSTEM.PORTOUT(tbase, c)
	END;
	IF traceBufAdr # 0 THEN	(* buffer saving on *)
		adr := (traceTail+1) MOD traceBufSize;
		IF adr # traceHead THEN
			SYSTEM.PUT(traceBufAdr + traceTail, c);
			traceTail := adr
		END
	END;
	IF tlpt > 0 THEN
		REPEAT
			SYSTEM.PORTIN(tlpt+1, SYSTEM.VAL(CHAR, status));
			SYSTEM.PORTIN(tlpt+1, SYSTEM.VAL(CHAR, status))
		UNTIL 7 IN status;
		SYSTEM.PORTOUT(tlpt, c);
		SYSTEM.PORTOUT(tlpt+2, 0DX); SYSTEM.PORTOUT(tlpt+2, 0DX);
		SYSTEM.PORTOUT(tlpt+2, 0CX)
	END;
	SetFlags(flags)
END WriteChar;

(** WriteString - Write a string *)

PROCEDURE WriteString*(s: ARRAY OF CHAR);
VAR i: INTEGER;
BEGIN
	i := 0;  WHILE s[i] # 0X DO WriteChar(s[i]);  INC(i) END
END WriteString;

(** WriteLn - Skip to the next line on trace output *)

PROCEDURE WriteLn*;
BEGIN
	WriteChar(0DX);  WriteChar(0AX)
END WriteLn;

(** WriteInt - Write "x" as a decimal number.  "w" is the field width. *)

PROCEDURE WriteInt*(x, w: LONGINT);
VAR i: LONGINT;  x0: LONGINT;  a: ARRAY 12 OF CHAR;  s: ARRAY 2 OF CHAR;
BEGIN
	IF x < 0 THEN
		IF x = MIN(LONGINT) THEN
			WriteString("-2147483648");
			RETURN
		ELSE
			DEC(w); x0 := -x
		END
	ELSE
		x0 := x
	END;
	i := 0;
	REPEAT
		a[i] := CHR(x0 MOD 10 + 30H); x0 := x0 DIV 10; INC(i)
	UNTIL x0 = 0;
	s[0] := " ";  s[1] := 0X;
	WHILE w > i DO WriteString(s); DEC(w) END;
	s[0] := "-";
	IF x < 0 THEN WriteString(s) END;
	REPEAT DEC(i); s[0] := a[i]; WriteString(s) UNTIL i = 0
END WriteInt;

(** WriteHex - Write "x" as a hexadecimal number. The absolute value of "w" is the field width.  If "w" is negative, two hex digits are printed (x MOD 100H), otherwise 8 digits are printed. *)

PROCEDURE WriteHex*(x, w: LONGINT);
VAR
	buf: ARRAY 10 OF CHAR;
	i, j: LONGINT;
BEGIN
	IF w >= 0 THEN j := 8 ELSE j := 2; w := -w END;
	FOR i := j+1 TO w DO WriteChar(" ") END;
	FOR i := j-1 TO 0 BY -1 DO
		buf[i] := CHR(x MOD 10H + 48);
		IF buf[i] > "9" THEN
			buf[i] := CHR(ORD(buf[i]) - 48 + 65 - 10)
		END;
		x := x DIV 10H
	END;
	buf[j] := 0X;
	WriteString(buf)
END WriteHex;

(** WriteMemory - Display a block of memory. *)

PROCEDURE WriteMemory*(adr, size: LONGINT);
VAR
	i, j: LONGINT;
	buf: ARRAY 4 OF CHAR;
BEGIN
	buf[1] := 0X;
	size := adr+size-1;
	FOR i := adr TO size BY 16 DO
		WriteHex(i, 9);
		FOR j := i TO i+15 DO
			IF j <= size THEN
				SYSTEM.GET(j, buf[0]);
				WriteHex(SYSTEM.VAL(SHORTINT, buf[0]), -3)
			ELSE
				buf := "   ";  WriteString(buf);  buf[1] := 0X
			END
		END;
		buf[0] := " ";	WriteString(buf);
		FOR j := i TO i+15 DO
			IF j <= size THEN
				SYSTEM.GET(j, buf[0]);
				IF (buf[0] < " ") OR (buf[0] >= CHR(127)) THEN
					buf[0] := "."
				END;
				WriteString(buf)
			END
		END;
		WriteLn
	END
END WriteMemory;

PROCEDURE Bits(s: ARRAY OF CHAR;  x: SET;  ofs, n: LONGINT);
BEGIN
	WriteString(s);  WriteChar("=");
	REPEAT
		DEC(n);
		IF (ofs+n) IN x THEN WriteChar("1") ELSE WriteChar("0") END
	UNTIL n = 0
END Bits;

PROCEDURE WriteFlags(f: LONGINT);
VAR s, t: SET;
BEGIN
	s := SYSTEM.VAL(SET, f);
	WriteHex(f, 8);  WriteString(" (");
	t := SYSTEM.LSH(s, -6)*{5} + SYSTEM.LSH(s, 4)*{4} + SYSTEM.LSH(s, 1)*{3} + SYSTEM.LSH(s, -2)*{2} + 
		SYSTEM.LSH(s, -5)*{1} + SYSTEM.LSH(s, -7)*{0};
	Bits("OCPAZS", t, 0, 6);
	Bits(" IT", s, 8, 2);
	Bits(" D", s, 10, 1);
	Bits(" IOPL", s, 12, 2);
	Bits(" NT", s, 14, 1);
	Bits(" AC,VM,RF", s, 16, 3);
	(*Bits(" ID,VIP,VIF", s, 19, 3);*)
	WriteChar(")")
END WriteFlags;

(*
(* WriteDesc - Write a 8-byte segment descriptor. *)

PROCEDURE WriteDesc(a: LONGINT);
VAR type, t, t0, t1: LONGINT;  s0, s1: SET;
BEGIN
	SYSTEM.GET(a, s0);  SYSTEM.GET(a+4, s1);  SYSTEM.GET(a, t0);  SYSTEM.GET(a+4, t1);
	WriteHex(a, 8);  WriteHex(t0, 9);  WriteHex(t1, 9);
	Bits(" P", s1, 15, 1);  Bits(", DPL", s1, 13, 2);  WriteString(", ");
	IF 12 IN s1 THEN	(* data/code *)
		type := 16;
		IF 22 IN s1 THEN WriteString("32-bit") ELSE WriteString("16-bit") END;
		IF 11 IN s1 THEN WriteString(" code");  Bits(", CRA", s1, 8, 3)
		ELSE WriteString(" data");  Bits(", EWA", s1, 8, 3)
		END;
		Bits(", avl", s1, 20, 1)
	ELSE
		type := ASH(t1, -8) MOD 16;
		CASE type OF
			0, 8, 10, 13: WriteString("type");  WriteInt(type, 1)
			|1, 3: WriteString("16-bit TSS");  Bits(", busy", s1, 9, 1)
			|2: WriteString("LDT")
			|4: WriteString("16-bit call gate")
			|5: WriteString("task gate")
			|6, 14: WriteString("interrupt gate");  Bits(", 32bit", s1, 11, 1)
			|7, 15: WriteString("trap gate");  Bits(", 32bit", s1, 11, 1)
			|9, 11: WriteString("32-bit TSS");  Bits(", busy", s1, 9, 1);  Bits(", avl", s1, 20, 1)
			|12: WriteString("32-bit call gate, params=");  WriteInt(t1 MOD 32, 1)
		END
	END;
	WriteLn;
	CASE type OF
		5: WriteString("  tsssel=");  WriteHex(ASH(t0, -16), 8);  WriteLn
		|6, 7, 12, 14, 15:
			WriteString("  segsel=");  WriteHex(ASH(t0, -16), 8);
			WriteString(", ofs=");  WriteHex(SYSTEM.VAL(LONGINT, s1 * {16..31} + s0 * {0..15}), 8);  WriteLn
		|9, 11, 16:
			t := SYSTEM.VAL(LONGINT, s1 * {24..31} + SYSTEM.LSH(s1 * {0..7}, 16) + SYSTEM.LSH(s0, -16));
			WriteString("  base=");  WriteHex(t, 8);
			t := SYSTEM.VAL(LONGINT, s1 * {16..19} + s0 * {0..16});
			WriteString(", limit=");  WriteHex(t, 8);
			IF 23 IN s1 THEN WriteString(" * 4k =");  WriteHex(t * 4096, 9) END;
			WriteLn
		ELSE (* skip *)
	END
END WriteDesc;

PROCEDURE WriteTSS(a: LONGINT);
VAR t: LONGINT;
	
	PROCEDURE Val(s: ARRAY OF CHAR; ofs: LONGINT);
	BEGIN
		SYSTEM.GET(a+ofs, ofs);
		WriteChar(" ");  WriteString(s);  WriteChar("=");  WriteHex(ofs, 8)
	END Val;

BEGIN
	WriteHex(a, 8);  WriteString(" 32-bit TSS");  Val("LINK", 0);  WriteString(" IOMAP=");
	SYSTEM.GET(a+100, t);  WriteHex(ASH(t, -16) MOD 10000H, 8);  WriteLn;
	Val("CS:", 76);  Val("DS:", 84);  Val("ES:", 72);  Val("SS:", 80);  Val("SS0", 8);  Val("ESP0", 4);  WriteLn;
	Val("EIP", 32);  Val("ESI", 64);  Val("EDI", 68);  Val("ESP", 56);  Val("SS1", 16);  Val("ESP1", 12);  WriteLn;
	Val("EAX", 40);  Val("EBX", 52);  Val("ECX", 44);  Val("EDX", 48);  Val("SS2", 24);  Val("ESP2", 20);  WriteLn;
	Val("EBP", 60);  Val("FS:", 88);  Val("GS:", 92);  Val("LDT", 96);  WriteString(" DTF=");  
	SYSTEM.GET(a+100, t);  WriteHex(t MOD 10000H, 8);  Val("PDBR", 28);  WriteLn;
	SYSTEM.GET(a+36, t);  WriteString(" EFLAGS=");  WriteFlags(t);  WriteLn
END WriteTSS;

PROCEDURE StoreIDT(adr: LONGINT);
CODE {SYSTEM.i386}
	MOV EAX, adr[EBP]
	SIDT [EAX]
END StoreIDT;

PROCEDURE StoreGDT(adr: LONGINT);
CODE {SYSTEM.i386}
	MOV EAX, adr[EBP]
	SGDT [EAX]
END StoreGDT;

PROCEDURE WriteGDT;
VAR ptr: ARRAY 2 OF LONGINT;  a, n, i: LONGINT;
BEGIN
	StoreGDT(SYSTEM.ADR(ptr)+2);
	a := ptr[1];  n := ASH(ptr[0], -16);
	WriteString("GDT base=");  WriteHex(a, 8);  WriteString(" limit=");  WriteHex(n, 8);  WriteLn;
	n := (n+1) DIV 8;  i := 0;
	WHILE i < n DO
		WriteHex(i, -2);  WriteChar(" ");
		WriteDesc(a);  INC(a, 8);  INC(i)
	END;
	WriteLn
END WriteGDT;

PROCEDURE WriteIDT;
VAR ptr: ARRAY 2 OF LONGINT;  a, n, i: LONGINT;
BEGIN
	StoreIDT(SYSTEM.ADR(ptr)+2);
	a := ptr[1];  n := ASH(ptr[0], -16);
	WriteString("IDT base=");  WriteHex(a, 8);  WriteString(" limit=");  WriteHex(n, 8);  WriteLn;
	n := (n+1) DIV 8;  i := 0;
	WHILE i < n DO
		WriteHex(i, -2);  WriteChar(" ");
		WriteDesc(a);  INC(a, 8);  INC(i)
	END;
	WriteLn
END WriteIDT;
*)

(* Wait - Waste a little time (jmp $+2) *)

PROCEDURE Wait;
CODE {SYSTEM.i386}
	JMP 0
END Wait;

PROCEDURE Reboot;
CODE {SYSTEM.i386, SYSTEM.Privileged}
	PUSH 0
	PUSH 0
	LIDT [ESP]
	INT 3
END Reboot;

(** -- Trap handling -- *)

(** GetMod - Return the loaded module that contains code address pc. *)

PROCEDURE GetMod*(pc : LONGINT): Module;
VAR m: Module;  base: LONGINT;
BEGIN
	m := modules;
	WHILE m # NIL DO
		base := SYSTEM.ADR(m.code[0]);	(* include 1 byte after module in module, therefore <= below *)
		IF (base <= pc) & (pc <= base + LEN(m.code)) THEN
			RETURN m
		ELSIF (SYSTEM.ADR(m.data[0]) <= pc) & (pc <= SYSTEM.ADR(m.data[0]) + LEN(m.data)) THEN
			RETURN m
		ELSE
			m := m.next
		END
	END;
	RETURN NIL
END GetMod;

PROCEDURE ShowInt(bp: LONGINT;  msg: ARRAY OF CHAR);
VAR pc: LONGINT;  m: Module;
BEGIN
	SYSTEM.GET(bp+48, pc);	(* EIP *)
	WriteString(msg);
	m := GetMod(pc);
	IF m # NIL THEN
		WriteString(m.name);  WriteString("  PC = ");
		WriteInt(pc-SYSTEM.ADR(m.code[0]), 1)
	ELSE
		WriteString("unknown module")
	END;
	WriteLn
END ShowInt;

(* SetupFPU - Setup FPU control word.  Note: FCR also set in Reals and System. *)

PROCEDURE SetupFPU;
CODE {SYSTEM.i386, SYSTEM.FPU}
	PUSH EAX	(* temp space on stack *)
	FNINIT
	FNSTCW [ESP]	(* store current CW *)
	WAIT	(* wait for store to finish *)
	AND BYTE [ESP], 0C0H	(* keep bits 6..15, clear 0..5 *)
	OR BYTE [ESP], 32H	(* mask denormal, underflow & precision loss *)
	FLDCW [ESP]	(* load new CW *)
	POP EAX	(* throw away temp *)
END SetupFPU;

(* GoFrom0To3 - Transfer control to loop at user level *)

PROCEDURE -GoFrom0To3(ss, sp, cs: LONGINT;  ip: Proc);
CODE {SYSTEM.i386}
	POP EDX	(* ip *)
	POP ECX	(* cs *)
	PUSHFD	(* EFLAGS3 *)
	PUSH ECX	(* CS3 *)
	PUSH EDX	(* EIP3 *)
	MOV EBP, 7FFFFFFFH
	IRETD	(* SS3 & ESP3 already on stack *)
END GoFrom0To3;

(* CR0 etc. - Return value of control/debug registers *)

PROCEDURE -CR0(): LONGINT;
CODE {SYSTEM.i386, SYSTEM.Privileged}
	MOV EAX, CR0
END CR0;

PROCEDURE -CR2(): LONGINT;
CODE {SYSTEM.i386, SYSTEM.Privileged}
	MOV EAX, CR2
END CR2;

PROCEDURE -CR3(): LONGINT;
CODE {SYSTEM.i386, SYSTEM.Privileged}
	MOV EAX, CR3
END CR3;

PROCEDURE -CR4(): LONGINT;
CODE {SYSTEM.Pentium, SYSTEM.Privileged}
	MOV EAX, CR4
END CR4;

PROCEDURE -DR0(): LONGINT;
CODE {SYSTEM.i386, SYSTEM.Privileged}
	MOV EAX, DR0
END DR0;

PROCEDURE -DR1(): LONGINT;
CODE {SYSTEM.i386, SYSTEM.Privileged}
	MOV EAX, DR1
END DR1;

PROCEDURE -DR2(): LONGINT;
CODE {SYSTEM.i386, SYSTEM.Privileged}
	MOV EAX, DR2
END DR2;

PROCEDURE -DR3(): LONGINT;
CODE {SYSTEM.i386, SYSTEM.Privileged}
	MOV EAX, DR3
END DR3;

PROCEDURE -DR6(): LONGINT;
CODE {SYSTEM.i386, SYSTEM.Privileged}
	MOV EAX, DR6
	MOV EBX, EAX
	AND EBX, 0FFFF1FF0H
	MOV DR6, EBX	; clear the status bits
END DR6;

PROCEDURE -DR7(): LONGINT;
CODE {SYSTEM.i386, SYSTEM.Privileged}
	MOV EAX, DR7
END DR7;

PROCEDURE -DS(): LONGINT;
CODE {SYSTEM.i386}
	XOR EAX, EAX
	MOV AX, DS
END DS;

PROCEDURE -ES(): LONGINT;
CODE {SYSTEM.i386}
	XOR EAX, EAX
	MOV AX, ES
END ES;

PROCEDURE -FS(): LONGINT;
CODE {SYSTEM.i386}
	XOR EAX, EAX
	MOV AX, FS
END FS;

PROCEDURE -GS(): LONGINT;
CODE {SYSTEM.i386}
	XOR EAX, EAX
	MOV AX, GS
END GS;

PROCEDURE -SS(): LONGINT;
CODE {SYSTEM.i386}
	XOR EAX, EAX
	MOV AX, SS
END SS;

PROCEDURE -StoreFPEnv(adr: LONGINT);
CODE {SYSTEM.i386, SYSTEM.FPU}
	POP EBX
	FNSTENV [EBX]	; also masks all exceptions
	FWAIT
END StoreFPEnv;

PROCEDURE -CLTS;
CODE {SYSTEM.i386, SYSTEM.Privileged}
	CLTS
END CLTS;

(* ExceptionHandler - Exception dispatcher *)

PROCEDURE ExceptionHandler;
VAR bp, err, fp, pc, cs, osp: LONGINT;  flags: SET;
BEGIN
	trapCR[0] := CR0();  trapCR[2] := CR2();  trapCR[3] := CR3();	(* CR1 is not documented *)
	IF cpu >= 5 THEN trapCR[4] := CR4() END;
	trapDR[0] := DR0();  trapDR[1] := DR1();  trapDR[2] := DR2();  trapDR[3] := DR3();
	trapDR[6] := DR6();  trapDR[7] := DR7();	(* DR4-5 are not documented *)
	CLTS;	(* ignore task switch flag *)
	SYSTEM.STI();
		(* get state from stack *)
	SYSTEM.GETREG(5, bp);	(* EBP *)
	SYSTEM.GET(bp+48, pc);	(* EIP *)
	SYSTEM.GET(bp+52, cs);	(* get CS'' *)
	SYSTEM.GET(bp+56, flags);	(* EFLAGS'' *)
	SYSTEM.GET(bp+40, err);	(* interrupt number *)
	IF oldcopro & (err = IRQ+13) THEN
		SYSTEM.PORTOUT(0F0H, 0X);	(* clear copro busy flag *)
		SYSTEM.CLI();
		SYSTEM.PORTOUT(IntB0, 20X);	(* signal EOI (we won't return to glue code) *)
		SYSTEM.PORTOUT(IntA0, 20X);
		SYSTEM.STI();
		err := 16
	END;
	err := -err;	(* convert interrupt number to non-positive error number *)
	IF err = -3 THEN	(* was HALT (INT 3) *)
		DEC(pc);	(* point to the int 3 instruction (assume 0CCX, not 0CDX 3X) *)
		IF 17 IN flags THEN	(* VM *)
			(* skip *)
		ELSIF cs MOD 4 = 0 THEN	(* level 0 *)
			SYSTEM.GET(bp+60, err)	(* get halt code (normally positive) *)
		ELSE
			SYSTEM.GET(bp+60, fp);	(* get outer ESP *)
			SYSTEM.GET(fp, err);	(* get halt code (normally positive) *)
			IF err = MAX(INTEGER) THEN
				SYSTEM.PUT(bp+60, fp+4)	(* pop outer stack *)
			END
		END
	ELSIF err = -4 THEN	(* was overflow (INTO) *)
		DEC(pc)	(* point to the into instruction *)
	ELSIF err = -14 THEN	(* was page fault *)
		IF pc = 0 THEN	(* was at EIP *)
			IF ~(17 IN flags) & (cs MOD 4 # 0) THEN
				SYSTEM.GET(bp+60, osp);	(* get call NIL return value *)
				SYSTEM.GET(osp, pc)
			END
		END
	ELSIF err = -16 THEN	(* FPU error *)
		IF copro THEN
			StoreFPEnv(SYSTEM.ADR(trapfpu[0]));
			SYSTEM.GET(SYSTEM.ADR(trapfpu[3]), pc);	(* EIP *)
			IF 2 IN trapfpu[1] THEN err := -32
			ELSIF 3 IN trapfpu[1] THEN err := -33
			ELSIF 0 IN trapfpu[1] THEN err := -34
			ELSIF 6 IN trapfpu[1] THEN err := -35
			ELSIF 1 IN trapfpu[1] THEN err := -36
			ELSIF 4 IN trapfpu[1] THEN err := -37
			ELSIF 5 IN trapfpu[1] THEN err := -38
			ELSE (* skip {err = -16} *)
			END
		END
	END;
	IF copro THEN SetupFPU END;	(* was FPU error *)
	SYSTEM.GET(bp+16, fp);	(* EBP *)
		(* call handler *)
	IF handler0 # NIL THEN handler0(err, fp, pc, trapCR[2]) END;
		(* upcall runs in supervisor mode on limited kernel stack.  overflow will cause reboot. *)
	IF handler # NIL THEN handler(err, fp, pc, trapCR[2]) END;
	IF ~((err = MAX(INTEGER)) & (cs MOD 4 # 0)) THEN	(* fall back to loop *)
		IF (loop # NIL) & ~inGC THEN
			GoFrom0To3(UserStackSel, StackOrg, UserCodeSel, loop)
		ELSE
			WriteString("Trap in kernel, err=");  WriteInt(err, 1);
			IF tlpt > 0 THEN WriteChar(0CX) END;	(* LF *)
			LOOP END
		END
	ELSE	(* return to HALT *)
	END
END ExceptionHandler;

(* NMIHandler - Handle non-maskable interrupts (e.g. Dell reset button) *)

PROCEDURE NMIHandler;
VAR bp: LONGINT;
BEGIN
	SYSTEM.GETREG(5, bp);	(* EBP *)
	ShowInt(bp, "NMI in ");
	LOOP END
END NMIHandler;

(* Unexpected - Default interrupt handler *)

PROCEDURE Unexpected;
VAR bp: LONGINT;  int, isr, irr: CHAR;
BEGIN
	SYSTEM.GETREG(5, bp);	(* EBP *)
	SYSTEM.GET(bp+40, int);
	IF (ORD(int) >= IRQ) & (ORD(int) <= IRQ+15) THEN
		IF ORD(int) >= IRQ+8 THEN
			SYSTEM.PORTOUT(IntB0, 0BX);  SYSTEM.PORTIN(IntB0, isr);
			SYSTEM.PORTOUT(IntB0, 0AX);  SYSTEM.PORTIN(IntB0, irr)
		ELSE
			SYSTEM.PORTOUT(IntA0, 0BX);  SYSTEM.PORTIN(IntA0, isr);
			SYSTEM.PORTOUT(IntA0, 0AX);  SYSTEM.PORTIN(IntA0, irr)
		END;
		IF TraceUnexpected THEN
			WriteString("xi");  WriteHex(ORD(int), -2);  
			WriteHex(ORD(isr), -2);  WriteHex(ORD(irr), -2)
			(*WriteString("Unexpected interrupt ");  WriteHex(ORD(int), -2);
			ShowInt(bp, "H in ")*)
		END
	ELSIF int = 0FX THEN	(* probably a spurious interrupt signalled by the APIC *)
		IF TraceUnexpected THEN
			WriteString("xi");  WriteHex(ORD(int), -2);
			(*WriteString("Unexpected interrupt ");  WriteHex(ORD(int), -2);
			ShowInt(bp, "H in ")*)
		END
	ELSE
		HALT(99)	(* unexpected software interrupt *)
	END
END Unexpected;

(** InstallTrap - Install the trap handler *)

PROCEDURE InstallTrap*(p: TrapHandler);
BEGIN
	handler := p
END InstallTrap;

(** InstallLoop - Install procedure to which control is transferred after a trap *)

PROCEDURE InstallLoop*(p: Proc);
BEGIN
	loop := p
END InstallLoop;

(** -- Interrupt handling -- *)

(** InstallIP - Install interrupt handler & enable IRQ if necessary. "p" must be defined as a normal "PROCEDURE p;".  On entry to p interrupts are disabled and may be enabled with SYSTEM.STI().  At exit from p the state of interrupts are restored.  The acknowledgement of a hardware interrupt is done by the Kernel.  No end-of-interrupt has to be signalled explicitly in procedure p.  IRQs are mapped starting at interrupt Kernel.IRQ.

	At entry to procedure p the stack is as follows:
		56	EFLAGS''
		52	CS''
		48	EIP''	(IRETD from glue code)
		44	errorcode
		40	intnum
		36	EAX
		32	ECX
		28	EDX
		24	EBX
		20	ESP' (overwritten)
		16	EBP
		12	ESI
		08	EDI
		04	EIP'	(RET from p)
		00	EBP'	<-- EBP
		xx	locals	<-- ESP

	Interrupt priorities (highest to lowest):
	
	IRQ	Common use
	00	Timer
	01	Keyboard
	08	RT clock
	09	EGA/VGA retrace
	10	Ethernet or other card
	11	card
	12	PS/2 mouse or card
	13	DMA/copro
	14	Hard disk
	15	card or IRQ error
	03	COM2/4
	04	COM1/3
	05	card
	06	Diskette
	07	LPT1 or IRQ error
*)

(*
	The location of [haltcode] depends on the interrupt level, which
	can be recognised from the lower 2 bits of CS''.
	If it was an interrupt from a level 3 to level 0 the stack will be:
	64	SS3
	60	ESP3	--> [haltcode]
	If it was from level 0 to level 0, it will be:
	60	[haltcode]
	If it was from V86 mode, it will be:
	80	GS3
	76	FS3
	72	DS3
	68	ES3
	64	SS3
	60	ESP3
*)

PROCEDURE InstallIP*(p: Proc;  i: INTEGER);
VAR mask: SET; j: SHORTINT;
BEGIN
	IF i = IRQ+2 THEN i := IRQ+9 END;	(* IRQ2 is used for cascading and remapped to IRQ9 *)
	
	IF intHandler[i, 0] = defaultHandler THEN	(* no user-installed handlers yet *)
		intHandler[i, 0] := p	(* overwrite default handler with user's handler *)
	ELSE
		j := 1; WHILE intHandler[i, j] # NIL DO INC(j) END;
		ASSERT(j < MaxInts-1);	(* table entry available *)
		intHandler[i, j+1] := NIL; intHandler[i, j] := p	(* add new handler *)
	END;
	
		(* enable hardware interrupt if required *)
	IF (i >= IRQ+0) & (i <= IRQ+7) THEN	(* controller 1 *)
		SYSTEM.PORTIN(IntA1, SYSTEM.VAL(CHAR, mask));
		EXCL(mask, i-IRQ);
		SYSTEM.PORTOUT(IntA1, SYSTEM.VAL(CHAR, mask))
	ELSIF (i >= IRQ+8) & (i <= IRQ+15) THEN	(* controller 2 *)
		SYSTEM.PORTIN(IntB1, SYSTEM.VAL(CHAR, mask));
		EXCL(mask, i-(IRQ+8));
		SYSTEM.PORTOUT(IntB1, SYSTEM.VAL(CHAR, mask))
	ELSE (* skip *)
	END
END InstallIP;

(** RemoveIP - Uninstall interrupt handler & disable IRQ if necessary *)

PROCEDURE RemoveIP*(p: Proc; i: INTEGER);
VAR mask: SET; j: SHORTINT;
BEGIN
	IF p # NIL THEN	(* uninstall specified handler *)
		j := 0; WHILE (intHandler[i, j] # p) & (intHandler[i, j] # NIL) DO INC(j) END;
		ASSERT(intHandler[i, j] = p)	(* found handler *)
	ELSE	(* wildcard specified *)
		ASSERT(intHandler[i, 1] = NIL);	(* there must be only one handler installed *)
		j := 0	(* remove it *)
	END;
	
	mask := GetFlags(); SYSTEM.CLI();
	REPEAT INC(j); intHandler[i, j-1] := intHandler[i, j] UNTIL intHandler[i, j] = NIL;	(* delete handler *)
	IF intHandler[i, 0] = NIL THEN	(* no handlers left *)
		intHandler[i, 0] := defaultHandler
	ELSE
		i := -1	(* do not mask interrupt below, there are more handlers *)
	END;
	SetFlags(mask);
	
		(* disable hardware interrupt if required *)
	IF (i >= IRQ+0) & (i <= IRQ+7) THEN	(* controller 1 *)
		SYSTEM.PORTIN(IntA1, SYSTEM.VAL(CHAR, mask));
		INCL(mask, i-IRQ);
		SYSTEM.PORTOUT(IntA1, SYSTEM.VAL(CHAR, mask))
	ELSIF (i >= IRQ+8) & (i <= IRQ+15) THEN	(* controller 2 *)
		SYSTEM.PORTIN(IntB1, SYSTEM.VAL(CHAR, mask));
		INCL(mask, i-(IRQ+8));
		SYSTEM.PORTOUT(IntB1, SYSTEM.VAL(CHAR, mask))
	ELSE (* skip *)
	END
END RemoveIP;

(** -- Memory/Object management -- *)

(** Available - Return the size in bytes of the remaining free heap space *)

PROCEDURE Available*(): LONGINT;
	VAR i, avail: LONGINT; ptr: FreeBlockPtr;
BEGIN
	avail := 0; i := 0;
	WHILE i <= N DO
		ptr := SYSTEM.VAL(FreeBlockPtr, A[i]);
		WHILE ptr # NIL DO
			INC(avail, ptr^.size); ptr := SYSTEM.VAL(FreeBlockPtr, ptr^.next)
		END;
		INC(i)
	END;
	RETURN avail
END Available;

(** LargestAvailable - Return the size in bytes of the largest free available memory block. Allocating objects with a size greater than this size will cause the memory allocation to fail. *) 

PROCEDURE LargestAvailable*(): LONGINT;
	VAR i, max: LONGINT; ptr: FreeBlockPtr;
BEGIN
	i := N; max := 0;
	WHILE (i >= 0) & (max = 0) DO
		ptr := SYSTEM.VAL(FreeBlockPtr, A[i]);
		WHILE ptr # NIL DO
			IF ptr^.size > max THEN max := ptr^.size END;
			ptr := SYSTEM.VAL(FreeBlockPtr, ptr^.next)
		END;
		DEC(i)
	END;
	RETURN max
END LargestAvailable;

(** Used - Return the size in bytes of the amount of memory currently in use in the heap. *) 

PROCEDURE Used*(): LONGINT;
BEGIN
	RETURN heapTop - 60000H - Available()
END Used;

PROCEDURE Mark (block: BlockPtr);
	TYPE Tag0 = POINTER TO RECORD (*size,*) ptroff: SET END;
		(* size skipped, because accessed via tag = actual tag + 4 *)
	VAR father, field, currElem: BlockPtr; offset: LONGINT; tag, downtag, marked: Tag0; arraybit: SET;
BEGIN
	SYSTEM.GET(SYSTEM.VAL(ADDRESS, block)-4, tag);
	IF ~(SubObjBit IN SYSTEM.VAL(SET, block)) THEN	(* not a subobject *)
		marked := SYSTEM.VAL(Tag0, SYSTEM.VAL(SET, tag) + mark);
		IF tag # marked THEN
			SYSTEM.PUT(SYSTEM.VAL(ADDRESS, block)-4, marked);
			SYSTEM.GET(SYSTEM.VAL(ADDRESS, SYSTEM.VAL(SET, tag) - array)-4, marked);
			EXCL(SYSTEM.VAL(SET, marked), MarkBit); (* necessary to mask mark bit *)
			SYSTEM.GET(SYSTEM.VAL(ADDRESS, marked)-4, arraybit);
			INCL(arraybit, MarkBit);
			SYSTEM.PUT(SYSTEM.VAL(ADDRESS, marked)-4, arraybit);
			arraybit := SYSTEM.VAL(SET, tag) * array;
			IF arraybit # {} THEN currElem := block^.firstElem;
				tag := SYSTEM.VAL(Tag0, SYSTEM.VAL(SET, tag) - arraybit)
			ELSE currElem := block
			END;
			father := NIL;
			LOOP
				INC(SYSTEM.VAL(ADDRESS, tag), 4);	(* step through rec fields by modifying tag (locally only) *)
				offset := SYSTEM.VAL(LONGINT, tag^.ptroff - {ArrayBit});	(* strip deep copy flag *)
				IF offset < 0 THEN
					INC(SYSTEM.VAL(ADDRESS, tag), offset);
					IF (arraybit # {}) & (currElem # block^.lastElemToMark) THEN
						INC(SYSTEM.VAL(ADDRESS, currElem), SYSTEM.VAL(LONGINT, tag^.ptroff - {ArrayBit}))
					ELSE (* up *)
							(* here the tag is modified to store the state of the traversal.  bits 2- are changed *)
						SYSTEM.PUT(SYSTEM.VAL(ADDRESS, block)-4, SYSTEM.VAL(SET, tag) + arraybit + mark);
						IF father = NIL THEN EXIT END;
						SYSTEM.GET(SYSTEM.VAL(ADDRESS, father)-4, tag);
						arraybit := SYSTEM.VAL(SET, tag) * array;
						tag := SYSTEM.VAL(Tag0, SYSTEM.VAL(SET, tag) - (array + mark));
						IF arraybit # {} THEN currElem := father^.currElem
						ELSE currElem := father
						END;
						offset (*field address*) := SYSTEM.VAL(ADDRESS, currElem) + SYSTEM.VAL(LONGINT, tag^.ptroff - {ArrayBit});
						SYSTEM.GET(offset, field);
						SYSTEM.PUT(offset, block);
						block := father;
						father := field
					END
				ELSE
					offset (*field address*) := SYSTEM.VAL(ADDRESS, currElem) + offset;
					SYSTEM.GET(offset, field);
					IF field # NIL THEN
						SYSTEM.GET(SYSTEM.VAL(ADDRESS, field)-4, downtag);
						IF subobj * SYSTEM.VAL(SET, field) = {} THEN	(* not a subobject *)
							marked := SYSTEM.VAL(Tag0, SYSTEM.VAL(SET, downtag) + mark);
							IF downtag # marked THEN (* down *)
								SYSTEM.PUT(SYSTEM.VAL(ADDRESS, field)-4, marked);
									(* here the tag is modified.  bits 2- are changed *)
								SYSTEM.PUT(SYSTEM.VAL(ADDRESS, block)-4, SYSTEM.VAL(SET, tag) + arraybit + mark);
								IF arraybit # {} THEN block^.currElem:= currElem END;
								SYSTEM.GET(SYSTEM.VAL(ADDRESS, SYSTEM.VAL(SET, downtag) - array)-4, marked);
								EXCL(SYSTEM.VAL(SET, marked), MarkBit); (* necessary to mask mark bit *)
								SYSTEM.GET(SYSTEM.VAL(ADDRESS, marked)-4, arraybit);
								INCL(arraybit, MarkBit);
								SYSTEM.PUT(SYSTEM.VAL(ADDRESS, marked)-4, arraybit);
								arraybit := SYSTEM.VAL(SET, downtag) * array;
								IF arraybit # {} THEN currElem := field^.firstElem ELSE currElem := field END;
								SYSTEM.PUT(offset, father);
								father := block;
								block := field;
								tag := SYSTEM.VAL(Tag0, SYSTEM.VAL(SET, downtag) - arraybit)
							END
						ELSE	(* do not mark subobject *)
							SYSTEM.GET(SYSTEM.VAL(ADDRESS, SYSTEM.VAL(SET, downtag) - array)-4, marked);
							EXCL(SYSTEM.VAL(SET, marked), MarkBit);
							SYSTEM.GET(SYSTEM.VAL(ADDRESS, marked)-4, downtag);
							INCL(SYSTEM.VAL(SET, downtag), MarkBit);
							SYSTEM.PUT(SYSTEM.VAL(ADDRESS, marked)-4, downtag)
						END
					END
				END
			END
		END
	ELSE (* do not mark subobject, subobjects are not traced *)
			(* remove array bit because of array subobjects in module block *)
		SYSTEM.GET(SYSTEM.VAL(ADDRESS, SYSTEM.VAL(SET, tag) - array)-4, tag);
		EXCL(SYSTEM.VAL(SET, tag), MarkBit);
		SYSTEM.GET(SYSTEM.VAL(ADDRESS, tag)-4, arraybit);
		INCL(arraybit, MarkBit);
		SYSTEM.PUT(SYSTEM.VAL(ADDRESS, tag)-4, arraybit)
	END
END Mark;

PROCEDURE CheckFinObjs;
	VAR n: FinObj; tag: LONGINT;
BEGIN
	n := FinObjs;	(* find all checked objects that are marked *)
	WHILE n # NIL DO
		SYSTEM.GET(n.obj - 4, tag);
		n.marked := MarkBit IN SYSTEM.VAL(SET, tag);
		n := n.next
	END;
	n := FinObjs;	(* now mark all objects reachable from the unmarked checked objects *)
		(* they will be freed at the next GC, unless the checked object is ressurrected *)
	WHILE n # NIL DO
		IF ~n.marked THEN Mark(SYSTEM.VAL(BlockPtr, n.obj)) END;
		n := n.next
	END
END CheckFinObjs;

PROCEDURE WriteType(t: ADDRESS);	(* t is tdesc *)
VAR m: LONGINT;  name: ARRAY 32 OF CHAR;
BEGIN
	SYSTEM.GET(t-4, t);
	SYSTEM.GET(t+48, m);
	SYSTEM.MOVE(m+4, SYSTEM.ADR(name[0]), 32);
	WriteString(name);  WriteChar(".");
	SYSTEM.MOVE(t+16, SYSTEM.ADR(name[0]), 32);
	IF name = "" THEN WriteString("<anon>")
	ELSE WriteString(name)
	END
END WriteType;

PROCEDURE Sweep;
	VAR p, end: Blockm4Ptr; lastp: FreeBlockPtr; tag, notmarked, tdesc: Tag; size, lastsize, i: LONGINT;
		lastA: ARRAY N+1 OF ADDRESS;
		t: LONGINT;  live, dead, lsize, ssize: LONGINT;
BEGIN
	i := 0;
	WHILE i <= N DO A[i] := nil; lastA[i] := SYSTEM.ADR(A[i]); INC(i) END;
	p := SYSTEM.VAL(Blockm4Ptr, firstBlock);
	end := SYSTEM.VAL(Blockm4Ptr, endBlock);
	lastsize := 0;  live := 0;  dead := 0;  lsize := 0;
	WHILE p # end DO
		tag := SYSTEM.VAL(Tag, SYSTEM.VAL(SET, p^.tag) - free);
		notmarked := SYSTEM.VAL(Tag, SYSTEM.VAL(SET, tag) - mark);
		tdesc := SYSTEM.VAL(Tag, SYSTEM.VAL(SET, notmarked) - array);
		IF notmarked # tdesc THEN (* array block *) size := p^.lastElemToMark + tdesc^.size - SYSTEM.VAL(ADDRESS, p)
		ELSE size := tdesc^.size + 4
		END;
		ssize := size;
		size := SYSTEM.VAL(LONGINT, SYSTEM.VAL(SET, size + B-1)-SYSTEM.VAL(SET, B-1));
		IF tag = notmarked THEN (* collect *)
			IF traceheap * {3..8, 11} # {} THEN
				IF 11 IN traceheap THEN INC(dead) END;
				IF notmarked # tdesc THEN
					IF 4 IN traceheap THEN
						WriteString("[DA ");  WriteHex(SYSTEM.VAL(LONGINT, p)+4, 8);
						SYSTEM.GET(SYSTEM.ADR(p.lastElemToMark)+12, t);
						WriteChar(" ");  WriteInt(t, 1); WriteChar(" ");  WriteType(SYSTEM.VAL(LONGINT, tdesc));
						WriteString("] ")
					END
				ELSIF SYSTEM.VAL(LONGINT, tdesc) = SYSTEM.VAL(LONGINT, p)+4 THEN
					IF traceheap * {5,8} # {} THEN
						IF FreeBit IN SYSTEM.VAL(SET, p^.tag) THEN
							IF 8 IN traceheap THEN
								WriteString("[FB ");  WriteHex(SYSTEM.VAL(LONGINT, p), 8);
								SYSTEM.GET(SYSTEM.ADR(p.lastElemToMark), t);
								WriteChar(" ");  WriteInt(t, 1);  WriteString("] ")
							END
						ELSE
							IF 5 IN traceheap THEN
								WriteString("[DS ");  WriteHex(SYSTEM.VAL(LONGINT, p), 8);
								SYSTEM.GET(SYSTEM.ADR(p.lastElemToMark), t);
								WriteChar(" ");  WriteInt(t, 1);  WriteString("] ")
							END
						END
					END
				ELSE
					IF 3 IN traceheap THEN
						WriteString("[DR ");  WriteHex(SYSTEM.VAL(LONGINT, p)+4, 8);
						WriteHex(SYSTEM.VAL(LONGINT, tdesc), 9);  WriteChar(" ");
						WriteType(SYSTEM.VAL(LONGINT, tdesc));  WriteString("] ")
					END
				END
			END;
			IF lastsize = 0 THEN lastp := SYSTEM.VAL(FreeBlockPtr, p) END;
			INC(lastsize, size)
		ELSE
			IF (11 IN traceheap) & (SYSTEM.VAL(LONGINT, p) # SYSTEM.VAL(LONGINT, sysres)-28) THEN 
				INC(live); INC(lsize, ssize)
			END;
			p^.tag := notmarked;
			IF lastsize > 0 THEN
				lastp^.size := lastsize - 4;
				lastp^.tag := SYSTEM.VAL(Tag, SYSTEM.VAL(SET, SYSTEM.ADR(lastp^.size)) + free);
				i := lastsize DIV B;
				IF i > N THEN i := N END;
				lastp^.next := nil;
				SYSTEM.PUT(lastA[i], lastp);
				lastA[i] := SYSTEM.ADR(lastp^.next);
				lastsize := 0
			END
		END;
		INC(SYSTEM.VAL(ADDRESS, p), size)
	END;
	(* last collected block: *)
	IF lastsize > 0 THEN
		lastp^.size := lastsize - 4;
		lastp^.tag := SYSTEM.VAL(Tag, SYSTEM.VAL(SET, SYSTEM.ADR(lastp^.size)) + free);
		i := lastsize DIV B;
		IF i > N THEN i := N END;
		lastp^.next := nil;
		SYSTEM.PUT(lastA[i], lastp);
		lastA[i] := SYSTEM.ADR(lastp^.next)
	END;
	IF 11 IN traceheap THEN
		WriteString("[");  WriteInt(live, 1);  WriteString(" live (");
		IF lsize # 0 THEN WriteInt(lsize DIV live, 1) END;
		WriteString("), ");
		WriteInt(dead, 1);  WriteString(" dead]")
	END
END Sweep;

(*
PROCEDURE CheckHeap(msg: ARRAY OF CHAR);	(* check heap consistency *)
VAR tag: Tag;  p, end: Blockm4Ptr; size, i: LONGINT;
BEGIN
	WriteString(msg);  WriteLn;
	p := SYSTEM.VAL(Blockm4Ptr, firstBlock);
	end := SYSTEM.VAL(Blockm4Ptr, endBlock);
	i := 0;
	WHILE p # end DO
		WriteHex(SYSTEM.VAL(LONGINT, p), 9);  WriteChar(".");
		WriteHex(SYSTEM.VAL(LONGINT, p^.tag), 8);
		tag := SYSTEM.VAL(Tag, SYSTEM.VAL(SET, p^.tag) - {MarkBit,ArrayBit,FreeBit});
		ASSERT(tag # NIL);
		IF ArrayBit IN SYSTEM.VAL(SET, p^.tag) THEN
			size := p^.lastElemToMark + tag^.size - SYSTEM.VAL(LONGINT, p)
		ELSE
			size := tag^.size + 4
		END;
		WriteChar(".");  WriteHex(size, 8);
		INC(i);  IF i MOD 3 = 0 THEN WriteLn END;
		size := SYSTEM.VAL(LONGINT, SYSTEM.VAL(SET, size + B-1)-SYSTEM.VAL(SET, B-1));
		INC(SYSTEM.VAL(LONGINT, p), size)
	END
END CheckHeap;
*)

PROCEDURE CheckCandidates;	(* nofcand > 0 *)
	VAR i, j, h, p: LONGINT; block: Blockm4Ptr; tag, notmarked, tdesc: Tag;
BEGIN
	(* first sort them in increasing order using shellsort *)
	h := 1; REPEAT h := h*3 + 1 UNTIL h > nofcand;
	REPEAT h := h DIV 3; i := h;
		WHILE i < nofcand DO p := candidates[i]; j := i;
			WHILE (j >= h) & (candidates[j-h] > p) DO
				candidates[j] := candidates[j-h]; j := j-h;
			END;
			candidates[j] := p; INC(i)
		END
	UNTIL h = 1;
	(* sweep phase *)
	block := SYSTEM.VAL(Blockm4Ptr, firstBlock);
	i := 0; p := candidates[i];
	LOOP
		IF p <= SYSTEM.VAL(ADDRESS, block) + 4 THEN
			IF p = SYSTEM.VAL(ADDRESS, block) + 4 THEN Mark(SYSTEM.VAL(BlockPtr, p)) END;
			INC(i);
			IF i = nofcand THEN EXIT END;
			p := candidates[i]
		ELSIF p <= SYSTEM.VAL(ADDRESS, block) + 28 THEN (* system block *)
			IF p = SYSTEM.VAL(ADDRESS, block) + 28 THEN Mark(SYSTEM.VAL(BlockPtr, SYSTEM.VAL(ADDRESS, block)+4)) END;
			INC(i);
			IF i = nofcand THEN EXIT END;
			p := candidates[i]
		ELSE
			tag := block^.tag;
			notmarked := SYSTEM.VAL(Tag, SYSTEM.VAL(SET, tag) - (mark + free));
			tdesc := SYSTEM.VAL(Tag, SYSTEM.VAL(SET, notmarked) - array);
			IF notmarked # tdesc THEN (* array block *) h := block^.lastElemToMark + tdesc^.size - SYSTEM.VAL(ADDRESS, block)
			ELSE h := tdesc^.size + 4
			END;
			INC(SYSTEM.VAL(ADDRESS, block), SYSTEM.VAL(LONGINT, SYSTEM.VAL(SET, h + B-1)-SYSTEM.VAL(SET, B-1)));
			IF block = SYSTEM.VAL(Blockm4Ptr, endBlock) THEN EXIT END
		END
	END;
	nofcand := 0
END CheckCandidates;

PROCEDURE Candidate(p: LONGINT);
	VAR tag, tag0: LONGINT;
BEGIN
	IF (p > firstBlock) & (p < endBlock) THEN
		IF p MOD B = 0 THEN
			SYSTEM.GET(p-4, tag);
			IF tag MOD 8 IN {0, 2} THEN	(* array or record, not yet marked, not free *)
				candidates[nofcand] := p; INC(nofcand);
				IF nofcand = LEN(candidates) THEN CheckCandidates END
			END
		ELSIF p MOD 16 = 8 THEN	(* subobj *)
			SYSTEM.GET(p-4, tag);
			IF (tag MOD B = 0) & (tag > firstBlock) & (tag < endBlock) THEN
				SYSTEM.GET(tag-4, tag0);
				IF tag0 = tag THEN
					candidates[nofcand] := p; INC(nofcand);
					IF nofcand = LEN(candidates) THEN CheckCandidates END
				END
			END
		ELSE (* skip *)
		END
	END
END Candidate;
		
PROCEDURE FinalizeObjs;
	VAR n, prev: FinObj;
BEGIN
	n := FinObjs;
	WHILE n # NIL DO
		IF ~n.marked THEN
			IF n = FinObjs THEN FinObjs := FinObjs.next ELSE prev.next := n.next END;
			IF 6 IN traceheap THEN
				WriteString("[Fin");  WriteHex(n.obj, 9);  WriteString("] ")
			END;
			n.fin(SYSTEM.VAL(SYSTEM.PTR, n.obj))	(* may ressurrect checked object *)
		ELSE prev := n
		END;
		n := n.next
	END
END FinalizeObjs;

(** GC - Immediately activate the garbage collector. *)

PROCEDURE ^NewSys(VAR p: ADDRESS; size: LONGINT);

PROCEDURE GC*;
	VAR m: Module; i, p, sp, ptradr: LONGINT; ptr: BlockPtr;  avail, largest, t0, tf0, tf1, st: LONGINT;
BEGIN
	IF AlwaysGC OR (GClevel > 0) THEN
		inGC := TRUE;
		IF traceheap * {0..15} # {} THEN
			WriteString("(GC");
			IF 10 IN traceheap THEN avail := Available();  largest := LargestAvailable() END;
			st := ticks
		END;
		m := modules;	(* ModuleDesc and ModuleBlock are marked via Kernel.modules *)
		WHILE m # NIL DO
			Mark(SYSTEM.VAL(BlockPtr, m));	(* the code, data etc. is marked via fields of m *)
			i := LEN(m.ptrTab); IF i > 0 THEN ptradr := SYSTEM.ADR(m.ptrTab[0]) END; 
			WHILE i > 0 DO
				SYSTEM.GET(ptradr, p); SYSTEM.GET(p, ptr);
				IF ptr # NIL THEN Mark(ptr) END;
				DEC(i); INC(ptradr, 4);
			END;
			i := LEN(m.tdescs); IF i > 0 THEN ptradr := SYSTEM.ADR(m.tdescs[0]) END; 
			WHILE i > 0 DO
				SYSTEM.GET(ptradr, ptr); Mark(ptr);
				DEC(i); INC(ptradr, 4);
			END;
			m := m^.next;
		END;
			(* check stack *)
		IF GCstack THEN
			SYSTEM.GETREG(4, sp);
			ASSERT(sp <= StackOrg);
			nofcand := 0;
			WHILE sp < StackOrg DO
				SYSTEM.GET(sp, p); Candidate(p); INC(sp, 4)
			END;
			IF nofcand > 0 THEN CheckCandidates END
		END;
		CheckFinObjs;
		Sweep;
		IF (reserve = NIL) & firstTry THEN
			IF LargestAvailable() >= ReserveSize THEN NewSys(p, ReserveSize); reserve := SYSTEM.VAL(BlockPtr, p) END
		END;
		IF 7 IN traceheap THEN
			t0 := SYSTEM.ADR(A[N]);  WriteString("[FP");
			LOOP
				SYSTEM.GET(t0, t0);
				IF t0 = 0 THEN EXIT END;
				tf0 := t0+12;  SYSTEM.GET(t0+4, tf1);  tf1 := tf1-8+tf0;
				INC(tf0, (-tf0) MOD 1000H);  DEC(tf1, tf1 MOD 1000H);
				IF tf0 < tf1 THEN WriteHex(tf0, 9);  WriteChar(" ");  WriteInt((tf1-tf0) DIV 1000H, 1) END;
				t0 := t0+8
			END;
			WriteString(" PF] ")
		END;
		inGC := FALSE;
		FinalizeObjs;
		IF traceheap * {0..15} # {} THEN
			st := ticks-st;
			IF 10 IN traceheap THEN
				avail := Available()-avail;  largest := LargestAvailable()-largest;
				WriteInt(avail, 1);  WriteString(" +");  WriteInt(largest, 1)
			END;
			WriteChar(" ");  WriteInt(st*1000 DIV TimeUnit, 1);  WriteString("ms");
			WriteString(" CG)"); WriteLn
		END
	END;
	IF break THEN
		break := FALSE;
		SYSTEM.HALT(13)	(* Keyboard interrupt *)
	END
END GC;

PROCEDURE NewBlock (size: LONGINT): InitPtr;	(* size MOD B = 0 *)
	VAR i, rest: LONGINT; adr, AN: ADDRESS; ptr: InitPtr; restptr: FreeBlockPtr;
BEGIN
	IF break THEN break := FALSE; SYSTEM.HALT(13) END;
	IF size < 0 (* NEW(p, MAX(LONGINT)) *) THEN SYSTEM.HALT(14) END;
	i := size DIV B;
	IF i > N THEN i := N END;
	adr := SYSTEM.ADR(A[0]) + 4*i;
	AN := SYSTEM.ADR(A[N]);	(* constant register *)
	LOOP
		SYSTEM.GET(adr, ptr);
		IF adr = AN THEN
			LOOP
				IF ptr = NIL THEN
					IF (* (TrapHandlingLevel = 0) & *) firstTry THEN GC;
						firstTry := FALSE; ptr := NewBlock(size); firstTry := TRUE;
						RETURN ptr
					ELSE reserve := NIL; GC; firstTry := TRUE; SYSTEM.HALT(14)
					END
				END;
				IF ptr^.z0 + 4 >= size THEN EXIT END;
				adr := SYSTEM.ADR(ptr^.z1); SYSTEM.GET(adr, ptr)
			END;
			EXIT
		END;
		IF ptr # NIL THEN EXIT END;
		INC(adr, 4)
	END;
	(* ptr # NIL *)
	SYSTEM.PUT(adr, ptr^.z1);
	rest := ptr^.z0 + 4 - size;
	restptr := SYSTEM.VAL(FreeBlockPtr, SYSTEM.VAL(ADDRESS, ptr) + size);
	IF rest > 0 THEN	(* >= B >= 16 *)
		i := rest DIV B;
		IF i > N THEN i := N END;
		restptr^.tag := SYSTEM.VAL(Tag, SYSTEM.VAL(SET, SYSTEM.ADR(restptr^.size)) + free);
		restptr^.size := rest - 4;
		restptr^.next := A[i]; A[i] := SYSTEM.VAL(ADDRESS, restptr)
	END;
	RETURN ptr
END NewBlock;

(* NewRec - Implementation of NEW.  Do not call directly. *)

PROCEDURE NewRec(VAR p: ADDRESS; tag: Tag);	(* implementation of NEW(ptr) *)
	VAR size: LONGINT; ptr, init: InitPtr; 
BEGIN (* tag^.size = rectyp^.size *)
	ASSERT(SYSTEM.VAL(LONGINT, tag) MOD 8 = 0);	(* no Oberon-X flags allowed *)
	IF tag = NIL THEN tag := SYSTEM.VAL(Tag, ptrElemTag) END;	(* descriptor for ARRAY * OF *)
	size := SYSTEM.VAL(LONGINT, SYSTEM.VAL(SET, tag^.size + (4 (*tag*) + B-1))-SYSTEM.VAL(SET, B-1));
	ptr := NewBlock(size);
	IF 0 IN traceheap THEN
		WriteString("[NR ");  WriteType(SYSTEM.VAL(LONGINT, tag));
	END;

	init := SYSTEM.VAL(InitPtr, SYSTEM.VAL(ADDRESS, ptr) + size - 32);
	init^.z0 := 0; init^.z1 := 0; init^.z2 := 0; init^.z3 := 0; init^.z4 := 0; init^.z5 := 0; init^.z6 := 0;
	WHILE init # ptr DO
		DEC(SYSTEM.VAL(ADDRESS, init), 32);
		init^.z0 := 0; init^.z1 := 0; init^.z2 := 0; init^.z3 := 0; init^.z4 := 0; init^.z5 := 0; init^.z6 := 0; init^.z7 := 0
	END;
	ptr^.tag := tag;
	p := SYSTEM.VAL(ADDRESS, ptr) + 4;
	IF 0 IN traceheap THEN
		WriteHex(SYSTEM.VAL(LONGINT, p), 9);  WriteString("] ")
	END
END NewRec;

(* NewSys - Implementation of SYSTEM.NEW.  Do not call directly. *)

PROCEDURE NewSys(VAR p: ADDRESS; size: LONGINT);
VAR ptr, init: InitPtr;  (*bp: LONGINT;*)
BEGIN
	IF 2 IN traceheap THEN
		WriteString("[NS ");  WriteInt(size, 1)
	END;
	size := SYSTEM.VAL(LONGINT, SYSTEM.VAL(SET, size + (28 + B-1))-SYSTEM.VAL(SET, B-1));
	ptr := NewBlock(size);
	init := SYSTEM.VAL(InitPtr, SYSTEM.VAL(ADDRESS, ptr) + size - 32);
	WHILE init # ptr DO
		init^.tag := NIL; init^.z0 := 0; init^.z1 := 0; init^.z2 := 0; init^.z3 := 0; init^.z4 := 0; init^.z5 := 0; init^.z6 := 0;
		DEC(SYSTEM.VAL(ADDRESS, init), 32);
	END;
	ptr^.tag := SYSTEM.VAL(Tag, SYSTEM.ADR(ptr^.z0));
	ptr^.z0 := size - 4;
	ptr^.z1 := -4;
	init^.z2 := 0;
	init^.z3 := 0;
	(*SYSTEM.GETREG(5, bp);
	SYSTEM.GET(bp+4, init^.z3);*)	(* save call address for debugging *)
	init^.z4 := 0;
	ptr^.z5 := SYSTEM.ADR(ptr^.z0);
	init^.z6 := 0;
	p := SYSTEM.VAL(ADDRESS, ptr) + 28;
	IF 2 IN traceheap THEN
		WriteHex(SYSTEM.VAL(LONGINT, p), 9);  WriteString("] ")
	END
END NewSys;

(* NewArr - Implementation of NEW.  Do not call directly. *)

PROCEDURE NewArr(VAR p: ADDRESS; eltag: Tag; nofelem, nofdim: LONGINT);
VAR size, firstElem, elSize, arrSize, vectSize: LONGINT; ptr, init: InitPtr; 
BEGIN
	ASSERT(SYSTEM.VAL(LONGINT, eltag) MOD 4 = 0);
	IF 1 IN traceheap THEN
		WriteString("[NA ");  WriteInt(nofelem, 1);  WriteChar(" ");  WriteInt(nofdim, 1)
	END;
	IF eltag = NIL THEN
		eltag := SYSTEM.VAL(Tag, ptrElemTag)	(* array of pointer to *)
	END;
	elSize := eltag^.size;
	arrSize := nofelem*elSize;
	IF arrSize=0 THEN NewSys(p, nofdim*4+12); RETURN END;
	ASSERT(arrSize > 0);
	vectSize := SYSTEM.VAL(LONGINT, SYSTEM.VAL(SET, 4*nofdim) + SYSTEM.VAL(SET, 4));	(* -> ADR(firstElem) MOD 8 = 0 *)
	IF eltag^.ptroff = -4 THEN (* no pointers in element type *)
		NewSys(p, arrSize + vectSize + 12);
		SYSTEM.PUT(p, SYSTEM.VAL(SET, arrSize + vectSize+15)-SYSTEM.VAL(SET, 3));
		RETURN
	END;
	size := SYSTEM.VAL(LONGINT, SYSTEM.VAL(SET, arrSize + vectSize + (16 + B-1))-SYSTEM.VAL(SET, B-1));
	ptr := NewBlock(size);
	init := SYSTEM.VAL(InitPtr, SYSTEM.VAL(ADDRESS, ptr) + size - 32);
	WHILE init # ptr DO
		init^.tag := NIL; init^.z0 := 0; init^.z1 := 0; init^.z2 := 0; init^.z3 := 0; init^.z4 := 0; init^.z5 := 0; init^.z6 := 0;
		DEC(SYSTEM.VAL(ADDRESS, init), 32);
	END;
	ptr^.tag := SYSTEM.VAL(Tag, SYSTEM.VAL(SET, eltag) + array);
	firstElem := SYSTEM.ADR(ptr^.z3) + vectSize;
	ptr^.z0 := firstElem + arrSize - elSize;
	(* ptr^.z1 is reserved for mark phase *)
	ptr^.z2 := firstElem;
	ptr^.z3 := 0; ptr^.z4 := 0; ptr^.z5 := 0; ptr^.z6 := 0;
	p := SYSTEM.VAL(ADDRESS, ptr) + 4;
	IF 1 IN traceheap THEN
		WriteHex(SYSTEM.VAL(LONGINT, p), 9);  WriteString("] ")
	END
END NewArr;

(** NewDMA - Allocate memory correctly aligned for byte or word DMA in device drivers. "adr" returns the virtual and "phys" the physical address of the memory.  Max size=64k. *)

PROCEDURE NewDMA*(size: LONGINT;  VAR adr, phys: ADDRESS);
VAR p, s, min, prev: ADDRESS;
BEGIN
	IF size >= 0 THEN	(* allocate *)
		IF 9 IN traceheap THEN
			WriteString("[ND ");  WriteInt(size, 1)
		END;
		INC(size, (-size) MOD 4);	(* size MOD 4 = 0 *)
		ASSERT((size > 0) & (size <= 10000H));
		prev := SYSTEM.ADR(dmafree);  min := MAX(LONGINT);
		LOOP
			SYSTEM.GET(prev, p);
			IF p = 0 THEN EXIT END;
			SYSTEM.GET(p, s);
			IF (s >= size) & (s < min) THEN adr := p;  phys := prev;  min := s END;	(* smallest fit so far *)
			prev := p+4
		END;
		IF min = MAX(LONGINT) THEN SYSTEM.HALT(14) END;
		SYSTEM.GET(adr, s);
		IF s = size THEN
			SYSTEM.GET(adr+4, p);  SYSTEM.PUT(phys, p)	(* unlink block *)
		ELSE
			DEC(s, size);  SYSTEM.PUT(adr, s);
			INC(adr, s)
		END;
		phys := adr;
		IF 9 IN traceheap THEN
			WriteHex(adr, 9);  WriteString("] ")
		END
	ELSE	(* deallocate *)
		DisposeDMA(-size, adr)	(* backward compatability *)
	END
END NewDMA;

(** DisposeDMA - Deallocate memory allocated with NewDMA.  "adr" is virtual address. *)

PROCEDURE DisposeDMA*(size: LONGINT;  adr: ADDRESS);
VAR p, s, prev: ADDRESS;
BEGIN
	IF 9 IN traceheap THEN
		WriteString("[ND ");  WriteInt(size, 1);  WriteHex(adr, 9);  WriteString("] ")
	END;
	INC(size, (-size) MOD 4);	(* size MOD 4 = 0 *)
	ASSERT((size > 0) & (size <= 10000H) & (adr >= dma0) & (adr+size <= dma1));
	prev := SYSTEM.ADR(dmafree);
	LOOP
		SYSTEM.GET(prev, p);
		IF p = 0 THEN EXIT END;
		SYSTEM.GET(p, s);
		IF (p+s = adr) & (adr MOD 10000H # 0) THEN	(* merge with previous block *)
			INC(s, size);  SYSTEM.PUT(p, s);
			RETURN
		ELSIF (adr+size = p) & (p MOD 10000H # 0) THEN	(* merge with next block *)
			SYSTEM.PUT(adr, size+s);  SYSTEM.GET(p+4, p);
			SYSTEM.PUT(adr+4, p);  SYSTEM.PUT(prev, adr);
			RETURN
		END;
		prev := p+4
	END;
	SYSTEM.PUT(adr, size);  SYSTEM.PUT(adr+4, dmafree);
	dmafree := adr
END DisposeDMA;

(* AllocatePage - Allocate a page. *)

PROCEDURE AllocatePage(VAR p: ADDRESS);
BEGIN
	IF pageheap = pageheap1 THEN
		WriteString("Kernel.AllocatePage: Out of pages");  WriteLn;	(* set PageHeap *)
		HALT(99)
	END;
	DEC(pageheap, PS);  p := pageheap
END AllocatePage;

(* MapPage - Map a page in the virtual address space *)

PROCEDURE MapPage(pd: ADDRESS;  virt, phys: LONGINT);
CONST R = 400000H;	(* 4M region *)
VAR i, t: LONGINT;  pt: ADDRESS;
BEGIN
	ASSERT(virt MOD PS = 0);  ASSERT(pd MOD PS = 0);
	i := virt DIV R MOD PTEs;
	SYSTEM.GET(pd + 4*i, t);
	IF ODD(t) THEN	(* pt present *) 
		pt := t - t MOD PS
	ELSE
		AllocatePage(pt);
		Fill4(pt, PTEs, SYSTEM.VAL(LONGINT, PageNotPresent));
		SYSTEM.PUT(pd + 4*i, pt + NormalPage)
	END;
	SYSTEM.PUT(pt + 4*(virt DIV PS MOD PTEs), phys)
END MapPage;

(* MappedPage - Return mapped page address *)

PROCEDURE MappedPage(pd: ADDRESS;  virt: LONGINT): LONGINT;
CONST R = 400000H;	(* 4M region *)
VAR i, t: LONGINT;  pt: ADDRESS;
BEGIN
	ASSERT(virt MOD PS = 0);  ASSERT(pd MOD PS = 0);
	i := virt DIV R MOD PTEs;
	SYSTEM.GET(pd + 4*i, t);
	IF ODD(t) THEN	(* pt present *) 
		pt := t - t MOD PS;
		SYSTEM.GET(pt + 4*(virt DIV PS MOD PTEs), t);
		RETURN t
	ELSE
		RETURN 0
	END
END MappedPage;

(* MapMem - Map area [virtAdr..virtAdr+size) directly to area [Adr(phys)..Adr(phys)+size). *)

PROCEDURE MapMem(pd: ADDRESS;  virtAdr, size, phys: LONGINT);
BEGIN
	ASSERT((virtAdr MOD PS = 0) & (size MOD PS = 0));
	WHILE size # 0 DO
		IF ODD(MappedPage(pd, virtAdr)) THEN HALT(99) END;	(* already mapped *)
		MapPage(pd, virtAdr, phys);
		INC(virtAdr, PS);  INC(phys, PS);  DEC(size, PS)
	END
END MapMem;

(** MapPhysical - Map a physical memory area into the virtual address space. {physAdr, size MOD PS = 0} *)

PROCEDURE MapPhysical*(physAdr, size: LONGINT;  VAR virtAdr: LONGINT);
VAR i, free: LONGINT;
BEGIN
	ASSERT((physAdr MOD PS = 0) & (size MOD PS = 0));
	IF (physAdr >= 0A0000H) & (physAdr+size <= 100000H) THEN	(* already mapped directly *)
		virtAdr := physAdr
	ELSE
		IF MapCacheSize # 0 THEN
			free := -1;
			FOR i := 0 TO MapCacheSize-1 DO
				IF mapcache[i].size # 0 THEN
					IF (physAdr = mapcache[i].physAdr) & (size = mapcache[i].size) THEN
						virtAdr := mapcache[i].virtAdr;
						RETURN
					END
				ELSE
					IF free = -1 THEN free := i END
				END
			END
		END;
		virtAdr := mapPtr-size;	(* allocate *)
		MapMem(kernelpd, virtAdr, size, physAdr + NormalPage);
		mapPtr := virtAdr;	(* commit *)
		IF (MapCacheSize # 0) & (free # -1) THEN
			mapcache[free].physAdr := physAdr; mapcache[free].size := size;
			mapcache[free].virtAdr := virtAdr
		END
	END
END MapPhysical;

(*
(** DisableTracing - Disable the GC tracing of a record field of the specified type.  For internal use only. *)

PROCEDURE DisableTracing*(recadr, fldadr: ADDRESS);
VAR t, i, j, ofs: ADDRESS;  n: LONGINT;
BEGIN
	SYSTEM.GET(recadr-4, t);
	n := 0;  i := t+4;  SYSTEM.GET(i, ofs);  j := 0;
	WHILE ofs >= 0 DO
		IF ofs = fldadr-recadr THEN j := i END;
		INC(n);  INC(i, 4);  SYSTEM.GET(i, ofs)
	END;
	ASSERT((ofs = -4*(n+1)) & (j # 0));
	SYSTEM.GET(j+4, ofs);
	WHILE ofs >= 0 DO
		SYSTEM.PUT(j, ofs);  INC(j, 4);  SYSTEM.GET(j+4, ofs)
	END;
	SYSTEM.PUT(j, -4*n)
END DisableTracing;
*)

(** RegisterObject - Register an object (POINTER TO RECORD or POINTER TO ARRAY, not SysBlk) for finalization.  Never perform upcalls in the fin procedure! basic = FALSE *)

PROCEDURE RegisterObject* (obj: SYSTEM.PTR;  fin: Finalizer;  basic: BOOLEAN);
	VAR n: FinObj;
BEGIN
	IF 6 IN traceheap THEN
		WriteString("[RegObj");  WriteHex(SYSTEM.VAL(LONGINT, obj), 9);  WriteString("] ")
	END;
	ASSERT(obj # NIL);
	NEW(n); n.next := FinObjs; n.obj := SYSTEM.VAL(LONGINT, obj); n.marked := FALSE; n.fin := fin;
	FinObjs := n
END RegisterObject;

(** InstallTermHandler - Install a procedure to execute when a module is freed.  Normally used to uninstall interrupt handlers or tasks and perform other cleanup duties.  Never perform upcalls in proc, because the upcalled module will no longer be in memory! *)

PROCEDURE InstallTermHandler* (h: Proc);	(* can also be used by modules below Modules, e.g. FileDir *)
	VAR codebase, handlerAdr: LONGINT; m: Module; found: BOOLEAN;
BEGIN
	m := modules; handlerAdr := SYSTEM.VAL(LONGINT, h); found := FALSE;
	WHILE (m # NIL) & ~found DO
		codebase := SYSTEM.ADR(m.code[0]);
		IF (codebase <= handlerAdr) & (handlerAdr <= codebase + LEN(m.code)) THEN found := TRUE
		ELSE m := m.next
		END
	END;
	IF found THEN m.term := h END;
END InstallTermHandler;

(** FinalizeModule - Execute all object finalizers registered by a module, and then execute its termination handler. *)

PROCEDURE FinalizeModule*(m: Module);
VAR h: Proc;  beg, end: LONGINT;  n, prev: FinObj;
BEGIN
	beg := SYSTEM.ADR(m.code[0]);  end := beg + LEN(m.code);
	n := FinObjs;
	WHILE n # NIL DO
		IF (beg <= SYSTEM.VAL(LONGINT, n.fin)) & (SYSTEM.VAL(LONGINT, n.fin) <= end) THEN
			IF n = FinObjs THEN FinObjs := FinObjs.next ELSE prev.next := n.next END;
			IF 6 IN traceheap THEN
				WriteString("[Fin");  WriteHex(n.obj, 9);  WriteString("] ")
			END;
			n.fin(SYSTEM.VAL(SYSTEM.PTR, n.obj))
		ELSE prev := n
		END;
		n := n.next
	END;
	IF m.term # NIL THEN
		IF 16 IN traceheap THEN
			WriteString("Calling TermHandler ");  WriteString(m.name);  WriteLn
		END;
		h := m.term; m.term := NIL; h
	END
END FinalizeModule;

(* StrToInt - Convert a string to an integer *)

PROCEDURE StrToInt(s: ARRAY OF CHAR): LONGINT;
VAR i, j: SHORTINT;  v, sgn, m: LONGINT;
BEGIN
	j := 0;  WHILE s[j] # 0X DO INC(j) END;
	IF (j > 0) & (CAP(s[j-1]) = "H") THEN m := 16; DEC(j) ELSE m := 10 END;
	v := 0;  i := 0;
	IF s[i] = "-" THEN sgn := -1; INC(i) ELSE sgn := 1 END;
	WHILE i < j DO
		IF (s[i] >= "0") & (s[i] <= "9") THEN v := v*m + (ORD(s[i])-ORD("0"))
		ELSIF (CAP(s[i]) >= "A") & (CAP(s[i]) <= "F") THEN v := v*m + (ORD(CAP(s[i]))-ORD("A")+10)
		ELSE sgn := 0;  j := i
		END;
		INC(i)
	END;
	RETURN sgn*v
END StrToInt;

(** -- Clock/Timer -- *)

(* RT clock driver from Gneiss *)

(* GetCMOS - Read a byte from CMOS setup memory *)

PROCEDURE GetCMOS(i: SHORTINT): INTEGER;
VAR c: CHAR;
BEGIN
	INCL(SYSTEM.VAL(SET, i), 7);  SYSTEM.CLI();
	SYSTEM.PORTOUT(70H, i);  Wait;  SYSTEM.PORTIN(71H, c);
	SYSTEM.PORTOUT(70H, 0X);  SYSTEM.STI();
	RETURN ORD(c)
END GetCMOS;

(* PutCMOS - Write a byte to CMOS setup memory *)

PROCEDURE PutCMOS(i: SHORTINT;  val: CHAR);
BEGIN
	INCL(SYSTEM.VAL(SET, i), 7);  SYSTEM.CLI();
	SYSTEM.PORTOUT(70H, i);  Wait;  SYSTEM.PORTOUT(71H, val);
	SYSTEM.PORTOUT(70H, 0X);  SYSTEM.STI()
END PutCMOS;

(* ReadClock - Read the RT clock. *)

PROCEDURE ReadClock;
BEGIN
	second := GetCMOS(0);  minute := GetCMOS(2);  hour := GetCMOS(4);
	day := GetCMOS(7);  month := GetCMOS(8);  year := GetCMOS(9)
END ReadClock;

(* ClockHandler - Handle periodic clock interrupt *)

PROCEDURE ClockHandler;
BEGIN
	SYSTEM.STI();	(* enable interrupts, we have a second to read the clock! *)
	INC(clockints);
	IF clockmode = 0 THEN
		IF 4 IN SYSTEM.VAL(SET, LONG(GetCMOS(0CH))) THEN ReadClock
		ELSE INC(clockmissed)
		END
	ELSE
		ReadClock
	END
END ClockHandler;

(* InitClock - Initialise clock *)

PROCEDURE InitClock;
CONST Delay = 3;
VAR t: LONGINT;  s: ARRAY 8 OF CHAR;
BEGIN
	second := -1;
	GetConfig("ClockMode", s);
	clockmode := StrToInt(s);	(* mode 0 - interrupt with test, mode 1 - poll, mode -1 - interrupt without test *)
	IF clockmode # 1 THEN
		InstallIP(ClockHandler, IRQ+8);
		PutCMOS(0BH, 12X);	(* 24 hour mode & 1 second interrupt *)
		t := ticks;
		REPEAT UNTIL (second # -1) OR (ticks - t > Delay*TimeUnit)	(* wait for first update *)
	END;
	IF second = -1 THEN	(* clock interrupt not functioning *)
		second := 0;  minute := 0;  hour := 0;  day := 0;  month := 0;  year := 0;
		IF clockmode # 1 THEN
			RemoveIP(ClockHandler, IRQ+8)
		END;
		clockmode := 1	(* poll *)
	END
END InitClock;

(* BCD2/ToBCD - Decode/Encode 2 BCD digits *)

PROCEDURE BCD2(x: INTEGER): LONGINT;
BEGIN
	RETURN (x DIV 16) * 10 + x MOD 16
END BCD2;

PROCEDURE ToBCD(x: LONGINT): INTEGER;
BEGIN
	RETURN SHORT((x DIV 10) * 16 + x MOD 10)
END ToBCD;

(** GetClock - Return current time and date *)

PROCEDURE GetClock*(VAR time, date: LONGINT);
VAR h, m, s, y, o, d: INTEGER;
BEGIN
	IF clockmode = 1 THEN	(* poll *)
		SYSTEM.CLI();	(* ref p. 750 of undocumented pc *)
		REPEAT	(* until same clock value read twice in a row *)
			REPEAT UNTIL ~(7 IN SYSTEM.VAL(SET, LONG(GetCMOS(0AH))));	(* no update in progress *)
			ReadClock;
			h := hour;  m := minute;  s := second;  y := year;  o := month;  d := day;
			ReadClock
		UNTIL (h = hour) & (m = minute) & (s = second) & (y = year) & (o = month) & (d = day);
		SYSTEM.STI()
	ELSE
		SYSTEM.CLI();
		h := hour;  m := minute;  s := second;  y := year;  o := month;  d := day;
		SYSTEM.STI()
	END;
	time := ASH(ASH(BCD2(h), 6) + BCD2(m), 6) + BCD2(s);
	y := SHORT(BCD2(y));	(* returns 0..99, or possibly 0..165 on "fixed" clocks *)
	IF y < 90 THEN INC(y, 100) END;	(* if 0..89, modify it to 100..189 *)
	(* now y is 90..189, i.e. 1990..2089 *)
	(* clients that use year MOD 128 will wrap from 2027 to 1900 *)
	date := ASH(ASH(y, 4) + BCD2(o), 5) + BCD2(d)
END GetClock;

(** SetClock - Set current time and date *)

PROCEDURE SetClock*(time, date: LONGINT);
BEGIN
	PutCMOS(0BH, 82X);	(* disable clock & interrupt *)
	second := ToBCD(time MOD 64);  time := ASH(time, -6);
	minute := ToBCD(time MOD 64);  hour := ToBCD(ASH(time, -6));
	day := ToBCD(date MOD 32);  date := ASH(date, -5);
	month := ToBCD(date MOD 16);  year := ToBCD(ASH(date, -4) MOD 100);
	PutCMOS(0, CHR(second));  PutCMOS(2, CHR(minute));  PutCMOS(4, CHR(hour));
	PutCMOS(7, CHR(day));  PutCMOS(8, CHR(month));  PutCMOS(9, CHR(year));
	PutCMOS(0BH, 12X)	(* 24 hour mode & 1 second interrupt *)
END SetClock;

(** GetTimer - Return "ticks" since initialisation (Kernel.TimeUnit ticks per second) *)

PROCEDURE GetTimer*(): LONGINT;
BEGIN
	RETURN ticks
END GetTimer;

(** SetTimer - Set timer to expire in approximately "ms" milliseconds. *)

PROCEDURE SetTimer*(VAR t: MilliTimer;  ms: LONGINT);
BEGIN
	IF TimeUnit # 1000 THEN	(* convert to ticks *)
		ASSERT((ms >= 0) & (ms <= MAX(LONGINT) DIV TimeUnit));
		ms := ms * TimeUnit DIV 1000
	END;
	IF ms < 5 THEN INC(ms) END;	(* Nyquist adjustment *)
	t.target := ticks + ms	(* overflow checking must be off *)
END SetTimer;

(** Expired - Test if a timer has expired.  Interrupts must be on. *)

PROCEDURE Expired*(VAR t: MilliTimer): BOOLEAN;
BEGIN
	RETURN ticks - t.target >= 0	(* overflow checking must be off *)
END Expired;

(** -- Miscellaneous -- *)

PROCEDURE Cap(ch: CHAR): CHAR;
BEGIN
	IF (ch >= "a") & (ch <= "z") THEN RETURN CAP(ch)
	ELSE RETURN ch
	END
END Cap;

(** GetConfig - Return value of configuration string.  Returns empty val if name not found. *)

PROCEDURE GetConfig*(name: ARRAY OF CHAR;  VAR val: ARRAY OF CHAR);
VAR i, src: LONGINT;  ch: CHAR;
BEGIN
	IF name[0] = "=" THEN
		HALT(99)	(* no longer supported, use GetInit or GetLog instead *)
	ELSE
		src := configadr;
		IF src # 0 THEN
			i := 0;  WHILE name[i] # 0X DO name[i] := Cap(name[i]);  INC(i) END;
			LOOP
				SYSTEM.GET(src, ch);
				IF ch = 0X THEN EXIT END;
				i := 0;
				LOOP
					SYSTEM.GET(src, ch);
					IF (Cap(ch) # name[i]) OR (name[i] = 0X) THEN EXIT END;
					INC(i);  INC(src)
				END;
				IF (ch = 0X) & (name[i] = 0X) THEN	(* found: (src^ = 0X) & (name[i] = 0X) *)
					i := 0;
					REPEAT
						INC(src); SYSTEM.GET(src, ch); val[i] := ch; INC(i);
						IF i = LEN(val) THEN val[0] := 0X; RETURN END	(* val too short *)
					UNTIL ch = 0X;
					val[i] := 0X; RETURN
				ELSE
					WHILE ch # 0X DO	(* skip to end of name *)
						INC(src); SYSTEM.GET(src, ch)
					END;
					INC(src);
					REPEAT	(* skip to end of value *)
						SYSTEM.GET(src, ch);  INC(src)
					UNTIL ch = 0X
				END
			END
		END;
		val[0] := 0X
	END
END GetConfig;

(** GetLog - Return characters from trace log. *)

PROCEDURE GetLog*(VAR val: ARRAY OF CHAR);
VAR i, max: LONGINT;  pass: BOOLEAN;
BEGIN
	i := 0;
	IF traceBufAdr # 0 THEN
		max := LEN(val)-1; pass := FALSE;
		WHILE (i < max) & (traceHead # traceTail) DO
			SYSTEM.GET(traceBufAdr + traceHead, val[i]);
			IF traceHead = traceMark THEN pass := TRUE END;
			traceHead := (traceHead+1) MOD traceBufSize;
			INC(i)
		END;
		IF pass THEN traceMark := traceHead END
	END;
	val[i] := 0X
END GetLog;

(** SetLogMark - Set a "mark" in the log. *)

PROCEDURE SetLogMark*;
BEGIN
	traceMark := traceTail
END SetLogMark;

(** GetMarkedLog - Retrieve the section of the log between the last "mark" and the end of the log, moving the mark along. *)

PROCEDURE GetMarkedLog*(VAR val: ARRAY OF CHAR);
VAR i, max: LONGINT;
BEGIN
	i := 0;
	IF traceBufAdr # 0 THEN
		max := LEN(val)-1;
		WHILE (i < max) & (traceMark # traceTail) DO
			SYSTEM.GET(traceBufAdr + traceMark, val[i]);
			traceMark := (traceMark+1) MOD traceBufSize;
			INC(i)
		END
	END;
	val[i] := 0X
END GetMarkedLog;

(** GetInit - Return parameter values from Init string. *)

PROCEDURE GetInit*(n: LONGINT;  VAR val: LONGINT);
BEGIN
	val := kpar[n]
END GetInit;

PROCEDURE DefaultEnableGC;
BEGIN
	INC(GClevel)
END DefaultEnableGC;

PROCEDURE DefaultDisableGC;
BEGIN
	DEC(GClevel)
END DefaultDisableGC;

PROCEDURE InitHeap;	(* heapTop, pageheap and modules already initialized *)
CONST LinkAdr = 1000H;
VAR
	pt, i, t, stacksize, size, dmasize: LONGINT;  f: FreeBlockPtr;  m: Module;  set: SET;
	td: POINTER TO RECORD filler: ARRAY 4 OF LONGINT; name: Name END;
	high0, high1, low0, low1, free0, free1, prev: ADDRESS;  c: CHAR;
	s: ARRAY 10 OF CHAR;
	p: Blockm4Ptr;  tag, tdesc: Tag;
BEGIN
	inGC := FALSE;  break := FALSE;  FinObjs := NIL;

		(* show statically linked modules *)
	m := modules;
	ASSERT(m # NIL);	(* modules initialized by linker! *)
(*
	WHILE m # NIL DO
		WriteHex(SYSTEM.ADR(m.code[0]), 8);  WriteChar(" ");
		WriteString(m.name);  WriteLn;
		m := m.next
	END;
*)

	m := modules;  WHILE m.name # "Kernel" DO m := m.next END;
	i := LEN(m.tdescs);  ptrElemTag := 0;
	REPEAT
		DEC(i);  t := m.tdescs[i];  SYSTEM.GET(t - 4, td);
		IF td.name = "PtrElemDesc" THEN ptrElemTag := t END
	UNTIL (ptrElemTag # 0);

	GetConfig("StackSize", s);  stacksize := StrToInt(s);
	DEC(stacksize, stacksize MOD 4);
	IF stacksize <= 0 THEN stacksize := DefaultStackSize	(* def *)
	ELSIF stacksize < 2*KernelStackSize THEN stacksize := 2*KernelStackSize	(* min *)
	END;

	GetConfig("DMASize", s);  dmasize := StrToInt(s);
	IF dmasize <= 0 THEN dmasize := 32*1024	(* def *)
	ELSIF dmasize < 8 THEN dmasize := 8	(* min *)
	END;
	INC(dmasize, (-dmasize) MOD PS (*4*));	(* dmasize MOD PS (*4*) = 0 *)
	
	GetConfig("PageHeap", s);  pageheap1 := StrToInt(s) * 1024;
	IF pageheap1 = 0 THEN pageheap1 := DefaultPageHeap END;
	INC(pageheap1, (-pageheap1) MOD PS);
	WriteInt((pageheap0-pageheap) DIV PS, 1);  WriteString(" pages allocated, ");
	WriteInt(pageheap1 DIV PS, 1);  WriteString(" pages reserved");  WriteLn;
	pageheap1 := pageheap - pageheap1;
	dma1 := pageheap1;	(* dma1 MOD 4k = 0 *)
	dma0 := dma1-dmasize;
	IF (dma0 DIV 10000H) # (dma1 DIV 10000H) THEN
		DEC(dma0, dma1 MOD 10000H)	(* at least one allocate of min(dmasize, 64k) will succeed *)
	END;
	IF dma0 MOD 10000H > 10000H-8 THEN DEC(dma0, 8) END;
	free0 := dma0;  prev := SYSTEM.ADR(dmafree);
	LOOP
		free1 := free0 + (10000H-free0 MOD 10000H);	(* next 64k boundary *)
		IF free1 > dma1 THEN free1 := dma1 END;
		IF free0 = free1 THEN EXIT END;
		IF 9 IN traceheap THEN
			WriteHex(free0, 8);  WriteHex(free1, 10);  WriteInt(free1-free0, 8);  WriteLn
		END;
		SYSTEM.PUT(free0, free1-free0);	(* size of free block *)
		SYSTEM.PUT(prev, free0);	(* link from prev. free block *)
		prev := free0+4;
		free0 := free1
	END;
	SYSTEM.PUT(prev, SYSTEM.VAL(LONGINT, 0));	(* last free block *)
	
		(* make DMA memory r/w for V86 mode *)
	ASSERT((dma0 < 400000H) & (dma1 < 400000H));	(* in first pt *)
	SYSTEM.GET(v86pd, pt);	(* get first pt *)
	DEC(pt, pt MOD PS);
	FOR i := dma0 DIV PS TO dma1 DIV PS - 1 DO
		SYSTEM.GET(pt + 4*i, set);  SYSTEM.PUT(pt + 4*i, set + {1})	(* set r/w *)
	END;

		(* get top part of heap *)
	high0 := SYSTEM.VAL(LONGINT, SYSTEM.VAL(SET, 100000H+stacksize+4+B-1) - SYSTEM.VAL(SET, B-1)) - 4;
	high1 := SYSTEM.VAL(LONGINT, SYSTEM.VAL(SET, heapTop-1) - SYSTEM.VAL(SET, B-1)) - 4;
	StackOrg := high0;	(* user stack org *)
		(* kernel stack sits below StackOrg, for stack overflows & GC *)

	SYSTEM.GET(LinkAdr+16H, low0);	(* patched by Linker *)
	ASSERT(low0 MOD B = B-4);	(* check if linker aligned correctly *)
	low1 := SYSTEM.VAL(LONGINT, SYSTEM.VAL(SET, dma0) - SYSTEM.VAL(SET, B-1)) - 4 - B;
	ASSERT(low1-2*B >= low0);	(* enough low memory? *)
	
	IF traceheap * {0..15} # {} THEN
		WriteHex(low0, 8);  WriteHex(low1, 10);  WriteHex(dma0, 10);  WriteHex(dma1, 10);  
		WriteHex(StackOrg, 10);  WriteHex(high0, 10);  WriteHex(high1, 10);  WriteLn
	END;

	ASSERT(high1 > 101000H);
		(* clear new heap and stack *)
	Fill4(low0, (low1-low0) DIV 4, 0);
	Fill4(100000H, (high1-100000H) DIV 4, 0);
	
		(* init heap *)
	firstBlock := LinkAdr+3CH;  endBlock := high1;
	
		(* find last block in Linker-generated heap (p^.tag will be NIL because of Fill4(low0, ...) above) *)
	p := SYSTEM.VAL(Blockm4Ptr, firstBlock);
	WHILE p^.tag # NIL DO
		tag := p^.tag;
		initres := SYSTEM.VAL(InitPtr, SYSTEM.VAL(ADDRESS, p)+(4+24));
		tdesc := SYSTEM.VAL(Tag, SYSTEM.VAL(SET, tag) - array - mark);
		IF array * SYSTEM.VAL(SET, tag) # {} THEN (* array block *)
			size := p^.lastElemToMark + tdesc^.size - SYSTEM.VAL(ADDRESS, p)
		ELSE
			size := tdesc^.size + 4
		END;
		size := SYSTEM.VAL(LONGINT, SYSTEM.VAL(SET, size + B-1)-SYSTEM.VAL(SET, B-1));
		INC(SYSTEM.VAL(ADDRESS, p), size)
	END;
	SYSTEM.GET(initres, c);  ASSERT(c = 0E8X);	(* call to first module body *)
	(* now initres points to last block, i.e. the init calls - they won't be GC'ed *)
	
		(* initial free blocks *)
	f := SYSTEM.VAL(FreeBlockPtr, high0);
	f^.tag := SYSTEM.VAL(Tag, SYSTEM.VAL(SET, SYSTEM.ADR(f^.size)) + free);
	f^.size := high1-high0-4;  f^.next := 0;

	f := SYSTEM.VAL(FreeBlockPtr, low0);
	f^.tag := SYSTEM.VAL(Tag, SYSTEM.VAL(SET, SYSTEM.ADR(f^.size)) + free);
	f^.size := low1-low0-4;  f^.next := high0;
	
		(* system reserved memory *)
	sysres := SYSTEM.VAL(InitPtr, low1);
	sysres^.tag := SYSTEM.VAL(Tag, SYSTEM.VAL(SET, SYSTEM.ADR(sysres^.z0)) + mark);
	sysres^.z0 := high0-low1-4;  sysres^.z1 := -4;
	sysres^.z5 := SYSTEM.ADR(sysres^.z0);
	sysres := SYSTEM.VAL(InitPtr, low1 + 28);

		(* init free list & reserve *)
	reserve := NIL;  firstTry := TRUE;
	GClevel := 1;  GCstack := FALSE;
	GC;
	GClevel := 0;  GCstack := TRUE;
	EnableGC := DefaultEnableGC;  DisableGC := DefaultDisableGC
END InitHeap;

(* IsRAM - Check if the specified address is RAM *)

PROCEDURE IsRAM(adr: LONGINT): BOOLEAN;
CONST Pattern1 = 0BEEFC0DEH;  Pattern2 = 0AA55FF00H;
VAR save, x: LONGINT;  ok: BOOLEAN;
BEGIN
	ok := FALSE;
	SYSTEM.GET(adr, save);
	SYSTEM.PUT(adr, Pattern1);	(* attempt 1st write *)
	x := Pattern2;	(* write something else *)
	SYSTEM.GET(adr, x);	(* attempt 1st read *)
	IF x = Pattern1 THEN	(* first test passed *)
		SYSTEM.PUT(adr, Pattern2);	(* attempt 2nd write *)
		x := Pattern1;	(* write something else *)
		SYSTEM.GET(adr, x);	(* attempt 2nd read *)
		ok := (x = Pattern2)
	END;
	SYSTEM.PUT(adr, save);
	RETURN ok
END IsRAM;

(* CheckMemory - Check amount of memory available and update memTop *)

PROCEDURE CheckMemory;	(* no tracing allowed here *)
CONST
	M = 100000H;  ExtMemAdr = M;  Step = M;
	MaxMem = ExtMemAdr+128*M;	(* avoid allocating too many pages *)
VAR
	s: ARRAY 16 OF CHAR;  adr: LONGINT;
BEGIN
	GetConfig("ExtMemSize", s);	(* in MB *)
	IF s[0] # 0X THEN	(* override detection *)
		memTop := ExtMemAdr + StrToInt(s) * M
	ELSE
		IF memTop >= 63*M THEN	(* search for more memory (ignore aliasing) *)
			adr := memTop-4;
			WHILE IsRAM(adr) DO
				memTop := adr+4;
				INC(adr, Step)
			END
		END;
		IF memTop > MaxMem THEN
			(*WriteInt((memTop-MaxMem) DIV 1024, 1);
			WriteString("KB ignored");  WriteLn;*)
			memTop := MaxMem
		END
	END;
	(*WriteInt((memTop-ExtMemAdr) DIV 1024, 1);
	WriteString("KB extended memory");  WriteLn*)
END CheckMemory;

PROCEDURE ReadBootTable;	(* set memTop, pageheap, configadr *)
CONST HeapAdr = 100000H;
VAR p, type, addr, size: LONGINT;
BEGIN
	memTop := 0;  pageheap := 0;  configadr := 0;
	p := bt;
	LOOP
		SYSTEM.GET(p, type);
		IF type = -1 THEN EXIT	(* end *)
		ELSIF type = 3 THEN	(* boot memory/top of low memory *)
			SYSTEM.GET(p+8, addr);  (*SYSTEM.GET(p+12, size);*)
			pageheap := addr - addr MOD PS	(* align to page boundary *)
		ELSIF type = 4 THEN	(* free memory/extended memory size *)
			SYSTEM.GET(p+8, addr);  SYSTEM.GET(p+12, size);
			IF addr = HeapAdr THEN memTop := HeapAdr+size END
		ELSIF type = 8 THEN	(* config strings *)
			configadr := p+8
		END;
		SYSTEM.GET(p+4, size);  INC(p, size)
	END;
	WHILE (memTop = 0) OR (pageheap = 0) DO END;	(* nothing else to do *)
	FOR p := 0 TO 2 DO SYSTEM.GET(408H + p*2, bioslpt[p]) END
END ReadBootTable;

(* ShowState - Show machine state *)

PROCEDURE ShowState(error, fp, pc, pf: LONGINT);
CONST MaxFrames = 10;
VAR x, bp, esp, cs, ds, es, fs, gs, ss: LONGINT;  m: Module;  flags, set: SET;
	
	PROCEDURE Val(s: ARRAY OF CHAR; ofs: LONGINT);
	BEGIN
		SYSTEM.GET(bp+ofs, ofs);
		WriteChar(" ");  WriteString(s);  WriteChar("=");  WriteHex(ofs, 8)
	END Val;

	PROCEDURE Var(s: ARRAY OF CHAR;  var: LONGINT);
	BEGIN
		WriteChar(" ");  WriteString(s);  WriteChar("=");  WriteHex(var, 8)
	END Var;
	
BEGIN
	IF handlingtrap THEN	
		WriteString(" [Recursive TRAP]");  WriteLn
	ELSE
		handlingtrap := TRUE;
		SYSTEM.GETREG(5, bp);	(* EBP *)
		SYSTEM.GET(bp, bp);	(* interrupt handler frame *)
		WriteLn;
		WriteString("TRAP ");  WriteInt(error, 1);
		SYSTEM.GET(bp+52, cs);  SYSTEM.GET(bp+56, flags);
		IF 17 IN flags THEN	(* from VM *)
			WriteString(", V86");
			SYSTEM.GET(bp+80, gs);  SYSTEM.GET(bp+76, fs);  SYSTEM.GET(bp+72, ds);  SYSTEM.GET(bp+68, es);
			SYSTEM.GET(bp+64, ss);  SYSTEM.GET(bp+60, esp)
		ELSIF cs MOD 4 = 0 THEN	(* from level 0 *)
			WriteString(", level 0");
			ds := DS();  es := ES();  fs := FS();  gs := GS();  ss := SS();
			SYSTEM.GET(bp+20, esp);  INC(esp, 16)
		ELSE	(* from level 3 *)
			ds := DS();  es := ES();  fs := FS();  gs := GS();
			SYSTEM.GET(bp+64, ss);  SYSTEM.GET(bp+60, esp)
		END;
		WriteChar(" "); WriteString(version);
		WriteLn;

			(* the registers *)
		set := GetFlags();
		Var("CS:", cs);  Var("DS:", ds);  Var("ES:", es);  Var("SS:", ss);  Var("CR0", trapCR[0]);
		Var("FPS", SYSTEM.VAL(LONGINT, trapfpu[1] * {0..15} + SYSTEM.LSH(trapfpu[2], 16)));  WriteLn;
		Var("EIP", pc);  Val("ESI", 12);  Val("EDI", 8);  Var("ESP", esp);  Var("CR2", trapCR[2]);  Var("SBT", StackOrg);  WriteLn;
		Val("EAX", 36);  Val("EBX", 24);  Val("ECX", 32);  Val("EDX", 28);  Var("CR3", trapCR[3]);  
		Var("KFL", SYSTEM.VAL(LONGINT, set));  WriteLn;
		Val("EBP", 16);  Var("FS:", fs);  Var("GS:", gs);  Val("ERR", 44);  Var("CR4", trapCR[4]);  Var("TCK", ticks);  WriteLn;
		IF SYSTEM.VAL(CHAR, trapDR[7]) # 0X THEN	(* some breakpoints enabled *)
			Var("DR0", trapDR[0]);  Var("DR1", trapDR[1]);  Var("DR2", trapDR[2]);  Var("DR3", trapDR[3]);
			Var("DR6", trapDR[6]);  Var("DR7", trapDR[7]);  WriteLn
		END;
		WriteString(" EFLAGS=");  WriteFlags(SYSTEM.VAL(LONGINT, flags));  WriteLn;
		
		IF error = -6 THEN WriteMemory(pc, 16) END;	(* show bad instruction *)
		IF ~(17 IN flags) THEN	(* ~VM => show procedures *)
			x := 0;
			LOOP
				m := GetMod(pc);
				IF m = NIL THEN
					WriteString("Unknown EIP=");  WriteHex(pc, 8);
					IF x # 0 THEN EXIT END
				ELSE
					WriteString(m.name);  WriteString(" PC=");
					WriteInt(pc-SYSTEM.ADR(m.code[0]), 1);
				END;
		        SYSTEM.GET(fp+4, pc); SYSTEM.GET(fp, fp); (* return addr from stack *)
				IF fp >= StackOrg THEN EXIT END;
				WriteString(", ");
				INC(x);  IF x = MaxFrames THEN WriteString("...");  EXIT END
			END;
			WriteLn
		END
	END;
	handlingtrap := FALSE
END ShowState;

PROCEDURE InitTrapHandling;
VAR i: SHORTINT;  int: Proc;
BEGIN
	int := ExceptionHandler;  handler0 := ShowState;  handler := NIL;
	handlingtrap := FALSE;
	loop := NIL;
	FOR i := 0 TO 31 DO	(* install exception handler for every "Intel" exception *)
		IF i = 2 THEN InstallIP(NMIHandler, i)
		ELSIF IgnoreTrap15 & (i = 15) THEN (* skip *)
		ELSE InstallIP(int, i)
		END
	END;
	IF oldcopro THEN InstallIP(int, IRQ+13) END	(* 387 compatible *)
END InitTrapHandling;

(*
	---------------- Machine initialisation ----------------
	From University of Stellenbosch's Gneiss kernel (pjm)

	References:
	1. J.H. Crawford and P.P. Gelsinger, "Programming the 80386", Sybex, 1987
	2. Intel, "80386 Programmer's Reference Manual, 1986
	3. Intel, "i486(tm) Microprocessor", April 1989, #240440-001
*)

(* Detect486 - Detect 486 (or above) *)

PROCEDURE Detect486(): BOOLEAN;
CODE {SYSTEM.i386}
	PUSHFD
	PUSHFD
	POP EBX
	BTS EBX, 18
	PUSH EBX
	POPFD
	PUSHFD
	POP EBX
	POPFD
	BT EBX, 18
	SETC AL
END Detect486;

(* Detect586 - Detect 586 (or above) *)

PROCEDURE Detect586(): BOOLEAN;
CODE {SYSTEM.i386}
	PUSHFD
	POP EAX
	MOV EBX, EAX	; save flags
	XOR EAX, 200000H	; toggle bit 21
	PUSH EAX
	POPFD	; try to modify bit 21
	PUSHFD
	POP EAX
	PUSH EBX	; restore flags
	POPFD
	XOR EAX, EBX	; has bit changed?
	SETNZ AL
END Detect586;

(* DetectCoprocessor - Detect 387 (or compatible) *)

PROCEDURE DetectCoprocessor(): BOOLEAN;
CODE {SYSTEM.i386, SYSTEM.FPU}
	PUSH -1
	FNINIT
	MOV ECX, 400
l1:
	LOOP l1
	FNSTSW [ESP]
	MOV ECX, 400
l2:
	LOOP l2
	CMP BYTE [ESP], 0
	SETZ AL
	POP EBX
END DetectCoprocessor;

(* SetupFlags - Set up flags (3, p. 20)
	Bit
	1,3,5,15,19..31 - no change
	0,2,4,6..7,11 - CF,PF,AF,ZF,SF,OF off
	8 - TF off
	9 - IF off (no interrupts)
	10 - DF off
	12..13 - IOPL = 3
	14 - NT off (no Windows)
	16 - RF off (no Interference)
	17- VM off (no virtual 8086 mode)
	18 - AC off (no 486 alignment checks) *)

PROCEDURE SetupFlags;
CODE {SYSTEM.i386}
	PUSHFD
	AND DWORD [ESP], 0FFF8802AH
	OR DWORD [ESP], 3000H
	POPFD
END SetupFlags;

(* Setup486Flags - Set up various 486-specific flags (3, p. 23)
	1. Enable exception 16 on math errors.
	2. Disable supervisor mode faults on write to read-only pages 
		(386-compatible for stack checking).
	3. Enable the Alignment Check field in EFLAGS *)

PROCEDURE Setup486Flags;
CODE {SYSTEM.i386, SYSTEM.Privileged}
	MOV EAX, CR0
	OR EAX, 00040020H
	AND EAX, 0FFFEFFFFH
	MOV CR0, EAX
END Setup486Flags;

(* Setup586Flags - Set up 586-specific things *)

PROCEDURE Setup586Flags;
CODE {SYSTEM.Pentium, SYSTEM.Privileged}
	MOV EAX, CR4
	BTR EAX, 2	; clear TSD
	MOV CR4, EAX
END Setup586Flags;

(* DisableMathTaskEx - Disable exceptions caused by math in
	new task.  (1, p. 479) *)

PROCEDURE DisableMathTaskEx;
CODE {SYSTEM.i386, SYSTEM.Privileged}
	MOV EAX,CR0
	AND AL, 0F5H
	MOV CR0, EAX
END DisableMathTaskEx;

(* EnableEmulation - Enable math emulation (1, p. 479)*)

PROCEDURE EnableEmulation;
CODE {SYSTEM.i386, SYSTEM.Privileged}
	MOV EAX, CR0
	OR AL, 4
	MOV CR0, EAX
END EnableEmulation;

(* DisableEmulation - Disable math emulation (1, p. 479) *)

PROCEDURE DisableEmulation;
CODE {SYSTEM.i386, SYSTEM.Privileged}
	MOV EAX, CR0
	AND AL, 0FBH
	MOV CR0, EAX
END DisableEmulation;

(* LoadIDT - Load interrupt descriptor table *)

PROCEDURE LoadIDT(base, size: LONGINT);
CODE {SYSTEM.i386, SYSTEM.Privileged}
	SHL size[EBP], 16
	MOV EBX, 2
	LIDT size[EBP][EBX]
END LoadIDT;

(* LoadGDT - Load global descriptor table *)

PROCEDURE LoadGDT(base, size: LONGINT);
CODE {SYSTEM.i386, SYSTEM.Privileged}
	SHL size[EBP], 16
	MOV EBX, 2
	LGDT size[EBP][EBX]
END LoadGDT;

(* LoadSegRegs - Load segment registers *)

PROCEDURE LoadSegRegs(data: LONGINT);
CODE {SYSTEM.i386}
	MOV EAX, data[EBP]
	MOV DS, AX
	MOV ES, AX
	XOR EAX, EAX
	MOV FS, AX
	MOV GS, AX
END LoadSegRegs;

(* SetTR - Set task register *)

PROCEDURE SetTR(tr: LONGINT);
CODE {SYSTEM.i386, SYSTEM.Privileged}
	LTR WORD tr[EBP]
END SetTR;

(* CPUID - CPUID instruction *)

PROCEDURE CPUID(VAR vendor: Vendor;  VAR version, features: LONGINT);
CODE {SYSTEM.Pentium}
	MOV EAX, 0
	CPUID
	CMP EAX, 0
	JNE ok
	
	MOV ESI, vendor[EBP]
	MOV [ESI], AL
	MOV ESI, version[EBP]
	MOV [ESI], EAX
	MOV ESI, features[EBP]
	MOV [ESI], EAX
	
	MOV ESP, EBP
	RET 12

ok:
	MOV ESI, vendor[EBP]
	MOV [ESI], EBX
	MOV 4[ESI], EDX
	MOV 8[ESI], ECX
	MOV BYTE 12[ESI], 0
	
	MOV EAX, 1
	CPUID
	
	MOV ESI, version[EBP]
	MOV [ESI], EAX
	MOV ESI, features[EBP]
	MOV [ESI], EDX
END CPUID;

(* APM - Check if APM supported. *)

PROCEDURE APM(VAR gdtofs, apmofs: LONGINT): BOOLEAN;
VAR gdtptr: ARRAY 2 OF LONGINT;
CODE {SYSTEM.i386}
	SGDT gdtptr[EBP]
	XOR EBX,EBX
	MOV  BX,gdtptr[EBP]	; limit
	INC EBX	; GDT size in bytes
	MOV AL,0
	CMP EBX,3*8
	JBE ret	; GDT too small, APM not installed
	LEA EAX,gdtptr[EBP]
	ADD EAX,2	; EAX points to GDT base
	MOV ECX,[EAX]
	MOV EDX,gdtofs[EBP]
	MOV [EDX],ECX
	ADD EBX,[EAX]	; offset after GDT (apmofs dd ?)
	MOV EAX,[EBX]	; get APM entry point
	MOV EDX,apmofs[EBP]
	MOV [EDX],EAX
	MOV AL,1
ret:
END APM;

(* APMPowerOff - Switch off the machine *)

PROCEDURE APMPowerOff;
VAR adr: ARRAY 3 OF INTEGER;
CODE {SYSTEM.i386, SYSTEM.Privileged}
	LEA EAX,adr[EBP]
	MOV EBX,apmofs
	MOV [EAX],EBX
	ADD EAX,4
	MOV WORD [EAX],6*8	; CS value
	MOV AX,8*8
;	MOV DS,AX	; setting DS would require restoring DS in interrupt glue
	MOV AX,5307H
	MOV BX,1
	MOV CX,3	; off
	LEA ESI,adr[EBP]
	DB 0FFH, 1EH	; CALL PWORD [ESI]
	CLI
wait:
	HLT
	JMP wait
END APMPowerOff;

(* InitProcessor - Initialise processor and interrupts. *)

PROCEDURE InitProcessor;
VAR mask: SET;  s: ARRAY 16 OF CHAR;  gdtofs: LONGINT;
BEGIN
		(* The bootstrap loader switched the machine to protected mode. *)
	SetupFlags;
	GetConfig("CPU", s);
	cpuversion := 0;
	IF s = "3" THEN cpu := 3
	ELSIF s = "4" THEN cpu := 4
	ELSIF s = "5" THEN cpu := 5
	ELSE
		IF Detect486() THEN
			IF Detect586() THEN
				CPUID(cpuvendor, cpuversion, cpufeatures);
				cpu := SHORT(SHORT(ASH(cpuversion, -8) MOD 16))	(* family *)
			ELSE
				cpu := 4
			END
		ELSE cpu := 3
		END
	END;
	IF cpuversion = 0 THEN cpuversion := cpu END;
	IF cpu >= 4 THEN Setup486Flags END;
	IF cpu >= 5 THEN Setup586Flags END;
	GetConfig("Copro", s);
	IF s = "0" THEN copro := FALSE
	ELSIF s = "" THEN copro := DetectCoprocessor()
	ELSE copro := TRUE
	END;
	oldcopro := copro & (cpu < 4);
	DisableMathTaskEx;
	IF copro THEN DisableEmulation; SetupFPU
	ELSE EnableEmulation
	END;

		(* initialise 8259 interrupt controller *)
		(* initialize controller chip 1 *)
	SYSTEM.PORTOUT(IntA0, 11X);	SYSTEM.PORTOUT(IntA1, CHR(IRQ+0));
	SYSTEM.PORTOUT(IntA1, 4X);	SYSTEM.PORTOUT(IntA1, 1X);
	SYSTEM.PORTOUT(IntA1, 0FFX);
		(* initialize controller chip 2 *)
	SYSTEM.PORTOUT(IntB0, 11X);	SYSTEM.PORTOUT(IntB1, CHR(IRQ+8));
	SYSTEM.PORTOUT(IntB1, 2X);	SYSTEM.PORTOUT(IntB1, 1X);
	SYSTEM.PORTOUT(IntB1, 0FFX);
		(* enable interrupts from second interrupt controller *)
	SYSTEM.PORTIN(IntA1, SYSTEM.VAL(CHAR, mask));
		(* chained to line 2 of controller 1 *)
	EXCL(mask, 2);
	SYSTEM.PORTOUT(IntA1, SYSTEM.VAL(CHAR, mask));

		(* initialise kernel TSS *)
	Fill4(SYSTEM.ADR(ktss), SIZE(TSSDesc) DIV 4, 0);
	(*ktss.Link := 0;*)
	ktss.ESP0 := 100000H+KernelStackSize;	(* kernel stack org *)
	ktss.ESS0 := KernelStackSel;
	(*ktss.ESP1 := 0;  ktss.ESS1 := NullSelector;*)
	(*ktss.ESP2 := 0;  ktss.ESS2 := NullSelector;*)
	(*ktss.CR3 := 0;  ktss.EIP := 0;*)
	(*ktss.EFLAGS := {};*)
	(*ktss.EAX := 0;  ktss.ECX := 0;  ktss.EDX := 0;  ktss.EBX := 0;*)
	(*ktss.ESP := esp;  ktss.EBP := 0;  ktss.ESI := 0;  ktss.EDI := 0;*)
	(*ktss.ES := NullSelector;  ktss.CS := NullSelector;*)
	(*ktss.SS := NullSelector;  ktss.DS := NullSelector;*)
	(*ktss.FS := NullSelector;  ktss.GS := NullSelector;*)
	(*ktss.LDT := NullSelector;  ktss.TaskAttributes := 0;*)
	IF V86 THEN
		ktss.IOBitmapOffset := SHORT(SYSTEM.ADR(ktss.IOBitmap) - SYSTEM.ADR(ktss.Link))
		(* all bits in IOBitmap are 0 => all IO allowed *)
	ELSE
		ktss.IOBitmapOffset := -1	(* no bitmap *)
	END;
	gdtofs := 0;  apmofs := -1;
	GetConfig("APM", s);
	IF s # "0" THEN
		IF APM(gdtofs, apmofs) THEN
			(*WriteString("APM at");  WriteHex(apmofs, 9);  WriteHex(gdtofs, 9);
			WriteLn*)
		END
	END;
	GetConfig("PowerSave", s);
	powersave := s[0] # "0";	(* default on *)

		(* initialise GDT *)
		(* GDT 0: Null segment *)
	SYSTEM.PUT(SYSTEM.ADR(gdt[0]), SYSTEM.VAL(LONGINT, 0));
	SYSTEM.PUT(SYSTEM.ADR(gdt[0])+4, SYSTEM.VAL(LONGINT, 0));
		(* GDT 1: Kernel code: base 0, limit 4G, type 10 (exe/rd), DPL 0, 32-bit code *)
	SYSTEM.PUT(SYSTEM.ADR(gdt[1]), SYSTEM.VAL(LONGINT, 0FFFFH));
	SYSTEM.PUT(SYSTEM.ADR(gdt[1])+4, SYSTEM.VAL(LONGINT, 0CF9A00H));
		(* GDT 2: Kernel data: base 0, limit 4G, type 2 (r/w), DPL 0, 32-bit stack *)
	SYSTEM.PUT(SYSTEM.ADR(gdt[2]), SYSTEM.VAL(LONGINT, 0FFFFH));
	SYSTEM.PUT(SYSTEM.ADR(gdt[2])+4, SYSTEM.VAL(LONGINT, 0CF9200H));
		(* GDT 3: User code: base 0, limit 4G, type 14 (exe/rd/conf), DPL 0, 32-bit code *)
	SYSTEM.PUT(SYSTEM.ADR(gdt[3]), SYSTEM.VAL(LONGINT, 0FFFFH));
	SYSTEM.PUT(SYSTEM.ADR(gdt[3])+4, SYSTEM.VAL(LONGINT, 0CF9E00H));
		(* GDT 4: User data: base 0, limit 4G, type 2 (r/w), DPL 3, 32-bit stack *)
	SYSTEM.PUT(SYSTEM.ADR(gdt[4]), SYSTEM.VAL(LONGINT, 0FFFFH));
	SYSTEM.PUT(SYSTEM.ADR(gdt[4])+4, SYSTEM.VAL(LONGINT, 0CFF200H));
		(* GDT 5: Kernel TSS *)
	gdt[5].limit0to15 := SIZE(TSSDesc)-1;	(* assume TSSDesc size < 32k *)
	gdt[5].base0to15 := SYSTEM.VAL(INTEGER, SHORT(SYSTEM.ADR(ktss)));
	gdt[5].base16to23 := CHR(ASH(SYSTEM.ADR(ktss), -16));
	gdt[5].base24to31 := CHR(ASH(SYSTEM.ADR(ktss), -24));
	gdt[5].accessByte := 89X;  (* present system segment, DPL 0, i386TSS *)
	gdt[5].granularityByte := 0X;	(* byte granularity, 0 high bits of TSSDesc size *)
		(* GDT 6..8: APM *)
	IF apmofs # -1 THEN
		SYSTEM.MOVE(gdtofs+3*8, SYSTEM.ADR(gdt[6]), 3*8)	(* copy APM segments *)
	END;
		(* load the segments *)
	LoadGDT(SYSTEM.ADR(gdt[0]), SIZE(GDT)-1);
	LoadSegRegs(DataSel);
	SetTR(KernelTR)
END InitProcessor;

(* V86Exit - Interrupt V86ExitInt handler to switch back to protected mode. *)

PROCEDURE V86Exit;
CODE {SYSTEM.i386}	(* { interrupts off } *)
	CLD	; save V86 registers
	LEA ESI, 8[EBP]
	MOV EDI, vregadr
	MOV ECX, 19
	REP MOVSD
	
	MOV EBP, vframe	; switch to kernel stack
END V86Exit;

(* V86Switch - Switch to V86 mode. *)

PROCEDURE V86Switch;
CODE {SYSTEM.i386, SYSTEM.Privileged}	(* { interrupts off } *)
	MOV vframe, EBP	; save kernel stack
	
	MOV EAX, v86pd	; switch to V86 page directory
	MOV CR3, EAX
	
	MOV ESP, vregadr	; switch to V86 mode
	POPAD
	ADD ESP, 8	; errCode & int
	IRETD
END V86Switch;

PROCEDURE -ReadMSR(msr: LONGINT;  lowadr, highadr: LONGINT);
CODE {SYSTEM.Pentium, SYSTEM.Privileged}
	POP EDI
	POP ESI
	POP ECX
	RDMSR
	MOV [ESI], EAX
	MOV [EDI], EDX
END ReadMSR;

PROCEDURE -WriteMSR(msr: LONGINT;  low, high: SET);
CODE {SYSTEM.Pentium, SYSTEM.Privileged}
	POP EDX
	POP EAX
	POP ECX
	WRMSR
END WriteMSR;

PROCEDURE -HLT;
CODE {SYSTEM.i386, SYSTEM.Privileged}
	HLT
END HLT;

(* V86IntHandler - Interrupt V86EnterInt handler to access V86 services.

	EAX = 0 -> Switch to V86 mode using the specified registers.  
		Return on execution of INT V86ExitInt in V86 mode.
		EBX -> address  of V86Regs state record
	
	(nyi) EAX = 1 -> memory copy (page 0 also allowed)
		ESI -> source address
		EDI -> destination address
		ECX -> number of bytes 
	
	EAX = 2 -> read MSR
		ECX = msr number
	Returns:
		EDX:EAX = msr value
	
	EAX = 3 -> write MSR
		ECX = msr number
		EDX:EBX = msr value
	
	EAX = 4 -> HLT
*)

PROCEDURE V86IntHandler;
VAR s: SET;  func, bp, reg: LONGINT;  low, high: SET;
BEGIN	(* level 0 *)
	SYSTEM.STI();
	SYSTEM.GETREG(5, bp);	(* EBP *)
	SYSTEM.GET(bp+56, s);  ASSERT(~(17 IN s));	(* not called from VM *)
	SYSTEM.GET(bp+52, s);  ASSERT(s * {0,1} = {0,1});	(* called from level 3 *)
	SYSTEM.GET(bp+36, func);	(* function number *)
	CASE func OF
		0:	(* switch to V86 *)
			SYSTEM.GET(bp+24, vregadr);	(* ADR(registers) *)
			SYSTEM.GET(vregadr+48, s);	(* EFLAGS *)
			s := (s * {0, 2, 4, 6, 7, 10, 11}) + (GetFlags() * {1, 3, 5, 12..31}) + {17};	(* +VM -IF *)
			SYSTEM.PUT(vregadr+48, s);
			SYSTEM.CLI();
			ktss.ESP0 := 100000H+KernelStackSize DIV 2;	(* use lower half for V86 mode *)
			V86Switch;
			ktss.ESP0 := 100000H+KernelStackSize;	(* use top half normally *)
			SYSTEM.STI()
		|2:	(* read MSR *)
			SYSTEM.GET(bp+32, reg);	(* ECX *)
			ReadMSR(reg, bp+36, bp+28)	(* ADR(EAX), ADR(EDX) *)
		|3:	(* write MSR *)
			SYSTEM.GET(bp+32, reg);	(* ECX *)
			SYSTEM.GET(bp+28, high);	(* EDX *)
			SYSTEM.GET(bp+24, low);	(* EBX *)
			WriteMSR(reg, low, high)
		|4:	(* HLT *)
			HLT
	END (* CASE *)
END V86IntHandler;

(* InterruptHandler

	Stack layout:
		72	GS
		68	FS
		64	DS
		60	ES
		-- if from VM --
		56	SS
		52	ESP
		-- if from level 3 or VM --
		48	EFLAGS
		44	CS
		40	EIP
		36	errorcode
		32	intnum
		28	EAX
		24	ECX
		20	EDX
		16	EBX
		12	ESP' (overwritten with interrupt handler table adr)
		08	EBP
		04	ESI
		00	EDI	<--- ESP
*)

PROCEDURE InterruptHandler;
CODE {SYSTEM.i386, SYSTEM.Privileged}  (* 3 bytes implicit code skipped: PUSH EBP;  MOV EBP, ESP *)
entry:
	PUSHAD	; save all registers
	TEST 48[ESP], 20000H	; test VM bit
	JNZ v86

call:
	MOV EBX, 32[ESP]	; EBX = int number
	SHL EBX, MaxIntsShift2
	ADD EBX, intHandlerAdr
	MOV 12[ESP], EBX	; save adr
	
loop1:
	CALL DWORD [EBX]	; call handler
	
	CLI
	ADD DWORD 12[ESP], 4	; set adr to next handler
	MOV EBX, 12[ESP]
	CMP DWORD [EBX], 0	; end of list?
	JNE loop1

exit:
	MOV EBX, 32[ESP]	; EBX = int number
	SUB EBX, IRQ
	JC noirq
	MOV AL, 20H
	CMP EBX, 8
	JB irq1
	CMP EBX, 16
	JAE noirq
	CLI
	OUT IntB0, AL	; acknowledge the interrupt
irq1:
	OUT IntA0, AL
noirq:
	POPAD
	ADD ESP, 8	; drop int & errorcode
	IRETD

v86:
	MOV AX, DataSel
	MOV DS, AX
	MOV ES, AX
	MOV EAX, kernelpd
	MOV CR3, EAX
	
	MOV EBX, 32[ESP]	; EBX = int number
	SHL EBX, MaxIntsShift2
	ADD EBX, intHandlerAdr
	MOV 12[ESP], EBX	; save adr
	
loop2:
	CALL DWORD [EBX]	; call handler
	
	CLI
	ADD DWORD 12[ESP], 4	; set adr to next handler
	MOV EBX, 12[ESP]
	CMP DWORD [EBX], 0	; end of list?
	JNE loop2

	MOV EAX, v86pd	; when using tasks, also set ktss.CR3
	MOV CR3, EAX
	JMP exit
	
END InterruptHandler;

(* InitInterrupts - Initialise interrupt handling

	The glue code is:
		PUSH 0	; entry point for interrupts without error code
		PUSH i	; entry point for interrupts with error code
		JMP InterruptHandler:entry
	
		(see also oldcopro code in ExceptionHandler)
*)

PROCEDURE InitInterrupts;
VAR a: LONGINT;  o, i, j: SHORTINT;  p: Proc;
BEGIN
	p := InterruptHandler;  defaultHandler := Unexpected;
	intHandlerAdr := SYSTEM.ADR(intHandler[0, 0]);
	FOR i := 0 TO IDTSize-1 DO	(* set up glue code *)
		intHandler[i, 0] := defaultHandler; intHandler[i, 1] := NIL;
		o := 0;
			(* PUSH error code, int num & regs *)
		glue[i][o] := 6AX; INC(o); glue[i][o] := 0X; INC(o);	(* PUSH 0 ; {o = 2} *)
		glue[i][o] := 6AX; INC(o); glue[i][o] := CHR(i); INC(o);	(* PUSH i *)
		a := SYSTEM.VAL(LONGINT, p)+3 - (SYSTEM.ADR(glue[i][o])+5);
		glue[i][o] := 0E9X; INC(o);	(* JMP InterruptHandler:entry *)
		SYSTEM.PUT(SYSTEM.ADR(glue[i][o]), a);
	
			(* set up IDT entry *)
		IF (i > 31) OR ~(i IN {8, 10..14, 17}) THEN a := SYSTEM.ADR(glue[i][0])	(* include PUSH 0 *)
		ELSE a := SYSTEM.ADR(glue[i][2])	(* skip PUSH 0, processor supplies error code *)
		END;
		idt[i].offsetBits0to15 := SHORT(a MOD 10000H);
		IF V86 OR (i < IRQ) OR (i = IRQ+1) OR (oldcopro & (i = IRQ+13)) THEN
			idt[i].selector := KernelCodeSel;	(* gdt[1] -> non-conformant segment => level 0 *)
			idt[i].gateType := SYSTEM.VAL(INTEGER, 0EE00H)	(* present, DPL 3, system, 386 interrupt *)
		ELSE
			idt[i].selector := UserCodeSel;	(* gdt[3] -> conformant segment => level 0 or 3 *)
			idt[i].gateType := SYSTEM.VAL(INTEGER, 08E00H)	(* present, DPL 0, system, 386 interrupt *)
		END;
		idt[i].offsetBits16to31 := SHORT(a DIV 10000H)
	END;
	LoadIDT(SYSTEM.ADR(idt[0]), SIZE(IDT)-1);
	SYSTEM.STI()	(* enable interrupts *)
END InitInterrupts;

(* EnableMM - Enable memory management.  Also flushes TLB and loads initial CR3. *)

PROCEDURE EnableMM(pd: LONGINT);
CODE {SYSTEM.i386, SYSTEM.Privileged}
	MOV EBX, pd[EBP]
	MOV CR3, EBX
	MOV EAX, CR0
	OR EAX, 80000000H
	MOV CR0, EAX
	JMP 0
END EnableMM;

(* InitMemory - Init memory management to catch NIL and bad pointer references *)

PROCEDURE InitMemory;
CONST R = 400000H;	(* 4M region *)
VAR i, j, m, n, phys: LONGINT; pt: ADDRESS; s: ARRAY 32 OF CHAR; t: SET;
BEGIN
	IF MapCacheSize # 0 THEN
		FOR i := 0 TO MapCacheSize-1 DO mapcache[i].size := 0 END
	END;
		(* create the address space (also see MapPhysical) *)
	AllocatePage(kernelpd);  ASSERT(kernelpd # 0);
	phys := 0;  i := 0;  m := (memTop+R-1) DIV R;
	WHILE i # m DO
		AllocatePage(pt);  ASSERT(pt # 0);
		j := 0;  n := PTEs;
		WHILE j # n DO
			IF phys >= memTop THEN
				n := j
			ELSE
				SYSTEM.PUT(pt + 4*j, phys + NormalPage);
				INC(phys, PS);  INC(j) 
			END
		END;
		WHILE j # PTEs DO SYSTEM.PUT(pt + 4*j, SYSTEM.VAL(LONGINT, PageNotPresent));  INC(j) END;
		SYSTEM.PUT(kernelpd + 4*i, pt + NormalPage);
		INC(i)
	END;
	WHILE i # PTEs DO SYSTEM.PUT(kernelpd + 4*i, SYSTEM.VAL(LONGINT, PageNotPresent)); INC(i) END;	(* page not present *)
	mapPtr := VesaAdr;
	i := 0;
	WHILE i # 10 DO
		s := "MapAdrX";  s[6] := CHR(48+i);  GetConfig(s, s);  m := StrToInt(s);
		s := "MapLenX";  s[6] := CHR(48+i);  GetConfig(s, s);  n := StrToInt(s);
		IF (m # 0) & (n # 0) THEN MapMem(kernelpd, m, n, m+NormalPage); INC(i)
		ELSE i := 10
		END
	END;
	GetConfig("MapVesa", s);
	IF s = "1" THEN
		ASSERT(kpar[1] MOD PS = 0);
		MapMem(kernelpd, VesaAdr, VesaSize, kpar[1]+NormalPage);	(* map at VesaAdr to DX:CX from Init code *)
		kpar[1] := VesaAdr	(* for compatibility with older display driver *)
	END;
		(* set up v86 paging *)
	IF V86 THEN
		AllocatePage(v86pd);  ASSERT(v86pd # 0);
		Copy4(kernelpd, v86pd, PTEs);
		AllocatePage(pt);  ASSERT(pt # 0);
		
		SYSTEM.GET(kernelpd, phys);	(* first page table *)
		DEC(phys, phys MOD PS);
		Copy4(phys, pt, PTEs);	(* map first 4M the same *)
		SYSTEM.PUT(v86pd, pt + NormalPage);
		
		FOR i := 1 TO 9FH DO	(* make 4k-640k read-only *)
			SYSTEM.GET(pt + 4*i, t);  SYSTEM.PUT(pt + 4*i, t - {1})	(* clear r/w *)
		END
	ELSE
		v86pd := 0
	END;
		(* unmap the first virtual page (to catch NIL references) *)
	SYSTEM.GET(kernelpd, pt);  DEC(pt, NormalPage);
	SYSTEM.PUT(pt, SYSTEM.VAL(LONGINT, PageNotPresent));
		(* the top 64k of the first 1M is r/o (to catch kernel stack overflows) *)
	FOR i := 0F0H TO 0FFH DO
		SYSTEM.GET(pt + 4*i, t);  SYSTEM.PUT(pt + 4*i, t - {1})	(* clear r/w *)
	END;
		(* the kernel stack is r/o (to catch user stack overflows). level 0 can write anyway. *)
	FOR i := 100H TO 100H + KernelStackSize DIV 1000H - 1 DO
		SYSTEM.GET(pt + 4*i, t);  SYSTEM.PUT(pt + 4*i, t - {1})	(* clear r/w *)
	END;
(*
		(* top 1k is filled with 0's to make NIL IS tests to return FALSE (re-use page 0) *)
	ASSERT(~ODD(MappedPage(kernelpd, 0FFFFF000H)));	(* not mapped yet *)
	Fill4(PS-1024, 1024 DIV 4, 0);	(* record extlevel up to 255 *)
	MapPage(kernelpd, 0FFFFF000H, 0 + NormalPage);
*)
		(* set boot flag *)
	SYSTEM.PUT(0472H, 01234H);	(* soft boot flag, for when we reboot *)
	ktss.CR3 := kernelpd;
	EnableMM(kernelpd)
END InitMemory;

PROCEDURE -Call15;
CODE {SYSTEM.i386}
	INT 15
END Call15;

(** Shutdown - Terminate Oberon after executing all module terminators.  If code = 1, perform an APM power-down, if code = 2, perform a soft reboot, or else just switch off interrupts and loop endlessly. *)

PROCEDURE Shutdown*(code: LONGINT);
VAR m: Module;  t: LONGINT;
BEGIN
	shutdown := code;
	m := modules;
	WHILE m # NIL DO
		FinalizeModule(m);
		m := m.next
	END;
	IF code = 2 THEN
		SYSTEM.CLI();	(* Note: soft boot flag was set in InitMemory *)
		SYSTEM.PORTOUT(70H, 8FX);	(* reset type: p. 5-37 AT Tech. Ref. *)
		Wait;  SYSTEM.PORTOUT(71H, 0X);
		Wait;  SYSTEM.PORTOUT(70H, 0DX);
		Wait;  SYSTEM.PORTOUT(64H, 0FEX);	(* reset CPU *)
		code := 1000;  WHILE code # 0 DO DEC(code) END;
		Reboot
	ELSIF (code = 1) & (apmofs # -1) THEN
		t := ticks;  REPEAT UNTIL (ticks - t > TimeUnit);	(* give disk time to finish *)
		IF ~IgnoreTrap15 THEN RemoveIP(NIL, 15) END;	(* remove existing handler *)
		InstallIP(APMPowerOff, 15);	(* must be called in ring 0 *)
		Call15
	END;
	SYSTEM.CLI();
	LOOP END
END Shutdown;

(** Idle - Called to save power when system is idle. *)

PROCEDURE Idle*(code: LONGINT);	(* code currently unused *)
CODE {SYSTEM.i386}
	CMP powersave, 0
	JE exit
	MOV EAX, 4
	INT V86EnterInt	; call to HLT
exit:
END Idle;

(* ------------------------------------------------------------------------------------------------*)

(* Timer - Timer interrupt handler *)

PROCEDURE *TimerHandler;
CODE {SYSTEM.i386}
	STI
	INC ticks
	CMP timer, 0
	JE exit
	CALL timer
exit:
END TimerHandler;

(* InitTimer - Initialise the 8253 timer *)

PROCEDURE InitTimer;
CONST
	Div = (2*Rate + TimeUnit) DIV (2*TimeUnit);	(* timer clock divisor *)
BEGIN
	ticks := 0;
	ASSERT(Div < 10000H);
	SYSTEM.PORTOUT(43H, 34X);  Wait;	(* mode 2, rate generator *)
	SYSTEM.PORTOUT(40H, SYSTEM.VAL(CHAR, Div));  Wait;
	SYSTEM.PORTOUT(40H, SYSTEM.VAL(CHAR, ASH(Div, -8)));
	InstallIP(TimerHandler, IRQ+0)
END InitTimer;

(* InitTracing - Initialise trace info *)

PROCEDURE InitTracing(VAR top: LONGINT);
VAR s: ARRAY 32 OF CHAR;  ch: CHAR;
BEGIN
		(* initialize trace buffer *)
	GetConfig("TraceSize", s);  traceBufSize := StrToInt(s);
	IF traceBufSize <= 0 THEN traceBufSize := 2048 END;	(* default *)
	DEC(top, traceBufSize);  traceBufAdr := top;
		(* initialize console tracing *)
	GetConfig("TraceConsole", s);
	traceConsole := s[0] = "1";
	GetConfig("Init", s);
	IF s = "9090" THEN traceConsole := TRUE END;
	IF traceConsole THEN Fill4(DisplayBase, DisplayWidth*DisplayHeight DIV 4, 07200720H) END;
		(* initialize V24 tracing *)
	traceHead := 0;  traceTail := 0;  traceMark := 0;
	GetConfig("TracePort", s);  tbase := SHORT(StrToInt(s));
	CASE tbase OF
		1: tbase := 3F8H
		|2: tbase := 2F8H
		|3: tbase := 3E8H
		|4: tbase := 2E8H
		ELSE (* tbase = actual port number *)
	END;
	GetConfig("TraceBPS", s);  tspeed := StrToInt(s);
	IF (tspeed # 0) & (tbase = 0) THEN tbase := 3F8H
	ELSIF (tspeed = 0) & (tbase # 0) THEN tspeed := 19200
	END;
	IF tspeed > 0 THEN
		SYSTEM.PORTIN(tbase+4, ch);	(* check if port present *)
		IF SYSTEM.VAL(SET, LONG(ORD(ch))) * {5..7} # {} THEN tspeed := 0;  tbase := 0 END
	END;
		(* initialize Centronics tracing *)
	GetConfig("TracePrinter", s);  tlpt := StrToInt(s);
	IF (tlpt >= 1) & (tlpt <= 3) THEN tlpt := bioslpt[tlpt-1] END;
	GetConfig("TraceHeap", s);
	traceheap := SYSTEM.VAL(SET, StrToInt(s))
END InitTracing;

(* InitRuntime - Initialise runtime support. *)

PROCEDURE InitRuntime;
VAR
	newRec: PROCEDURE(VAR p: ADDRESS;  tag: Tag);
	newSys: PROCEDURE(VAR p: ADDRESS;  size: LONGINT);
	newArr: PROCEDURE(VAR p: ADDRESS;  eltag: Tag;  nofelem, nofdim: LONGINT);
BEGIN
	newRec := NewRec;	(* used in Modules and Types *)
	newSys := NewSys;
	newArr := NewArr;
	runtime[0] := SYSTEM.VAL(ADDRESS, newRec);	(* 253 *)
	runtime[1] := SYSTEM.VAL(ADDRESS, newSys);	(* 252 *)
	runtime[2] := SYSTEM.VAL(ADDRESS, newArr);	(* 251 *)
	IF V86 THEN
		IF V86EnterInt < 32 THEN RemoveIP(NIL, V86EnterInt) END;
		InstallIP(V86IntHandler, V86EnterInt);
		IF V86ExitInt < 32 THEN RemoveIP(NIL, V86ExitInt) END;
		InstallIP(V86Exit, V86ExitInt)
	END
END InitRuntime;

(* SwitchToLevel3 - Switch to user level stack and outer protection level *)

PROCEDURE -SwitchToLevel3(ss, sp, cs: LONGINT);
CODE {SYSTEM.i386}
	POP ECX	; cs
	POP EBX	; sp
	POP EAX	; ss
	POP EBP	; throw away module saved EBP
	MOV -8[EBX], 7FFFFFFFH	; EBP on new stack
	POP DWORD -4[EBX]	; transfer module ret addr to new stack
	LEA EBP, -8[EBX]	; new stack top
	PUSH EAX	; SS3
	PUSH EBP	; ESP3
	PUSHFD	; EFLAGS3
	PUSH ECX	; CS3
	CALL L1	; PUSH offset L1 (EIP3)
L1:
	ADD DWORD [ESP], 5
	IRETD
END SwitchToLevel3;

(* Init code called from OBL.  EAX = boot table offset.  2k stack is available. *)

BEGIN
	SYSTEM.GETREG(0, bt);	(* EAX from bootstrap loader *)
	SYSTEM.GETREG(6, kpar[0]);  SYSTEM.GETREG(7, kpar[1]);	(* par0 & par1 *)
	tspeed := 0;  pspeed := 0;  displayPos := 0;  traceBufAdr := 0;  timer := NIL;
	traceConsole := FALSE;  shutdown := 0;  tlpt := 0;
	ReadBootTable;	(* set memTop, pageheap, configadr *)
	CheckMemory;	(* adjust memTop *)
	pageheap0 := pageheap;  pageheap1 := 0;  heapTop := memTop;
	InitTracing(heapTop);
	version := "PC Native 24.08.2002";
	WriteString("ETH Oberon / ");  WriteString(version);  WriteLn;
	InitProcessor;  InitInterrupts;  InitTrapHandling;
	InitMemory;  InitHeap;
		(* switch to level 3 (after InitHeap inits StackOrg) *)
	SwitchToLevel3(UserStackSel, StackOrg, UserCodeSel);
		(* on new stack *)
	InitTimer;  (*InitBeeps;*)
	InitClock;  InitRuntime
END Kernel.

(*
TraceHeap:
0	1	NEW record
1	2	NEW array
2	4	SYSTEM.NEW
3	8	deallocate record
4	16	deallocate array
5	32	deallocate sysblk
6	64	finalisation
7	128	free pages
8	256	show free blocks
9	512	NewDMA
10	1024	before/after memory
11	2048	live/dead
16	65536	termhandlers
*)
BIER   \   "        g      d
     C  TextGadgets.NewStyleProc  