#   Oberon10.Scn.Fnt  T   T  MODULE PowerDoc; (*JG 11.8.98*)
  IMPORT Input, Display, Viewers, Files, Modules, Objects, Fonts, Texts, Oberon, Attributes, Links, Pictures, Rembrandt,
    Gadgets, Out;

(* Basic syntax

  CompilationUnit = { Lib | Doc }.
  Lib = "(" LIB name { Frame } ")" [ comment ].
  Doc = "(" DOC [ name ] { DocSpecs } ") [ comment ].
  DocSpecs = Meta | Frame | StreamDef.
  Meta = "(" META string ")" [ comment ].
  Frame = "(" FRAME [ name ] { FrameSpecs } ")" [ comment ].
  FrameSpecs = 
    Position | Size | BG | Object |
    Grid | Padding | Boundary | Separators |
    Frame | StreamDef | StreamImp |
    Declaration.
  Position = "(" POS num "@" num ")" [ comment ].
  Size = "(" SIZE num "@" num ")" [ comment ].
  Grid = "(" GRID { Extent } "@" { Extent } ")" [ comment ].
  Extent = num ":" ( num [ "%" ] | "*" ) [ comment ].
  Padding = "(" PAD num [ "@" num ] ")" [ comment ].
  Boundary = "(" BND num [ "@" num ] ")" [ comment ].
  Separators = "(" SEP num [ "@" num ] ")" [ comment ].
  BG = "(" BG name ")" [ comment ].
  Object = "(" OBJ name { Declaration } ")" [ comment ].
  StreamDef = "(" STD name Dest ")" [ comment ].
  StreamImp = "(" STI name Dest ")" [ comment ].
  Dest = "(" DST { name } ")" | "(" DSP [ num "@" num" ] ")" [ comment ].
  Declaration = "(" name Object ")" [ comment ]. *)

  CONST
    Inval = Texts.Inval; Name = Texts.Name; String = Texts.String; Int = Texts.Int; Real = Texts.Real; Char = Texts.Char;
    Obj = 100; Head = 101; Tail = 102; (*additional symbol classes*)
    MaxCells = 16; MaxLen = 50; MinInt = -32768; eps = 1.0E-6;
    CR = 0DX;

  TYPE
    Declaration = POINTER TO RECORD
      next: Declaration;
      class: INTEGER;
      obj: Object;
      i: LONGINT;
      x: REAL;
      s: ARRAY 32 OF CHAR;
      name: ARRAY 16 OF CHAR
    END;

    Scope = POINTER TO RECORD
      up: Scope;
      par: INTEGER;
      lead: INTEGER;
      lib: Objects.Library;
      col: INTEGER;
      voff: INTEGER;
      com: LONGINT (*command position*)
    END;

    WordReader = RECORD (Texts.Reader)
      text: Texts.Text;
      scope: Scope;
      pos: LONGINT;
      len, wid: INTEGER;
      nextCh: CHAR;
      str: ARRAY MaxLen OF CHAR
    END;

    RefPoint = POINTER TO RECORD
      next: RefPoint;
      scope: Scope;
      pos: LONGINT;
      Y: INTEGER
    END;

    Destination = POINTER TO RECORD
      next: Destination;
      frame: Frame
    END;

    Stream = POINTER TO RECORD
      next: Stream;
      text: Texts.Text;
      HPad, VPad: REAL;
      scope: Scope;
      pos: LONGINT;
      terminal: BOOLEAN;
      ref: RefPoint;
      dst, lastDst: Destination;
      name: ARRAY 32 OF CHAR
    END;

    Frame* = POINTER TO RECORD
      anc, next: Frame; (*ancestor, sibbling*)
      dsc, lastDsc: Frame; (*descendants*)
      dcl, lastDcl: Declaration; (*declarations*)
      std, lastStd: Stream; (*stream declarations*)
      sti, lastSti: Stream; (*stream imports*)
      R, C, RSpan, CSpan: INTEGER; (*location/size in terms of grid position*)
      X, Y, W, H: REAL; (*location/size*)
      obj: Object; (*contents object*)
      BGP: Pictures.Picture; (*background pattern*)
      HPad, VPad: REAL; (*padding*)
      HBnd, VBnd: REAL; (*boundary*)
      HSep, VSep: REAL; (*separating lines*)
      eofRows, eofCols: INTEGER; (*grid structure*)
      RP, CP: ARRAY 32 OF REAL; (*size/pos, > 0: abs, = 0: auto, < 0: rel*)
      name, BGPName: ARRAY 32 OF CHAR
    END;

    Object = POINTER TO RECORD
      next: Object;
      obx: Objects.Object; (*associated Oberon object*)
      attr, lastAttr: Declaration; (*attributes*)
      gen: ARRAY 32 OF CHAR (*generator*)
    END;

    Document* = POINTER TO RECORD (Frame)
      view: Viewer;
      meta: ARRAY 64 OF CHAR
    END;

    Viewer = POINTER TO RECORD (Viewers.Viewer)
      isOverlay: BOOLEAN;
      doc: Document;
      frame: Frame
    END;

    Context* = RECORD
      doc*: Document;
      frame*: Frame;
      text*: Texts.Text;
      pos*: LONGINT;
      keys*: SET;
      X*, Y*: INTEGER;
      XW, YW, WW: INTEGER
    END;

  VAR
    C*: Context; S: Texts.Scanner; 
    nofErr: INTEGER; ind: INTEGER; SW, SH: INTEGER; box: Fonts.Char;
    DefaultScope: Scope; BoxPat: ARRAY 12 OF SET;
    LibOp, DocOp, MetaOp, FrameOp, StrDefOp, StrImpOp, ObjOp, DstOp, DspOp,
    PosOp, SizeOp, GridOp, BGOp, ParOp, PadOp, BndOp, SepOp, FontOp, ColOp,
    LeadOp, ImpOp, CmdOp: ARRAY 4 OF CHAR;
    res: ARRAY 1000 OF CHAR; (*resources*)

(*** Error reports ***)

  PROCEDURE PErr (item: ARRAY OF CHAR);
  BEGIN INC(nofErr);
    Out.String("pos "); Out.Int(Texts.Pos(S), 1); Out.Char(" ");
    Out.String(item); Out.String(" expected"); Out.Ln;
    GetSym
  END PErr;

  PROCEDURE Err (msg: ARRAY OF CHAR);
  BEGIN INC(nofErr); Out.String(msg); Out.Ln
  END Err;

(*** Scanner support ***)

  PROCEDURE GetSym;
  BEGIN Texts.Scan(S);
    IF S.class = Char THEN
      IF S.c = "(" THEN Texts.Scan(S); S.class := Head
        ELSIF S.c = ")" THEN S.class := Tail
      END
    ELSIF S.class = Int THEN S.x := S.i
    END
  END GetSym;

  PROCEDURE Matching (VAR s, t: ARRAY OF CHAR): BOOLEAN;
    VAR i: INTEGER;
  BEGIN i := 0;
    WHILE (s[i] # 0X) & (t[i] # 0X) & (s[i] = t[i]) DO INC(i) END;
    RETURN t[i] = 0X
  END Matching;

  PROCEDURE CheckTail (item: ARRAY OF CHAR);
  BEGIN IF S.class = Tail THEN GetSym ELSE PErr(item) END;
    IF S.class = Name THEN GetSym END
  END CheckTail;

(*** Text Reader support ***)

  PROCEDURE ReadWord (VAR WR: WordReader);
    VAR i, w: INTEGER; ch: CHAR; obj: Objects.Object; 
  BEGIN Texts.Read(WR, ch);
    WHILE ~WR.eot & (ch <= " ") DO INC(WR.pos); Texts.Read(WR, ch) END;
    i := 0; w := 0;
    WHILE ~WR.eot & (ch > " ") & (i # MaxLen) DO WR.str[i] := ch;
      IF WR.scope.lib IS Fonts.Font THEN
        WR.scope.lib.GetObj(WR.scope.lib, ORD(ch), obj); w := w + obj(Fonts.Char).dx
      ELSE w := w + box.dx
      END;
      INC(i); Texts.Read(WR, ch)
    END;
    WR.len := i; WR.wid := w; WR.nextCh := ch
  END ReadWord;

  PROCEDURE ReadTextWord* (VAR WR: WordReader);
    VAR pos: LONGINT; pbal: INTEGER; ch: CHAR; fnt: Fonts.Font;
      local: Scope; cont, old, new: Frame; obj: Object;
  BEGIN
    WR.pos := Texts.Pos(WR); ReadWord(WR);
    IF WR.len = 2 THEN
      IF (WR.str[0] = "{") & (WR.str[1] = "%") THEN
        NEW(local); local^ := WR.scope^; local.up := WR.scope;
        WR.scope := local; pos := Texts.Pos(WR);
        Texts.OpenScanner(S, WR.text, pos); GetSym;
        LOOP
          IF S.class # Head THEN EXIT END;
          IF Matching(S.s, FontOp) THEN GetSym;
            IF S.class = Name THEN fnt := Fonts.This(S.s);
              IF fnt # NIL THEN WR.scope.lib := fnt END;
              GetSym (*tail*)
            END 
          ELSIF Matching(S.s, ColOp) THEN GetSym;
            IF S.class = Int THEN WR.scope.col := SHORT(S.i); GetSym END
          ELSIF Matching(S.s, ParOp) THEN GetSym;
            IF S.class = Int THEN WR.scope.par := SHORT(S.i); GetSym END
          ELSIF Matching(S.s, LeadOp) THEN GetSym;
            IF S.class = Int THEN WR.scope.lead := SHORT(S.i); GetSym END
          ELSIF Matching(S.s, CmdOp) THEN WR.scope.com := Texts.Pos(S);
            GetSym;
            IF S.class = Name THEN pbal := 1; 
              REPEAT Texts.Scan(S);
                IF S.class = Char THEN
                  IF S.c = "(" THEN INC(pbal) ELSIF S.c = ")" THEN DEC(pbal) END
                END
              UNTIL pbal = 0
            END
          END;
          pos := Texts.Pos(S); (*pos after tail*)
          GetSym
        END;
        Texts.OpenReader(WR, WR.text, pos); ReadTextWord(WR)
      ELSIF (WR.str[0] = "%") & (WR.str[1] = "}") THEN
        IF WR.scope.up # NIL THEN WR.scope := WR.scope.up END;
        ReadTextWord(WR)
      END
    END
  END ReadTextWord;

(*** Object support ***)

  PROCEDURE ModifySize (F: Display.Frame; W, H: INTEGER);
    VAR M: Display.ModifyMsg;
  BEGIN
    IF (F # NIL) & ((F.W # W) OR (F.H # H)) THEN
      M.id := Display.extend; M.mode := Display.state; M.F := F;
      M.X := F.X; M.Y := F.Y + F.H - H; M.W := W; M.H := H;
      M.dW := W - F.W; M.dH := H - F.H; M.dX := 0; M.dY := -M.dH;
      M.dlink := NIL; M.x := 0; M.y := 0; M.res := -1; Objects.Stamp(M);
      F.handle(F, M)
    END
  END ModifySize;

  PROCEDURE ConsumeObj (F, f: Display.Frame; u, v: INTEGER);
    VAR C: Display.ConsumeMsg;
  BEGIN
    IF (F # NIL) & (f # NIL) THEN f.slink := NIL;
      C.id := Display.drop; C.F := F; C.res := -1; C.dlink := NIL; C.x := 0; C.y := 0;
      C.u := u; C.v := v; C.obj := f; F.handle(F, C)
	END
  END ConsumeObj;

  PROCEDURE GetPublicObj (name: ARRAY OF CHAR; VAR obj: Objects.Object);
    VAR i, j, k, ref: INTEGER; lib: Objects.Library;
      objname: ARRAY 32 OF CHAR;
  BEGIN i := 0;
    WHILE (name[i] # 0X) & (name[i] # ".") DO INC(i); END;
    IF name[i] = "." THEN INC(i); j := i; k := 0;
      WHILE name[j] # 0X DO objname[k] := name[j]; INC(j); INC(k) END;
      objname[k] := 0X;
	  name[i] := "L"; name[i+1] := "i"; name[i+2] := "b"; name[i+3] := 0X;
	  lib := Objects.ThisLibrary(name);
	  IF lib # NIL THEN
		Objects.GetRef(lib.dict, objname, ref);
		IF ref # MIN(INTEGER) THEN lib.GetObj(lib, ref, obj); END
	  END
    ELSE obj := NIL
    END
  END GetPublicObj;

  PROCEDURE CopyPublicObj (name: ARRAY OF CHAR; deep: BOOLEAN): Objects.Object;
    VAR C: Objects.CopyMsg; obj: Objects.Object;
  BEGIN
    GetPublicObj(name, obj);
	IF obj # NIL THEN Objects.Stamp(C); C.obj := NIL;
      IF deep THEN C.id := Objects.deep ELSE C.id := Objects.shallow END;
      obj.handle(obj, C); obj := C.obj
    END;
    RETURN obj
  END CopyPublicObj;

  PROCEDURE CreateObj (gen: ARRAY OF CHAR): Objects.Object;
    VAR res: INTEGER;
  BEGIN Objects.NewObj := NIL;
    Oberon.Call(gen, Oberon.Par, FALSE, res); RETURN Objects.NewObj
  END CreateObj;

  PROCEDURE BindObj (obj: Objects.Object; lib: Objects.Library);
    VAR ref: INTEGER; M: Objects.BindMsg;
  BEGIN
    IF lib # NIL THEN 
      IF (obj.lib = NIL) OR (obj.lib.name[0] = 0X) & (obj.lib # lib) THEN lib.GenRef(lib, ref);
        IF ref >= 0 THEN lib.PutObj(lib, ref, obj); M.lib := lib; obj.handle(obj, M) END
      END
    END
  END BindObj;

  PROCEDURE Update (obj: Objects.Object);
    VAR MU: Gadgets.UpdateMsg; MD: Display.DisplayMsg;
  BEGIN
    IF obj IS Display.Frame THEN
      MD.device := Display.screen; MD.id := Display.full; MD.F := obj(Display.Frame); Display.Broadcast(MD)
	ELSE MU.obj := obj; MU.F := NIL; Display.Broadcast(MU)
    END
  END Update;

(*** Composer, seventh pass: display foreground ***)

  PROCEDURE INT (x: REAL): INTEGER;
  BEGIN RETURN SHORT(ENTIER(x + 0.5))
  END INT;

  PROCEDURE RFStream (str: Stream; frame: Frame; VAR WR: WordReader);
    VAR XS, XE, XC, YC, YE, lsp, i: INTEGER; fnt: Fonts.Font; char: Fonts.Char; obj: Objects.Object;
    ref, lastRef: RefPoint;
  BEGIN
    XS := INT(frame.X + str.HPad); XE := INT(frame.X + frame.W - str.HPad);
    YC := INT(frame.Y + frame.H - str.VPad); YE := INT(frame.Y + str.VPad);
    lastRef := NIL;
    IF WR.scope.lib IS Fonts.Font THEN
      fnt := WR.scope.lib(Fonts.Font); lsp := fnt.height + WR.scope.lead
    ELSE lsp := box.h + WR.scope.lead
    END;
    IF (WR.len = 2) & (WR.str[0] = "%") & (WR.str[1] = "\") THEN
      lsp := lsp + WR.scope.par; ReadTextWord(WR)
    END;
    WHILE ~WR.eot & (YC >= YE + lsp) DO YC := YC - lsp; XC := XS;
      NEW(ref); ref.Y := YC; ref.scope := WR.scope; ref.pos := WR.pos;
      IF lastRef # NIL THEN lastRef.next := ref ELSE str.ref := ref END;
      lastRef := ref;
      WHILE ~WR.eot & ((WR.len # 2) OR (WR.str[0] # "%") OR (WR.str[1] # "\")) & (XC + WR.wid <= XE) DO
        i := 0;
        WHILE i # WR.len DO
          IF WR.scope.lib IS Fonts.Font THEN
            WR.scope.lib.GetObj(WR.scope.lib, ORD(WR.str[i]), obj); char := obj(Fonts.Char)
          ELSE char := box
          END;
          Display.CopyPattern(WR.scope.col, char.pat, XC + char.x, YC + char.y, Display.paint);
          XC := XC + char.dx;
          INC(i)
        END;
        WR.lib.GetObj(WR.lib, ORD(" "), obj); char := obj(Fonts.Char);
        XC := XC + char.dx;
        ReadTextWord(WR)
      END;
      IF ~WR.eot THEN
        IF WR.scope.lib IS Fonts.Font THEN
          fnt := WR.scope.lib(Fonts.Font); lsp := fnt.height + WR.scope.lead
        ELSE lsp := box.h + WR.scope.lead
        END;
        IF (WR.len = 2) & (WR.str[0] = "%") & (WR.str[1] = "\") THEN
          lsp := lsp + WR.scope.par; ReadTextWord(WR)
        END
      END
    END
  END RFStream;

  PROCEDURE DFStream (str: Stream; frame: Frame; VAR WR: WordReader);
    VAR sti: Stream; dst: Destination;
  BEGIN sti := frame.sti;
    WHILE (sti # NIL) & (sti.name # str.name) DO sti := sti.next END;
    IF sti # NIL THEN
      sti.text := str.text; sti.scope := WR.scope; sti.pos := WR.pos; (*late eval*)
      IF sti.terminal THEN RFStream(sti, frame, WR)
      ELSE dst := sti.dst;
        WHILE ~WR.eot & (dst # NIL) DO DFStream(sti, dst.frame, WR); dst := dst.next END
      END
    END
  END DFStream;

  PROCEDURE DFFrame (frame: Frame);
    VAR dsc: Frame; str: Stream; dst: Destination; WR: WordReader; i, j: INTEGER;
      Mapped: ARRAY MaxCells, MaxCells OF BOOLEAN;
  BEGIN
    dsc := frame.dsc;
    WHILE dsc # NIL DO DFFrame(dsc); dsc := dsc.next END;
    IF frame.HBnd > 0 THEN
      Display.ReplConst(Display.FG, INT(frame.X), INT(frame.Y), INT(frame.W), INT(frame.HBnd), Display.replace);
      Display.ReplConst(Display.FG, INT(frame.X), INT(frame.Y) + INT(frame.H) - INT(frame.HBnd),
        INT(frame.W), INT(frame.HBnd), Display.replace)
    END;
    IF frame.VBnd > 0 THEN
      Display.ReplConst(Display.FG, INT(frame.X), INT(frame.Y), INT(frame.VBnd), INT(frame.H), Display.replace);
      Display.ReplConst(Display.FG, INT(frame.X) + INT(frame.W) - INT(frame.VBnd), INT(frame.Y),
        INT(frame.VBnd), INT(frame.H), Display.replace)
    END;
    IF (frame.HSep > 0) OR (frame.VSep > 0) THEN
      FOR i := 0 TO MaxCells-1 DO
        FOR j := 0 TO MaxCells-1 DO Mapped[i, j] := FALSE END
      END;
      dsc := frame.dsc;
      WHILE dsc # NIL DO
        FOR i := 0 TO dsc.RSpan-1 DO Mapped[dsc.R + i, dsc.C] := TRUE END;
        FOR j := 0 TO dsc.CSpan-1 DO Mapped[dsc.R, dsc.C + j] := TRUE END;
        dsc := dsc.next
      END;
      dsc := frame.dsc;
      WHILE dsc # NIL DO
        IF frame.HSep > 0 THEN
          Display.ReplConst(Display.FG, INT(dsc.X) - INT(frame.HPad/2), INT(dsc.Y) - INT(frame.VPad/2),
            INT(dsc.W) + INT(frame.HPad), INT(frame.HSep), Display.replace);
          IF ~Mapped[dsc.R + dsc.RSpan, dsc.C] THEN
            Display.ReplConst(Display.FG, INT(dsc.X) - INT(frame.HPad/2), INT(dsc.Y + dsc.H) + INT(frame.HPad/2),
              INT(dsc.W) + INT(frame.HPad), INT(frame.HSep), Display.replace)
          END
        END;
        IF frame.VSep > 0 THEN
          Display.ReplConst(Display.FG, INT(dsc.X) - INT(frame.HPad/2), INT(dsc.Y) - INT(frame.VPad/2),
            INT(frame.VSep), INT(dsc.H) + INT(frame.VPad), Display.replace);
          IF ~Mapped[dsc.R, dsc.C + dsc.CSpan] THEN
            Display.ReplConst(Display.FG, INT(dsc.X + dsc.W) + INT(frame.VPad/2), INT(dsc.Y) - INT(frame.VPad/2),
              INT(frame.VSep), INT(dsc.H) + INT(frame.VPad), Display.replace)
          END
        END;
        dsc := dsc.next
      END
    END;
    str := frame.std;
    WHILE str # NIL DO
      WR.text := str.text; WR.scope := str.scope;
      Texts.OpenReader(WR, str.text, str.pos); ReadTextWord(WR);
      IF str.terminal THEN RFStream(str, frame, WR)
      ELSE dst := str.dst;
        WHILE ~WR.eot & (dst # NIL) DO DFStream(str, dst.frame, WR); dst := dst.next END
      END;
      str := str.next
    END
  END DFFrame;

(*** Composer, sixth pass: display background ***)

  PROCEDURE DBFrame (frame: Frame);
    VAR dsc: Frame; X, Y, Xlim, Ylim, col: INTEGER;
  BEGIN
    IF frame.BGPName[0] # 0X THEN
      IF ("0" <= frame.BGPName[0]) & (frame.BGPName[0] <= "9") THEN
        col := ORD(frame.BGPName[0]) - ORD("0");
        IF ("0" <= frame.BGPName[1]) & (frame.BGPName[1] <= "9") THEN
          col := col*10 + ORD(frame.BGPName[1]) - ORD("0")
        END;
        Display.ReplConst(col, INT(frame.X), INT(frame.Y), INT(frame.W), INT(frame.H), Display.replace)
      ELSE (* replicate picture *)
        Y := INT(frame.Y); Ylim := Y + INT(frame.H);
        WHILE Y + frame.BGP.height <= Ylim DO X := INT(frame.X); Xlim := X + INT(frame.W);
          WHILE X + frame.BGP.width <= Xlim DO
            Pictures.DisplayBlock(frame.BGP, 0, 0, frame.BGP.width, frame.BGP.height, X, Y, Display.replace);
            X := X + frame.BGP.width
          END;
          IF X < Xlim THEN
            Pictures.DisplayBlock(frame.BGP, 0, 0, Xlim - X, frame.BGP.height, X, Y, Display.replace)
          END;
          Y := Y + frame.BGP.height
        END;
        IF Y < Ylim THEN X := INT(frame.X); Xlim := X + INT(frame.W);
          WHILE X + frame.BGP.width <= Xlim DO
            Pictures.DisplayBlock(frame.BGP, 0, 0, frame.BGP.width, Ylim - Y, X, Y, Display.replace);
            X := X + frame.BGP.width
          END;
          IF X < Xlim THEN
            Pictures.DisplayBlock(frame.BGP, 0, 0, Xlim - X, Ylim - Y, X, Y, Display.replace)
          END
        END
      END
    END;
    dsc := frame.dsc;
    WHILE dsc # NIL DO DBFrame(dsc); dsc := dsc.next END
  END DBFrame;

(*** Composer, fifth pass: building/composing ***)

  PROCEDURE BFrame (cont, frame: Frame);
    VAR dsc: Frame;
  BEGIN
    IF frame.obj # NIL THEN 
      ModifySize(frame.obj.obx(Display.Frame), INT(frame.W), INT(frame.H)); dsc := frame.dsc;
      WHILE dsc # NIL DO BFrame(frame, dsc); dsc := dsc.next END;
      IF (cont # NIL) & (cont.obj # NIL) THEN
        ConsumeObj(cont.obj.obx(Display.Frame), frame.obj.obx(Display.Frame),
        INT(frame.X) - INT(cont.X), INT(frame.Y) - INT(cont.Y + cont.H) + 2 (*!*))
      END
    ELSE dsc := frame.dsc;
      WHILE dsc # NIL DO BFrame(cont, dsc); dsc := dsc.next END
    END
  END BFrame;

(*** Composer, fourth pass: absolute locating ***)

  PROCEDURE RStream (str: Stream; frame: Frame; VAR WR: WordReader);
    VAR XS, XE, XC, YC, YE, lsp, i: INTEGER; fnt: Fonts.Font; char: Fonts.Char; obj: Objects.Object;
  BEGIN
    XS := INT(frame.X + str.HPad); XE := INT(frame.X + frame.W - str.HPad);
    YC := INT(frame.Y + frame.H - str.VPad); YE := INT(frame.Y + str.VPad);
    IF WR.scope.lib IS Fonts.Font THEN
      fnt := WR.scope.lib(Fonts.Font); lsp := fnt.height + WR.scope.lead
    ELSE lsp := box.h + WR.scope.lead
    END;
    IF (WR.len = 2) & (WR.str[0] = "%") & (WR.str[1] = "\") THEN
      lsp := lsp + WR.scope.par; ReadTextWord(WR)
    END;
    WHILE ~WR.eot & (YC >= YE + lsp) DO YC := YC - lsp; XC := XS;
      WHILE ~WR.eot & ((WR.len # 2) OR (WR.str[0] # "%") OR (WR.str[1] # "\")) & (XC + WR.wid <= XE) DO
        i := 0;
        WHILE i # WR.len DO
          IF WR.scope.lib IS Fonts.Font THEN
            WR.scope.lib.GetObj(WR.scope.lib, ORD(WR.str[i]), obj); char := obj(Fonts.Char)
          ELSE char := box
          END;
          XC := XC + char.dx;
          INC(i)
        END;
        WR.lib.GetObj(WR.lib, ORD(" "), obj); char := obj(Fonts.Char);
        XC := XC + char.dx;
        ReadTextWord(WR)
      END;
      IF ~WR.eot THEN
        IF WR.scope.lib IS Fonts.Font THEN
          fnt := WR.scope.lib(Fonts.Font); lsp := fnt.height + WR.scope.lead
        ELSE lsp := box.h + WR.scope.lead
        END;
        IF (WR.len = 2) & (WR.str[0] = "%") & (WR.str[1] = "\") THEN
          lsp := lsp + WR.scope.par; ReadTextWord(WR)
        END
      END
    END
  END RStream;

  PROCEDURE LStream (str: Stream; frame: Frame; VAR WR: WordReader);
    VAR sti: Stream; dst: Destination;
  BEGIN sti := frame.sti;
    WHILE (sti # NIL) & (sti.name # str.name) DO sti := sti.next END;
    IF sti # NIL THEN
      sti.text := str.text; sti.scope := WR.scope; sti.pos := WR.pos;
      IF sti.terminal THEN RStream(sti, frame, WR)
      ELSE dst := sti.dst;
        WHILE ~WR.eot & (dst # NIL) DO LStream(sti, dst.frame, WR); dst := dst.next END
      END
    END
  END LStream;

  PROCEDURE LFrame (frame: Frame);
    VAR dsc: Frame;
  BEGIN dsc := frame.dsc;
    WHILE dsc # NIL DO
      dsc.X := frame.X + frame.HPad + dsc.X; dsc.Y := frame.Y + frame.VPad + dsc.Y; LFrame(dsc);
      dsc := dsc.next
    END
  END LFrame;

(*** Composer, third pass: layout computation ***)

  PROCEDURE CFrameW (frame: Frame);
    VAR dsc: Frame; C, i, j: INTEGER; CP: ARRAY 16 OF REAL;
  BEGIN
    IF frame.CP[1] = 0 THEN (*auto*) dsc := frame.dsc;
      WHILE dsc # NIL DO CFrameW(dsc); dsc := dsc.next END;
      i := 1;
      WHILE i <= frame.eofCols DO CP[i] := 0; INC(i) END;
      i := 1;
      WHILE i # frame.eofCols DO dsc := frame.dsc;
        WHILE dsc # NIL DO
          IF dsc.C = i THEN j := i + dsc.CSpan;
            IF CP[j] < CP[i] + frame.HPad + dsc.W THEN CP[j] := CP[i] + frame.HPad + dsc.W END
          END;
          dsc := dsc.next
        END;
        INC(i)
      END;
      IF frame.W = 0 THEN frame.W := CP[frame.eofCols] + frame.HPad END
    ELSE
      IF frame.CP[1] > 0 THEN (*abs*) CP[1] := 0; i := 1;
        WHILE i # frame.eofCols DO CP[i+1] := CP[i] + frame.HPad + frame.CP[i] ; INC(i) END;
        IF frame.W = 0 THEN frame.W := CP[frame.eofCols] - frame.HPad END
      ELSIF frame.CP[1] < 0 THEN (*rel*)
        IF (frame.W = 0) & (frame.obj # NIL) THEN frame.W := frame.obj.obx(Display.Frame).W END;
        CP[1] := 0; i := 1;
        IF frame.CP[1] = MinInt THEN
          WHILE i # frame.eofCols DO
            CP[i+1] := CP[i] + 1/(frame.eofCols-1)*(frame.W - frame.HPad); INC(i)
          END
        ELSE
          WHILE i # frame.eofCols DO
            CP[i+1] := CP[i] - frame.CP[i]*(frame.W - frame.HPad); INC(i)
          END
        END
      END;
      dsc := frame.dsc;
      WHILE dsc # NIL DO
        dsc.W := CP[dsc.C + dsc.CSpan] - CP[dsc.C] - frame.HPad; CFrameW(dsc); dsc := dsc.next
      END
    END;
    dsc := frame.dsc;
    WHILE dsc # NIL DO dsc.X := CP[dsc.C]; dsc := dsc.next END
  END CFrameW;

  PROCEDURE CFrameH (frame: Frame);
    VAR dsc: Frame; R, i, j: INTEGER; RP: ARRAY 16 OF REAL;
  BEGIN
    IF frame.RP[1] = 0 THEN (*auto*) dsc := frame.dsc;
      WHILE dsc # NIL DO CFrameH(dsc); dsc := dsc.next END;
      i := 1;
      WHILE i <= frame.eofRows DO RP[i] := 0; INC(i) END;
      i := 1;
      WHILE i # frame.eofRows DO dsc := frame.dsc;
        WHILE dsc # NIL DO
          IF dsc.R = i THEN j := i + dsc.RSpan;
            IF RP[j] < RP[i] + frame.VPad + dsc.H THEN RP[j] := RP[i] + frame.VPad + dsc.H END
          END;
          dsc := dsc.next
        END;
        INC(i)
      END;
      IF frame.H = 0 THEN frame.H := RP[frame.eofRows] + frame.VPad END
    ELSE
      IF frame.RP[1] > 0 THEN (*abs*) RP[1] := 0; i := 1;
        WHILE i # frame.eofRows DO RP[i+1] := RP[i] + frame.VPad + frame.RP[i]; INC(i) END;
        IF frame.H = 0 THEN frame.H := RP[frame.eofRows] - frame.VPad END
      ELSIF frame.RP[1] < 0 THEN (*rel*)
        IF (frame.H = 0) & (frame.obj # NIL) THEN frame.H := frame.obj.obx(Display.Frame).H END;
        RP[1] := 0; i := 1;
        IF frame.RP[1] = MinInt THEN
          WHILE i # frame.eofRows DO
            RP[i+1] := RP[i] + 1/(frame.eofRows-1)*(frame.H - frame.VPad); INC(i)
          END
        ELSE
          WHILE i # frame.eofRows DO
            RP[i+1] := RP[i] - frame.RP[i]*(frame.H - frame.VPad); INC(i)
          END
        END
      END;
      dsc := frame.dsc;
      WHILE dsc # NIL DO
        dsc.H := RP[dsc.R + dsc.RSpan] - RP[dsc.R] - frame.VPad; CFrameH(dsc); dsc := dsc.next
      END
    END;
    dsc := frame.dsc;
    WHILE dsc # NIL DO dsc.Y := RP[dsc.R]; dsc := dsc.next END
  END CFrameH;

(*** Composer, second pass: object generation ***)

  PROCEDURE Find (scope: Frame; VAR name: ARRAY OF CHAR; VAR dcl: Declaration);
  BEGIN
    REPEAT dcl := scope.dcl;
      WHILE (dcl # NIL) & (dcl.name # name) DO dcl := dcl.next END;
      scope := scope.anc
    UNTIL (dcl # NIL) OR (scope = NIL)
  END Find;

  PROCEDURE ExtensionIS (VAR name: ARRAY OF CHAR; ext: ARRAY OF CHAR): BOOLEAN;
    VAR i, j: INTEGER;
  BEGIN i := 0;
    WHILE (name[i] # 0X) & (name[i] # ".") DO INC(i) END;
    IF name[i] = "." THEN INC(i) END;
    j := 0;
    WHILE (name[i] # 0X) & (ext[j] # 0X) & (name[i] = ext[j]) DO INC(i); INC(j) END;
    RETURN name[i] = ext[j]
  END ExtensionIS;

  PROCEDURE New (VAR gen: ARRAY OF CHAR): Objects.Object;
    VAR obx: Objects.Object; F: Rembrandt.Frame; P: Pictures.Picture; M: Objects.AttrMsg;
  BEGIN
    IF ExtensionIS(gen, "Pict") THEN
      NEW(P); Pictures.Open(P, gen, TRUE);
      IF (P.width > 0) & (P.height > 0) THEN NEW(F); Rembrandt.NewP(F, P); obx := F
        ELSE obx := NIL
      END
    ELSE obx := CopyPublicObj(gen, TRUE);
      IF obx = NIL THEN obx := CreateObj(gen) END
    END;
    IF obx = NIL THEN Err("object generation failed") END;
    RETURN obx
  END New;

  PROCEDURE SetAttr (obx: Objects.Object; VAR name: ARRAY OF CHAR; VAR dcl: Declaration);
  BEGIN 
    IF dcl.class = String THEN Attributes.SetString(obx, name, dcl.s)
      ELSIF dcl.class = Int THEN Attributes.SetInt(obx, name, dcl.i)
      ELSIF dcl.class = Real THEN Attributes.SetReal(obx, name, dcl.x)
      ELSIF dcl.class = Obj THEN Links.SetLink(obx, name, dcl.obj.obx)
    END
  END SetAttr;

  PROCEDURE GObject (scope: Frame; obj: Object);
    VAR attr, dcl: Declaration;
  BEGIN
    obj.obx := New(obj.gen);
    IF obj.obx # NIL THEN attr := obj.attr;
      WHILE attr # NIL DO dcl := NIL;
        IF attr.class = Name THEN Find(scope, attr.s, dcl);
          IF dcl # NIL THEN SetAttr(obj.obx, attr.name, dcl)
            ELSE Err("attribute not found")
          END
        ELSE SetAttr(obj.obx, attr.name, attr)
        END;
        attr := attr.next
      END
    END
  END GObject;

  PROCEDURE GStream (str: Stream);
    VAR f: Files.File; len: LONGINT;
  BEGIN f := Files.Old(str.name);
    IF f # NIL THEN NEW(str.text); Texts.Load(str.text, f, 1, len)
      ELSE Err("invalid text name")
    END
  END GStream;

  PROCEDURE GFrame (frame: Frame);
    VAR dcl: Declaration; dsc: Frame; str: Stream;
  BEGIN
    IF frame.obj # NIL THEN GObject(frame, frame.obj);
      IF (frame.obj.obx # NIL) & ~(frame.obj.obx IS Display.Frame) THEN
        Err("not a display frame")
      END
    END;
    IF frame.BGPName[0] # 0X THEN
      IF ("0" > frame.BGPName[0]) OR (frame.BGPName[0] > "9") THEN
        NEW(frame.BGP); Pictures.Open(frame.BGP, frame.BGPName, TRUE);
        IF (frame.BGP.width = 0) OR (frame.BGP.height = 0) THEN Err("invalid picture") END
      END
    END;
    dcl := frame.dcl;
    WHILE (nofErr = 0) & (dcl # NIL) DO
      IF dcl.class = Obj THEN GObject(frame, dcl.obj) END;
      dcl := dcl.next
    END;
    dsc := frame.dsc;
    WHILE (nofErr = 0) & (dsc # NIL) DO GFrame(dsc); dsc := dsc.next END;
    str := frame.std;
    WHILE str # NIL DO GStream(str); str := str.next END
  END GFrame;

(*** Composer, first pass: parsing & internalization ***)

  PROCEDURE PStream (anc: Frame; VAR str: Stream);
    VAR dst: Destination; frame: Frame;
  BEGIN NEW(str); GetSym;
    IF S.class = Name THEN COPY(S.s, str.name); GetSym;
      IF Matching(S.s, DstOp) THEN GetSym;
        WHILE S.class = Name DO NEW(dst); frame := anc.dsc;
          WHILE (frame # NIL) & (frame.name # S.s) DO frame := frame.next END;
          IF frame # NIL THEN dst.frame := frame ELSE PErr("frameref") END;
          IF str.lastDst # NIL THEN str.lastDst.next := dst ELSE str.dst := dst END;
          str.lastDst := dst; GetSym
        END;
        CheckTail(") DST")
      ELSIF Matching(S.s, DspOp) THEN str.terminal := TRUE; GetSym;
        IF (S.class = Int) OR (S.class = Real) THEN str.HPad := S.x; GetSym;
          IF (S.class = Char) & (S.c = "@") THEN GetSym;
            IF (S.class = Int) OR (S.class = Real) THEN str.VPad := S.x; GetSym
              ELSE PErr("VPad")
            END
          ELSE str.VPad := str.HPad
          END
        END;
        CheckTail(") DSP")
      ELSE PErr("stream operation")
      END
    ELSE PErr("name")
    END;
    CheckTail(") STR")
  END PStream;

  PROCEDURE PDeclaration* (VAR dcl: Declaration);
  BEGIN
    NEW(dcl); COPY(S.s, dcl.name); GetSym;
    IF S.class = Int THEN dcl.class := Int; dcl.i := S.i; GetSym
      ELSIF S.class = Real THEN dcl.class := Real; dcl.x := S.x; GetSym
      ELSIF S.class = String THEN dcl.class := String; COPY(S.s, dcl.s); GetSym
      ELSIF S.class = Name THEN dcl.class := Name; COPY(S.s, dcl.s); GetSym
      ELSIF (S.class = Head) & Matching(S.s, ObjOp) THEN dcl.class := Obj; PObj(dcl.obj)
      ELSE PErr("dcl")
    END;
    CheckTail(") Declaration")
  END PDeclaration;

  PROCEDURE PObj* (VAR obj: Object);
    VAR attr: Declaration;
  BEGIN NEW(obj); GetSym;
    IF S.class = Name THEN COPY(S.s, obj.gen); GetSym ELSE PErr("gen") END;
    WHILE S.class = Head DO PDeclaration(attr);
      IF obj.lastAttr # NIL THEN obj.lastAttr.next := attr ELSE obj.attr := attr END;
      obj.lastAttr := attr
    END;
    CheckTail(") OBJ")
  END PObj;

  PROCEDURE PFrame* (anc: Frame; VAR frame: Frame);
    VAR R, C, rep, i: INTEGER; size: REAL; dcl: Declaration; dsc: Frame; str: Stream;
  BEGIN
    NEW(frame); frame.anc := anc; GetSym;
    IF S.class = Name THEN COPY(S.s, frame.name); GetSym END;
    WHILE S.class = Head DO
      IF Matching(S.s, PosOp) THEN GetSym;
        IF S.class = Int THEN frame.R := SHORT(S.i); GetSym;
          IF (S.class = Char) & (S.c = ":") THEN GetSym;
            IF S.class = Int THEN frame.RSpan := SHORT(S.i); GetSym
              ELSE PErr("row span")
            END
          END
        ELSE PErr("R")
        END;
        IF (S.class = Char) & (S.c = "@") THEN GetSym ELSE PErr("@") END;
        IF S.class = Int THEN frame.C := SHORT(S.i); GetSym;
          IF (S.class = Char) & (S.c = ":") THEN GetSym;
            IF S.class = Int THEN frame.CSpan := SHORT(S.i); GetSym
              ELSE PErr("col span")
            END
          END
        ELSE PErr("C")
        END;
        CheckTail(") POS")
      ELSIF Matching(S.s, SizeOp) THEN GetSym;
        IF (S.class = Int) OR (S.class = Real) THEN frame.W := S.x; GetSym;
          IF (S.class = Char) & (S.c = "@") THEN GetSym;
            IF (S.class = Int) OR (S.class = Real) THEN frame.H := S.x; GetSym
              ELSE PErr("H")
            END
          ELSE frame.H := frame.W
          END
        ELSE PErr("W")
        END;
        CheckTail(") SIZE")
      ELSIF Matching(S.s, GridOp) THEN GetSym; i:= 1;
        WHILE S.class = Int DO rep := SHORT(S.i); GetSym;
          IF (S.class = Char) & (S.c = ":") THEN GetSym;
            IF (S.class = Int) OR (S.class = Real) THEN size := S.x ELSE size := 0 END;
            GetSym;
            IF (S.class = Char) & (S.c = "%") THEN size := -size/100; GetSym END
          ELSE size := MinInt
          END;
          WHILE rep # 0 DO frame.RP[i] := size; INC(i); DEC(rep) END
        END;
        frame.eofRows := i;
        IF (S.class = Char) & (S.c = "@") THEN GetSym ELSE PErr("@") END;
        i := 1;
        WHILE S.class = Int DO rep := SHORT(S.i); GetSym;
          IF (S.class = Char) & (S.c = ":") THEN GetSym;
            IF (S.class = Int) OR (S.class = Real) THEN size := S.x ELSE size := 0 END;
            GetSym;
            IF (S.class = Char) & (S.c = "%") THEN size := -size/100; GetSym END
          ELSE size := MinInt
          END;
          WHILE rep # 0 DO frame.CP[i] := size; INC(i); DEC(rep) END
        END;
        frame.eofCols := i;
        CheckTail(") GRID")
      ELSIF Matching(S.s, BGOp) THEN GetSym;
        IF (S.class = Name) OR (S.class = String) THEN COPY(S.s, frame.BGPName); GetSym
          ELSE PErr("BGName")
        END;
        CheckTail(") BG")
      ELSIF Matching(S.s, PadOp) THEN GetSym;
        IF (S.class = Int) OR (S.class = Real) THEN frame.HPad := S.x; GetSym;
          IF (S.class = Char) & (S.c = "@") THEN GetSym;
            IF (S.class = Int) OR (S.class = Real) THEN frame.VPad := S.x; GetSym
              ELSE PErr("VPad")
            END
          ELSE frame.VPad := frame.HPad
          END
        ELSE PErr("HPad")
        END;
        CheckTail(") PAD")
      ELSIF Matching(S.s, BndOp) THEN GetSym;
        IF (S.class = Int) OR (S.class = Real) THEN frame.HBnd := S.x; GetSym;
          IF (S.class = Char) & (S.c = "@") THEN GetSym;
            IF (S.class = Int) OR (S.class = Real) THEN frame.VBnd := S.x; GetSym
              ELSE PErr("VBnd")
            END
          ELSE frame.VBnd := frame.HBnd
          END
        ELSE PErr("HBnd")
        END;
        CheckTail(") BND")
      ELSIF Matching(S.s, SepOp) THEN GetSym;
        IF (S.class = Int) OR (S.class = Real) THEN frame.HSep := S.x; GetSym;
          IF (S.class = Char) & (S.c = "@") THEN GetSym;
            IF (S.class = Int) OR (S.class = Real) THEN frame.VSep := S.x; GetSym
              ELSE PErr("VSep")
            END
          ELSE frame.VSep := frame.HSep
          END
        ELSE PErr("HSep")
        END;
        CheckTail(") SEP")
      ELSIF Matching(S.s, ObjOp) THEN PObj(frame.obj)
      ELSIF Matching(S.s, FrameOp) THEN PFrame(frame, dsc);
        IF frame.lastDsc # NIL THEN frame.lastDsc.next := dsc ELSE frame.dsc := dsc END;
        frame.lastDsc := dsc
      ELSIF Matching(S.s, StrDefOp) THEN PStream(frame, str);
        str.pos := 0; str.scope := DefaultScope;
        IF frame.lastStd # NIL THEN frame.lastStd.next := str ELSE frame.std := str END;
        frame.lastStd := str
      ELSIF Matching(S.s, StrImpOp) THEN PStream(frame, str);
        IF frame.lastSti # NIL THEN frame.lastSti.next := str ELSE frame.sti := str END;
        frame.lastSti := str
      ELSE PDeclaration(dcl);
        IF frame.lastDcl # NIL THEN frame.lastDcl.next := dcl ELSE frame.dcl := dcl END;
        frame.lastDcl := dcl
      END
    END;
    IF frame.eofRows <= 0 THEN frame.eofRows := 1; frame.RP[1] := -1 END;
    IF frame.eofCols <= 0 THEN frame.eofCols := 1; frame.CP[1] := -1 END;
    R := 1; C := 1; dsc := frame.dsc;
    WHILE dsc # NIL DO
      IF dsc.R <= 0 THEN dsc.R := R END;
      IF dsc.C <= 0 THEN dsc.C := C END;
      IF dsc.RSpan <= 0 THEN dsc.RSpan := 1 END;
      IF dsc.CSpan <= 0 THEN dsc.CSpan := 1 END;
      IF dsc.C + dsc.CSpan < frame.eofCols THEN R := dsc.R; C := dsc.C + dsc.CSpan
        ELSE R := dsc.R + 1; C := 1
      END;
      dsc := dsc.next
    END;
    CheckTail(") FRAME")
  END PFrame;

  PROCEDURE PDoc (VAR doc: Document);
    VAR frame: Frame; str: Stream; obj: Object;
  BEGIN NEW(doc); GetSym;
    IF S.class = Name THEN COPY(S.s, doc.name); GetSym END;
    WHILE S.class = Head DO
      IF Matching(S.s, FrameOp) THEN PFrame(doc, frame);
        IF (frame.W = 0) & (frame.H = 0) & (frame.obj = NIL) THEN
          NEW(obj); obj.gen := "PowerDoc.NewView";
          frame.obj := obj; frame.W := SW; frame.H := SH;
          IF doc.lastDsc # NIL THEN doc.lastDsc.next := frame ELSE doc.dsc := frame END;
          doc.lastDsc := frame
        ELSE PErr("page frame")
        END
      ELSIF Matching(S.s, StrDefOp) THEN PStream(doc, str);
        IF doc.lastStd # NIL THEN doc.lastStd.next := str ELSE doc.std := str END;
        doc.lastStd := str
      ELSIF Matching(S.s, MetaOp) THEN GetSym;
        IF S.class = String THEN COPY(S.s, doc.meta); GetSym; CheckTail("MET")
          ELSE PErr("meta")
        END
      ELSE PErr("DocSpec")
      END
    END 
  END PDoc;

(*** Display handling ***)

  PROCEDURE Inside (XP, YP, X, Y, W, H: REAL): BOOLEAN;
  BEGIN RETURN (XP >= X) & (YP >= Y) & (XP < X + W) & (YP < Y + H)
  END Inside;

  PROCEDURE HandleFiller (V: Objects.Object; VAR M: Objects.ObjMsg);
  BEGIN
    WITH V: Viewers.Viewer DO
      IF M IS Display.ModifyMsg THEN
        WITH M: Display.ModifyMsg DO
          IF (M.F = V) & (M.id = Display.extend) THEN
            Display.ReplConst(12 (*Display.BG*), V.X, M.Y, V.W, V.Y - M.Y, 0)
          END
        END
      END
    END
  END HandleFiller;

  PROCEDURE ToLocation (V: Viewers.Viewer; VAR M: Oberon.InputMsg);
    VAR dsc: Display.Frame; X, Y: INTEGER;
  BEGIN
    X := M.X - V.X; Y := M.Y - (V.Y + V.H - 1); dsc := V.dsc;
    WHILE (dsc # NIL) & ~Inside(X, Y, dsc.X, dsc.Y, dsc.W, dsc.H) DO dsc := dsc.next END;
    IF dsc # NIL THEN M.x := V.X; M.y := V.Y + V.H - 1; M.dlink := V; dsc.handle(dsc, M) END
  END ToLocation;

  PROCEDURE OMToDescendants (V: Viewers.Viewer; VAR M: Objects.ObjMsg);
    VAR dsc: Display.Frame;
  BEGIN dsc := V.dsc;
     WHILE dsc # NIL DO dsc.handle(dsc, M); dsc := dsc.next END
  END OMToDescendants;

  PROCEDURE FMToDescendants (V: Viewers.Viewer; VAR M: Display.FrameMsg);
    VAR dsc: Display.Frame;
  BEGIN dsc := V.dsc;
    WHILE dsc # NIL DO
      M.x := V.X; M.y := V.Y + V.H - 1; M.dlink := V; dsc.handle(dsc, M); dsc := dsc.next
    END
  END FMToDescendants;

  PROCEDURE Copy (V: Viewer; VAR V1: Viewer);
  BEGIN NEW(V1); V1.handle := V.handle; V1.frame := V.frame
  END Copy;

  PROCEDURE Consume (V: Viewers.Viewer; X, Y: INTEGER; F: Display.Frame);
  BEGIN F.X := X; F.Y := Y; F.next := V.dsc; V.dsc := F
  END Consume;

  PROCEDURE Restore (V: Viewer);
    VAR dsc: Display.Frame; MC: Display.ControlMsg; MD: Display.DisplayMsg;
      str: Stream; dst: Destination; WR: WordReader;
  BEGIN
    Oberon.RemoveMarks(V.X, V.Y, V.W, V.H);
    DBFrame(V.frame); dsc := V.dsc;
    WHILE dsc # NIL DO
      Objects.Stamp(MC); MC.id := Display.restore; MC.F := dsc; MC.res := -1;
      MC.x := V.X; MC.y := V.Y + V.H - 1; MC.dlink := V; dsc.handle(dsc, MC);
      Objects.Stamp(MD); MD.id := Display.full; MD.device := Display.screen; MD.F := dsc; MD.res := -1;
      MD.x := V.X; MD.y := V.Y + V.H - 1; MD.dlink := V; dsc.handle(dsc, MD);
      dsc := dsc.next
    END;
    DFFrame(V.frame);
    str := V.frame.sti;
    WHILE str # NIL DO
      WR.text := str.text; WR.scope := str.scope;
      Texts.OpenReader(WR, str.text, str.pos); ReadTextWord(WR);
      dst := str.dst;
      WHILE ~WR.eot & (dst # NIL) DO DFStream(str, dst.frame, WR); dst := dst.next END;
      str := str.next
    END
  END Restore;

(*** Frame Scope Support ***)

  PROCEDURE Container* (scope: Frame): Frame;
  BEGIN
    WHILE scope.obj = NIL DO scope := scope.anc END;
    RETURN scope
  END Container;

  PROCEDURE ThisFrame* (scope: Frame; VAR name: ARRAY OF CHAR): Frame;
    VAR frame, dsc: Frame;
  BEGIN frame := NIL;
    WHILE (scope # NIL) & (frame = NIL) DO dsc := scope.dsc;
      WHILE (dsc # NIL) & (frame = NIL) DO
        IF dsc.name = name THEN frame := dsc END;
        dsc := dsc.next
      END;
      scope := scope.anc
    END;
    RETURN frame
  END ThisFrame;

  PROCEDURE GetCarrier* (scope: Frame; VAR obj: Object);
  BEGIN
    IF scope.obj # NIL THEN scope.obj.next := obj; obj := scope.obj
    ELSE scope := scope.dsc;
      WHILE scope # NIL DO GetCarrier(scope, obj); scope := scope.next END
    END
  END GetCarrier;

(*** Command Support ***)

  PROCEDURE FlipMark*;
  BEGIN Display.ReplConst(Display.FG, C.XW, C.YW, C.WW, 2, Display.invert)
  END FlipMark;

  PROCEDURE Release* (VAR keysum: SET);
    VAR keys: SET; X, Y: INTEGER;
  BEGIN keysum := {}; Input.Mouse(keys, X, Y);
    WHILE keys # {} DO keysum := keysum + keys;  Input.Mouse(keys, X, Y) END
  END Release;

  PROCEDURE Apply*;
    VAR keysum: SET; frame: Frame; obj: Object; attr: Declaration;
  BEGIN
    FlipMark; Release(keysum); FlipMark;
    IF keysum # {0, 1, 2} THEN
      IF S.class = Name THEN
        frame := ThisFrame(C.frame, S.s);
        IF (frame # NIL) & (frame.sti = NIL) THEN obj := NIL; GetCarrier(frame, obj);
          IF obj # NIL THEN
            WHILE S.class = Head DO GetSym; PDeclaration(attr); SetAttr(obj.obx, attr.name, attr) END;
            Update(obj.obx)
          END
        END
      END
    END
  END Apply;

  PROCEDURE Replace*;
    VAR keysum: SET; cont, old, new: Frame; obj: Object;
      F: Display.Frame; MC: Display.ControlMsg; MD: Display.DisplayMsg;
  BEGIN
    FlipMark; Release(keysum); FlipMark;
    IF keysum # {0, 1, 2} THEN GetSym;
      IF S.class = Name THEN
        old := ThisFrame(C.frame, S.s);
        IF (old # NIL) & (old.sti = NIL) THEN obj := NIL; GetCarrier(old, obj);
          IF obj = NIL THEN nofErr := 0; GetSym; PFrame(NIL, new);
            IF (nofErr = 0) & (new.sti = NIL) THEN GetCarrier(new, obj);
              new.W := old.W; new.H := old.H; GFrame(new);
              IF nofErr = 0 THEN
                cont := Container(old); new.X := old.X; new.Y := old.Y;
                Oberon.RemoveMarks(INT(new.X), INT(new.Y), INT(new.W), INT(new.H));
                CFrameW(new); CFrameH(new); LFrame(new); BFrame(cont, new); DBFrame(new);
                 WHILE obj # NIL DO F := obj.obx(Display.Frame);
                   Objects.Stamp(MC); MC.id := Display.restore; MC.F := F; MC.res := -1;
                   MC.x := INT(cont.X); MC.y := INT(cont.Y + cont.H) - 2 (*!*); MC.dlink := NIL; F.handle(F, MC);
                   Objects.Stamp(MD); MD.id := Display.full; MD.device := Display.screen; MD.F := F; MD.res := -1;
                   MD.x := INT(cont.X); MD.y := INT(cont.Y + cont.H) - 2 (*!*); MD.dlink := NIL; F.handle(F, MD);
                   obj := obj.next
                 END;
                 DFFrame(new);
                 new.anc := old.anc; new.next := old.next; old^ := new^
               END
             END
           END
        END
      END
    END
  END Replace;

  PROCEDURE Pop*;
    VAR keysum: SET; frame: Frame; M: Oberon.InputMsg;
  BEGIN
    nofErr := 0; GetSym; PFrame(NIL, frame);
    IF nofErr = 0 THEN GFrame(frame);
      IF nofErr = 0 THEN
        IF (frame.W = 0) & (frame.BGP # NIL) THEN frame.W := frame.BGP.width END;
        IF (frame.H = 0) & (frame.BGP # NIL) THEN frame.H := frame.BGP.height END;
        CFrameW(frame); CFrameH(frame); LFrame(frame); BFrame(NIL, frame);
        Oberon.RemoveMarks(INT(frame.X), INT(frame.Y), INT(frame.W), INT(frame.H));
        Display.CopyBlock(C.X, C.Y, INT(frame.W), INT(frame.H), 0, Display.UBottom, Display.replace);
        frame.X := C.X; frame.Y := C.Y; DBFrame(frame); DFFrame(frame);
        Release(keysum); 
        Display.CopyBlock(0, Display.UBottom, INT(frame.W), INT(frame.H), INT(frame.X), INT(frame.Y), Display.replace)
      END
    END
  END Pop;

  PROCEDURE Overlay*;
    VAR keysum: SET; frame: Frame; obj: Object; VP: Viewer; VF: Viewers.Viewer;
  BEGIN
    FlipMark; Release(keysum); FlipMark;
    IF keysum # {0, 1, 2} THEN 
      nofErr := 0; GetSym; PFrame(C.doc, frame);
      IF (frame.W = 0) & (frame.H = 0) & (frame.obj = NIL) THEN
        NEW(obj); obj.gen := "PowerDoc.NewView";
        frame.obj := obj; frame.W := SW; frame .H := SH;
      ELSE PErr("page frame")
      END;
      IF nofErr = 0 THEN GFrame(frame);
        IF nofErr = 0 THEN CFrameW(frame); CFrameH(frame); LFrame(frame); BFrame(NIL, frame);
          VP := frame.obj.obx(Viewer);
          VP.doc := C.doc; VP.frame := frame; VP.isOverlay := TRUE; C.doc.view := VP;
          NEW(VF); VF.handle := HandleFiller; Viewers.OpenTrack(0, VP.W-1, VF);
          Viewers.Open(VP, 0, SH-1); Restore(VP)
        END
      END
    END
  END Overlay;

  PROCEDURE Flip*;
    VAR keysum: SET; old, new: Frame; VP: Viewer; VF: Viewers.Viewer;
  BEGIN
    FlipMark; Release(keysum); FlipMark;
    IF keysum # {0, 1, 2} THEN 
      old := C.doc.view.frame; GetSym;
      IF S.class = Name THEN new := C.doc.dsc;
        WHILE (new # NIL) & (new.name # S.s) DO new := new.next END
      ELSE new := old.next
      END;
      IF new = NIL THEN new := C.doc.dsc END;
      IF new # old THEN
        Viewers.Close(old.obj.obx(Viewer));
        VP := new.obj.obx(Viewer); VP.doc := C.doc; VP.frame := new; C.doc.view := VP;
        NEW(VF); VF.handle := HandleFiller; Viewers.OpenTrack(0, VP.W-1, VF);
        Viewers.Open(VP, 0, SH-1); Restore(VP)
      END
    END
  END Flip;

(*** Compiler & Renderer ***)

  PROCEDURE ParseStream (VAR M: Oberon.InputMsg; doc: Document; frame: Frame; str: Stream);
    VAR WR: WordReader; ref: RefPoint; obx: Objects.Object;
      Mod: Modules.Module; Proc: Modules.Command;
      keysum: SET; XW, WW, SW: REAL; pos, len: LONGINT; i, j: INTEGER;
      name: ARRAY 32 OF CHAR;
  BEGIN
    IF str.terminal & Inside(M.X, M.Y, frame.X + str.HPad, frame.Y + str.VPad, frame.W - 2*str.HPad, frame.H- 2*str.VPad) THEN
      ref := str.ref;
      WHILE (ref # NIL) & (M.Y < ref.Y) DO ref := ref.next END;
      IF ref # NIL THEN
        Oberon.DrawCursor(Oberon.Mouse, Oberon.Arrow, M.X, M.Y);
        WR.text := str.text; WR.scope := ref.scope;
        Texts.OpenReader(WR, str.text, ref.pos); ReadTextWord(WR);
        pos := ref.pos; XW := frame.X + str.HPad;
        LOOP
          len := WR.len; WW := WR.wid;
          IF WR.eot OR (XW + WW > M.X) THEN EXIT END;
          WR.lib.GetObj(WR.lib, ORD(" "), obx); SW := obx(Fonts.Char).dx; ReadTextWord(WR);
          IF (WR.len = 2) & (WR.str[0] = "%") & (WR.str[1] = "\") OR (XW + WW + WR.wid >= frame.X + frame.W - str.HPad)
            THEN EXIT
          END;
          pos := pos + len + 1; XW := XW + WW + SW
        END;
        IF WR.scope.com # 0 THEN
          Texts.OpenScanner(S, WR.text, WR.scope.com); Texts.Scan(S);
          COPY(S.s, name); i := 0; j := 0;
          WHILE name[j] # 0X DO
            IF name[j] = "." THEN i := j END;
            INC(j)
          END;
          IF i > 0 THEN name[i] := 0X;
            Mod := Modules.ThisMod(name);
            IF Modules.res = 0 THEN INC(i); j := i;
              WHILE name[j] # 0X DO name[j - i] := name[j]; INC(j) END;
              name[j - i] := 0X;
              Proc := Modules.ThisCommand(Mod, name);
              IF Modules.res = 0 THEN
                C.doc := doc; C.frame := frame; C.text := WR.text; C.pos := Texts.Pos(S);
                C.keys := M.keys; C.X := M.X; C.Y := M.Y;
                C.XW := INT(XW); C.YW := INT(ref.Y); C.WW := INT(WW);
                Proc
              END
            END
          END
        ELSE Release(keysum)
        END;
        M.res := 0
      END
    END
  END ParseStream;

  PROCEDURE HandleCmd (VAR M: Oberon.InputMsg; doc: Document; frame: Frame);
    VAR dsc: Frame; str: Stream;
  BEGIN
    IF Inside(M.X, M.Y, frame.X, frame.Y, frame.W, frame.H) THEN dsc := frame.dsc;
      WHILE (M.res < 0) & (dsc # NIL) DO HandleCmd(M, doc, dsc); dsc := dsc.next END;
      IF M.res < 0 THEN str := frame.std;
        WHILE (M.res < 0) & (str # NIL) DO ParseStream(M, doc, frame, str); str := str.next END;
        IF M.res < 0 THEN str := frame.sti;
          WHILE (M.res < 0) & (str # NIL) DO ParseStream(M, doc, frame, str); str := str.next END
        END
      END
    END
  END HandleCmd;

  PROCEDURE HandleViewer (V: Objects.Object; VAR M: Objects.ObjMsg);
    VAR V1: Viewer; dsc: Display.Frame;
  BEGIN
    WITH V: Viewer DO
      IF M IS Display.FrameMsg THEN
        WITH M: Display.FrameMsg DO
          IF M IS Oberon.InputMsg THEN
            WITH M: Oberon.InputMsg DO
              IF M.id = Oberon.track THEN ToLocation(V, M);
                IF M.res < 0 THEN Oberon.DrawCursor(Oberon.Mouse, Oberon.Arrow, M.X, M.Y);
                  IF M.keys # {} THEN HandleCmd(M, V.doc, V.frame);
                    IF M.res < 0 THEN
                      REPEAT Input.Mouse(M.keys, M.X, M.y); Oberon.DrawCursor(Oberon.Mouse, Oberon.Arrow, M.X, M.Y)
                      UNTIL M.keys = {}
                    END
                  END
                END
              ELSE OMToDescendants(V, M)
              END
            END
          ELSIF M IS Oberon.ControlMsg THEN
            WITH M: Oberon.ControlMsg DO
              IF M.id = Oberon.mark THEN
                IF Inside(M.X, M.Y, V.X, V.Y, V.W, V.H) THEN
                  Oberon.DrawCursor(Oberon.Mouse, Oberon.Arrow, M.X, M.Y);
                  Oberon.DrawCursor(Oberon.Pointer, Oberon.Star, M.X, M.Y)
                END
              ELSIF M.id = Oberon.neutralize THEN
                IF V.isOverlay THEN Viewers.Close(V) END
              ELSE FMToDescendants(V, M)
              END
            END
          ELSIF M IS Display.LocateMsg THEN
            WITH M: Display.LocateMsg DO
              IF M.F = NIL THEN FMToDescendants(V, M) END
            END
          ELSIF M IS Display.ModifyMsg THEN
            WITH M: Display.ModifyMsg DO
              IF M.F # V THEN dsc := V.dsc;
                WHILE (dsc # NIL) & (dsc # M.F) DO dsc := dsc.next END;
                IF M.F # dsc THEN FMToDescendants(V, M) END
              END
            END
          ELSIF M IS Display.ConsumeMsg THEN
            WITH M: Display.ConsumeMsg DO
              IF M.F = V THEN
                IF (M.id = Display.drop) & (M.obj # NIL) & (M.obj IS Display.Frame) THEN
                  Consume(V, M.u, M.v, M.obj(Display.Frame))
                END
              ELSE FMToDescendants(V, M)
              END
            END
          ELSIF M IS Display.ControlMsg THEN
            WITH M: Display.ControlMsg DO
              IF (M.F = NIL) OR (M.F = V) THEN
                IF M.id = Display.restore THEN Restore(V)
                  ELSIF M.id = Display.suspend THEN
                  ELSE FMToDescendants(V, M)
                END
              ELSE FMToDescendants(V, M)
              END
            END
          ELSE FMToDescendants(V, M)
          END
        END
      ELSIF M IS Objects.CopyMsg THEN
         WITH M: Objects.CopyMsg DO Copy(V, V1); M.obj := V1 END
      ELSIF M IS Objects.AttrMsg THEN
        WITH M: Objects.AttrMsg DO
          IF (M.id = Objects.get) & (M.name = "Locked") THEN
            M.res := 0; M.class := Objects.Bool; M.b := TRUE
          END
        END
      ELSE OMToDescendants(V, M)
      END
    END
  END HandleViewer;

  PROCEDURE Compile*;
    VAR doc: Document; libCont, frame: Frame; str: Stream;
      i: INTEGER; L: Objects.Library; libName: ARRAY 32 OF CHAR;
  BEGIN
    Texts.OpenScanner(S, Oberon.Par.text, Oberon.Par.pos); GetSym;
    WHILE S.class = Head DO
      IF Matching(S.s, LibOp) THEN GetSym;
        IF S.class = Name THEN
          COPY(S.s, libName); libCont := NIL; nofErr := 0; GetSym;
          WHILE (nofErr = 0) & (S.class = Head) & Matching(S.s, FrameOp) DO PFrame(NIL, frame);
            IF (frame.name[0] # 0X) & (frame.obj # NIL) THEN frame.next := libCont; libCont := frame
              ELSE PErr("named object")
            END
          END;
          IF nofErr = 0 THEN frame := libCont;
            WHILE (nofErr = 0) & (frame # NIL) DO GFrame(frame); frame := frame.next END;
            IF nofErr = 0 THEN
              L := Objects.ThisLibrary(libName); frame := libCont;
              WHILE frame # NIL DO
                CFrameW(frame); CFrameH(frame); LFrame(frame); BFrame(NIL, frame); BindObj(frame.obj.obx, L);
                Objects.PutName(L.dict, frame.obj.obx.ref, frame.name);
                frame := frame.next
              END;
              L.Store(L)
            END
          END
        ELSE PErr("libname")
        END
      ELSIF  Matching(S.s, DocOp) THEN PDoc(doc);
        IF (nofErr = 0) & (doc.dsc # NIL) & (doc.name[0] # 0X) THEN frame := doc.dsc;
          WHILE (nofErr = 0) & (frame # NIL) DO GFrame(frame); frame := frame.next END;
          IF nofErr = 0 THEN str := doc.std;
            WHILE (nofErr = 0) & (str # NIL) DO GStream(str); str := str.next END;
            IF nofErr = 0 THEN
              COPY(doc.name, libName); i := 0;
              REPEAT INC(i) UNTIL libName[i] = 0X;
              libName[i] := "."; libName[i+1] := "L"; libName[i+2] := "i"; libName[i+3] := "b"; libName[i+4] := 0X;
              L := Objects.ThisLibrary(libName); frame := doc.dsc;
              WHILE frame # NIL DO
                CFrameW(frame); CFrameH(frame); LFrame(frame); BFrame(NIL, frame); BindObj(frame.obj.obx, L);
                frame := frame.next
              END;
              L.Store(L)
            END
          END
        END
      END
    END
  END Compile;

  PROCEDURE Open*;
    VAR doc: Document; dsc: Frame; str: Stream; dst: Destination;
      WR: WordReader; VF: Viewers.Viewer; VP: Viewer;
  BEGIN
    Texts.OpenScanner(S, Oberon.Par.text, Oberon.Par.pos); GetSym;
    IF (S.class = Head) & Matching(S.s, DocOp) THEN nofErr := 0; PDoc(doc);
      IF (nofErr = 0) & (doc.dsc # NIL) THEN dsc := doc.dsc;
        WHILE (nofErr = 0) & (dsc # NIL) DO GFrame(dsc); dsc := dsc.next END;
        IF nofErr = 0 THEN str := doc.std;
          WHILE (nofErr = 0) & (str # NIL) DO GStream(str); str := str.next END;
          IF nofErr = 0 THEN dsc := doc.dsc;
            WHILE dsc # NIL DO
              CFrameW(dsc); CFrameH(dsc); LFrame(dsc); BFrame(NIL, dsc); dsc := dsc.next
            END;
            str := doc.std;
            WHILE str # NIL DO
              WR.text := str.text; WR.pos := str.pos; WR.scope := DefaultScope;
              Texts.OpenReader(WR, str.text, str.pos); ReadTextWord(WR); dst := str.dst;
              WHILE ~WR.eot & (dst # NIL) DO LStream(str, dst.frame, WR); dst := dst.next END;
              str := str.next
            END;
            VP := doc.dsc.obj.obx(Viewer); VP.doc := doc; VP.frame := doc.dsc; doc.view := VP;
            NEW(VF); VF.handle := HandleFiller; Viewers.OpenTrack(0, VP.W-1, VF);
            Viewers.Open(VP, 0, SH-1); Restore(VP)
          END
        END
      END
    END
  END Open;

(*** Resource Linker ***)

  PROCEDURE EQ (VAR name: ARRAY OF CHAR; VAR i: INTEGER): BOOLEAN;
    VAR j: INTEGER;
  BEGIN j := 0;
    WHILE (res[i] # 0X) & (name[j] # 0X) & (res[i] = name[j]) DO INC(i); INC(j) END;
    RETURN res[i] = name[j]
  END EQ;

  PROCEDURE AddResource (VAR name: ARRAY OF CHAR);
    VAR i, j: INTEGER;
  BEGIN j := 0;
    WHILE (name[j] # 0X) & (name[j] # ".") DO INC(j) END;
    name[j] := 0X; i := 0;
    WHILE (i # ind) & ~EQ(name, i) DO
      WHILE res[i] # 0X DO INC(i) END;
      INC(i)
    END;
    IF i = ind THEN j := 0;
      WHILE name[j] # 0X DO res[ind] := name[j]; INC(ind); INC(j) END;
      res[ind] := 0X; INC(ind)
    END
  END AddResource;

  PROCEDURE ReadWord0 (VAR WR: WordReader);
    VAR i: INTEGER; ch: CHAR; 
  BEGIN Texts.Read(WR, ch);
    WHILE ~WR.eot & (ch <= " ") DO Texts.Read(WR, ch) END;
    i := 0;
    WHILE ~WR.eot & (ch > " ") & (i # MaxLen) DO
      WR.str[i] := ch; INC(i); Texts.Read(WR, ch)
    END;
    WR.len := i; WR.nextCh := ch
  END ReadWord0;

  PROCEDURE ReadTextWord0 (VAR WR: WordReader);
    VAR pos: LONGINT; pbal: INTEGER; ch: CHAR;
  BEGIN ReadWord0(WR);
    IF WR.len = 2 THEN
      IF (WR.str[0] = "{") & (WR.str[1] = "%") THEN pos := Texts.Pos(WR);
        Texts.OpenScanner(S, WR.text, pos); GetSym;
        LOOP
          IF S.class # Head THEN EXIT END;
          IF Matching(S.s, FontOp) THEN GetSym;
            IF S.class = Name THEN AddResource(S.s); GetSym (*tail*) END 
          ELSIF Matching(S.s, ColOp) THEN GetSym;
            IF S.class = Int THEN GetSym END
          ELSIF Matching(S.s, ParOp) THEN GetSym;
            IF S.class = Int THEN GetSym END
          ELSIF Matching(S.s, LeadOp) THEN GetSym;
            IF S.class = Int THEN GetSym END
          ELSIF Matching(S.s, CmdOp) THEN GetSym;
            IF S.class = Name THEN AddResource(S.s); pbal := 1; 
              REPEAT Texts.Scan(S);
                IF S.class = Char THEN
                  IF S.c = "(" THEN INC(pbal) ELSIF S.c = ")" THEN DEC(pbal) END
                END
              UNTIL pbal = 0
            END
          END;
          pos := Texts.Pos(S); (*pos after tail*) GetSym
        END;
        Texts.OpenReader(WR, WR.text, pos); ReadTextWord0(WR)
      ELSIF (WR.str[0] = "%") & (WR.str[1] = "}") THEN ReadTextWord0(WR)
      END
    END
  END ReadTextWord0;

  PROCEDURE AddStreamResources (str: Stream);
    VAR WR: WordReader;
  BEGIN WR.text := str.text;
    Texts.OpenReader(WR, str.text, str.pos);
    REPEAT ReadTextWord0(WR) UNTIL WR.eot
  END AddStreamResources;

  PROCEDURE AddFrameResources (frame: Frame);
    VAR dsc: Frame; str: Stream; obx: Objects.Object; M: Objects.AttrMsg;
  BEGIN
    IF frame.obj # NIL THEN obx := frame.obj.obx; M.id := Objects.get;
      M.name := "Gen"; obx.handle(obx, M);
      IF (M.res = 0) & (M.class = Objects.String) THEN AddResource(M.s) END;
      M.name := "Cmd"; obx.handle(obx, M);
      IF (M.res = 0) & (M.class = Objects.String) THEN AddResource(M.s) END
    END;
    str := frame.std;
    WHILE str # NIL DO AddStreamResources(str); str := str.next END;
    dsc := frame.dsc;
    WHILE dsc # NIL DO AddFrameResources(dsc); dsc := dsc.next END
  END AddFrameResources;

  PROCEDURE Link*;
    VAR doc: Document; dsc: Frame; str: Stream; i: INTEGER;
  BEGIN
    Texts.OpenScanner(S, Oberon.Par.text, Oberon.Par.pos); GetSym;
    IF (S.class = Head) & Matching(S.s, DocOp) THEN nofErr := 0; PDoc(doc);
      IF (nofErr = 0) & (doc.dsc # NIL) THEN dsc := doc.dsc;
        WHILE (nofErr = 0) & (dsc # NIL) DO GFrame(dsc); dsc := dsc.next END;
        IF nofErr = 0 THEN str := doc.std;
          WHILE (nofErr = 0) & (str # NIL) DO GStream(str); str := str.next END;
          IF nofErr = 0 THEN ind := 0; dsc := doc.dsc;
            WHILE dsc # NIL DO AddFrameResources(dsc); dsc := dsc.next END;
            str := doc.std;
            WHILE str # NIL DO AddStreamResources(str); str := str.next END
          END
        END
      END
    END;
    i := 0;
    WHILE i # ind DO
      WHILE res[i] # 0X DO Out.Char(res[i]); INC(i) END;
      Out.Ln; INC(i)
    END
  END Link;

  PROCEDURE NewView*;
    VAR V: Viewer;
  BEGIN NEW(V); V.handle := HandleViewer; Objects.NewObj := V
  END NewView;

BEGIN
  NEW(DefaultScope); DefaultScope.lead := 0;
  DefaultScope.lib := Fonts.Default; DefaultScope.col := Display.FG; DefaultScope.voff := 0;
  SW := Display.Width DIV 8 * 5; SH := Display.Height;
  BoxPat[0] := {0..11}; 
  BoxPat[1] := {0, 11}; BoxPat[2] := {0, 11}; BoxPat[3] := {0, 11}; BoxPat[4] := {0, 11};
  BoxPat[5] := {0, 11}; BoxPat[6] := {0, 11}; BoxPat[7] := {0, 11}; BoxPat[8] := {0, 11};
  BoxPat[9] := {0, 11}; BoxPat[10] := {0, 11}; BoxPat[11] := {0.. 11};
  NEW(box);
  box.dx := 12; box.x := 0; box.y := -3; box.w := 12; box.h := 12;
  box.pat := Display.NewPattern(12, 12, BoxPat);
  LibOp := "LIB"; DocOp := "DOC"; MetaOp := "MET"; FrameOp := "FRA"; ObjOp := "OBJ";
  StrDefOp := "STD"; StrImpOp := "STI"; DstOp := "DST"; DspOp := "DSP"; PosOp := "POS"; SizeOp := "SIZ";
  BGOp := "BG"; GridOp := "GRI"; ParOp := "PAR"; PadOp := "PAD"; BndOp := "BND"; SepOp:= "SEP";
  FontOp := "FON"; ColOp := "COL"; LeadOp := "LEA"; ImpOp := "IMP"; CmdOp := "COM"
END PowerDoc.
