TextDocs.NewDoc     lF   CColor    Flat  Locked  Controls  Org x*   BIER`   b        3 #   Oberon10.Scn.Fnt  9   9  (* ETH Oberon, Copyright 2001 ETH Zuerich Institut fuer Computersysteme, ETH Zentrum, CH-8092 Zuerich.
Refer to the "General ETH Oberon System Source License" contract available at: http://www.oberon.ethz.ch/ *)

MODULE OFSDosBasedVolumes;	(* pjm / js *)

(* OFS.Volume implementation of DOS files containing a file system (FAT16) *)

IMPORT SYSTEM, Kernel, Disks, OFS;

CONST
	Reserved = 32;	(* sectors reserved for writing during trap handling *)
	BS = 512;	(* supported device block size *)
	VBS = 2048;	(* default volume block size *)
	MaxDrive = "H";

TYPE
	Volume* = POINTER TO RECORD (OFS.Volume)
		next: Volume;
		dev: Disks.Device;
		numused: LONGINT;	(* number of used Sectors *)
		nummaxdisk: LONGINT;	(* total number of Sectors on real disk *)
		map: POINTER TO ARRAY OF LONGINT;
		partitionoffset, partitionlen, dbpvb: LONGINT;
	END;

	DataBlock = ARRAY MAX(LONGINT) OF CHAR;
	
VAR
	volumes: Volume; (* list of mounted volumes *)

PROCEDURE AllocBlock*(vol: OFS.Volume; hint: OFS.Address; VAR adr: OFS.Address);
BEGIN
	WITH vol: Volume DO
		IF OFS.ReadOnly IN vol.flags THEN HALT(21) END;
		IF (OFS.Boot IN vol.flags) & (vol.size - vol.numused <= Reserved) THEN HALT(20) END;
		ASSERT(hint >= 0);
		IF hint > vol.size THEN hint := 0 END;
		adr := hint+1;
		LOOP
			IF adr > vol.size THEN adr := 0 END;
			IF vol.map[adr] < 0 THEN
				INC(adr)	(* in use *)
			ELSE
				vol.map[adr] := -vol.map[adr];
				EXIT
			END;
			IF adr = hint THEN HALT(20) END
		END;
		INC(vol.numused)
	END
END AllocBlock;

PROCEDURE FreeBlock*(vol: OFS.Volume; adr: OFS.Address);
BEGIN
	WITH vol: Volume DO
		IF (adr < 1) OR (adr > vol.size) THEN SYSTEM.HALT(15) END;
		IF OFS.ReadOnly IN vol.flags THEN HALT(21) END;
		vol.map[adr] := ABS(vol.map[adr]);
		DEC(vol.numused)
	END
END FreeBlock;

PROCEDURE MarkBlock*(vol: OFS.Volume; adr: OFS.Address);
BEGIN
	WITH vol: Volume DO
		IF (adr < 1) OR (adr > vol.size) THEN SYSTEM.HALT(15) END;
		IF OFS.ReadOnly IN vol.flags THEN HALT(21) END;
		vol.map[adr] := -ABS(vol.map[adr]);
		INC(vol.numused)
	END
END MarkBlock;

PROCEDURE Marked*(vol: OFS.Volume; adr: OFS.Address): BOOLEAN;
BEGIN
	WITH vol: Volume DO
		IF (adr < 1) OR (adr > vol.size) THEN SYSTEM.HALT(15) END;
		IF OFS.ReadOnly IN vol.flags THEN HALT(21) END;
		RETURN vol.map[adr] < 0
	END
END Marked;

PROCEDURE Available*(vol: OFS.Volume): LONGINT;
BEGIN
	WITH vol: Volume DO
		RETURN vol.size-vol.numused
	END
END Available;

(** Get block from adr [1..size] of volume vol *)
PROCEDURE GetBlock*(vol: OFS.Volume; adr: LONGINT; VAR blk: ARRAY OF SYSTEM.BYTE);
VAR res: LONGINT;  dev: Disks.Device;
BEGIN
	WITH vol: Volume DO
		IF (adr < 1) OR (adr > vol.size) THEN SYSTEM.HALT(15) END;
		dev := vol.dev;
		ASSERT(vol.partitionoffset > 0);	(* startfs initialized *)
		ASSERT(LEN(blk) >= vol.blockSize);
		dev.transfer(dev, Disks.Read, vol.partitionoffset + ABS(vol.map[adr]), vol.dbpvb, SYSTEM.VAL(DataBlock, blk), 0, res);
		IF res # Disks.Ok THEN SYSTEM.HALT(17) END
	END
END GetBlock;

(** Put block to adr [1..size] of volume vol *)
PROCEDURE PutBlock*(vol: OFS.Volume; adr: LONGINT; VAR blk: ARRAY OF SYSTEM.BYTE);
VAR res: LONGINT;  dev: Disks.Device;
BEGIN
	WITH vol: Volume DO
		IF (adr < 1) OR (adr > vol.size) THEN SYSTEM.HALT(15) END;
		dev := vol.dev;
		ASSERT(vol.partitionoffset > 0);	(* startfs initialized *)
		ASSERT(LEN(blk) >= vol.blockSize);
		dev.transfer(dev, Disks.Write, vol.partitionoffset + ABS(vol.map[adr]), vol.dbpvb, SYSTEM.VAL(DataBlock, blk), 0, res);
		IF res # Disks.Ok THEN SYSTEM.HALT(17) END
	END
END PutBlock;

(** Finalize a volume and close its device. *)
PROCEDURE Finalize*(vol: OFS.Volume);
VAR res: LONGINT;  vt, prev: Volume;
BEGIN
	WITH vol: Volume DO
		prev := NIL; vt := volumes;
		WHILE vt # vol DO prev := vt; vt := vt.next END;
		ASSERT(vt # NIL);
		IF prev # NIL THEN prev.next := vt.next ELSE volumes := vt.next END;
		Disks.Close(vol.dev, res);	(* ignore res *)
		vol.dev := NIL; OFS.FinalizeVol(vol)
	END
END Finalize;

PROCEDURE Cap(ch: CHAR): CHAR;
BEGIN
	IF (ch >= "a") & (ch <= "z") THEN RETURN CAP(ch)
	ELSE RETURN ch
	END
END Cap;

(* ParseName - Parse one more component of a path, starting at i. *)

PROCEDURE ParseName(VAR file: ARRAY OF CHAR;  VAR i: LONGINT;  VAR name: ARRAY OF CHAR);
VAR j: LONGINT;
BEGIN
	j := 0;
	IF (file[i] = "/") OR (file[i] = "\") THEN
		INC(i);
		WHILE (file[i] = ".") & (j # 2) DO name[j] := file[i]; INC(i); INC(j) END;	(* allow up to 2 leading dots *)
		WHILE (file[i] # 0X) & (file[i] # ".") & (file[i] # "/") & (file[i] # "\") & (j # 8) DO
			file[i] := Cap(file[i]); name[j] := file[i]; INC(i); INC(j)
		END;
		WHILE j # 8 DO name[j] := " "; INC(j) END;
		IF file[i] = "." THEN INC(i) END;
		WHILE (file[i] # 0X) & (file[i] # "/") & (file[i] # "\") & (j # 11) DO
			file[i] := Cap(file[i]); name[j] := file[i]; INC(i); INC(j)
		END;
		WHILE j # 11 DO name[j] := " "; INC(j) END
	END;
	name[j] := 0X
END ParseName;

(* Equal - Return TRUE iff str1[i1..i1+len-1] = str2[i2..i2+len-1] *)

PROCEDURE Equal(VAR str1, str2: ARRAY OF CHAR;  i1, i2, len: LONGINT): BOOLEAN;
BEGIN
	WHILE len # 0 DO
		IF str1[i1] # str2[i2] THEN RETURN FALSE END;
		INC(i1);  INC(i2);  DEC(len)
	END;
	RETURN TRUE
END Equal;

(* FindName - Find a name in a directory entry. *)

PROCEDURE FindName(VAR b: ARRAY OF CHAR;  name: ARRAY OF CHAR;  VAR attr: SET;  VAR start, size: LONGINT): BOOLEAN;
VAR j: LONGINT;
BEGIN
	j := 0;
	WHILE j # 16 DO
		IF Equal(b, name, j*32, 0, 11) THEN
			start := 0;  attr := {};
			SYSTEM.GET(SYSTEM.ADR(b[j*32+0BH]), SYSTEM.VAL(CHAR, attr));
			SYSTEM.GET(SYSTEM.ADR(b[j*32+1AH]), SYSTEM.VAL(INTEGER, start));
			SYSTEM.GET(SYSTEM.ADR(b[j*32+1CH]), size);
			RETURN TRUE
		ELSE INC(j)
		END
	END;
	RETURN FALSE
END FindName;

PROCEDURE IsDOS(type: LONGINT): BOOLEAN;	(* see Partitions.IsDOS *)
BEGIN
	RETURN (type = 1) OR (type = 4) OR (type = 6)	(* DOS partition *)
END IsDOS;

(* GetBPB - Read BPB and return parameters. *)

PROCEDURE GetBPB(vol: Volume; VAR csize, fatb, rootb, rootsize, datab: LONGINT; VAR error: ARRAY OF CHAR): BOOLEAN;
VAR b: ARRAY BS OF CHAR;  x, fatsize, numfat, res: LONGINT;
BEGIN
	vol.dev.transfer(vol.dev, Disks.Read, vol.partitionoffset, 1, b, 0, res);
	x := 0;  SYSTEM.GET(SYSTEM.ADR(b[0BH]), SYSTEM.VAL(INTEGER, x));
	IF x # BS THEN COPY("block size bad", error); RETURN FALSE END;
	x := 0;  SYSTEM.GET(SYSTEM.ADR(b[0DH]), SYSTEM.VAL(CHAR, x));
	csize := x*BS;	(* bytes per cluster *)
	numfat := ORD(b[10H]);	(* FAT copies *)
	x := 0;  SYSTEM.GET(SYSTEM.ADR(b[16H]), SYSTEM.VAL(INTEGER, x));
	fatsize := x;	(* sectors in FAT *)
	x := 0;  SYSTEM.GET(SYSTEM.ADR(b[0EH]), SYSTEM.VAL(INTEGER, x));
	fatb := x;	(* start of FAT *)
	rootb := fatb + numfat*fatsize;
	x := 0;  SYSTEM.GET(SYSTEM.ADR(b[11H]), SYSTEM.VAL(INTEGER, x));
	rootsize := (x*32) DIV BS;
	datab := rootb + rootsize;	(* start of data *)
	x := 0;  SYSTEM.GET(SYSTEM.ADR(b[13H]), SYSTEM.VAL(INTEGER, x));
	IF x = 0 THEN
		SYSTEM.GET(SYSTEM.ADR(b[20H]), x)	(* big number of sectors *)
	END;
	x := (x-datab) DIV (csize DIV BS);	(* total clusters *)
	IF x <= 4078 THEN COPY("not 16-bit FAT", error); RETURN FALSE
	ELSE RETURN TRUE
	END
END GetBPB;

(* NextCluster - Return next cluster or 0 if eof or -1 if bad block or error. b is 512 byte buffer and 
	cache is block number loaded in b. *)

PROCEDURE NextCluster(vol: Volume; VAR b: ARRAY OF CHAR;  VAR cache: LONGINT;  fatb, cluster: LONGINT): LONGINT;
VAR k, x, res: LONGINT;
BEGIN
	k := cluster DIV (BS DIV 2) + fatb;	(* there are BS DIV 2 next entries per FAT block, indexed by cluster *)
	IF k # cache THEN
		vol.dev.transfer(vol.dev, Disks.Read, vol.partitionoffset + k, 1, b, 0, res);
		cache := k
	END;
		(* get next in chain *)
	x := 0;  SYSTEM.GET(SYSTEM.ADR(b[(cluster MOD (BS DIV 2))*2]), SYSTEM.VAL(INTEGER, x));
	IF x >= 0FFF8H THEN RETURN 0	(* was last cluster *)
	ELSIF x >= 0FFF0H THEN RETURN -1
	ELSE RETURN x
	END
END NextCluster;

(* FindFile - Find a file and return its position and size. *)

PROCEDURE FindFile(vol: Volume; VAR file, error: ARRAY OF CHAR; 
					csize, fatb, rootb, rootsize, datab: LONGINT;  VAR start, fsize: LONGINT): BOOLEAN;
VAR
	b: ARRAY BS OF CHAR;  disk: ARRAY 12 OF CHAR;
	 i, k, m, n, dir, res: LONGINT;  found: BOOLEAN;  attr: SET;
BEGIN
	k := 2;  ParseName(file, k, disk);
	i := 0;  found := FALSE;  start := -1;
	WHILE (i # rootsize) & ~found DO
		vol.dev.transfer(vol.dev, Disks.Read, vol.partitionoffset + rootb + i, 1, b, 0, res);
		found := FindName(b, disk, attr, start, fsize);
		INC(i)
	END;
	WHILE found & (file[k] # 0X) & (4 IN attr) DO	(* more to come *)
		dir := start;  ParseName(file, k, disk);  found := FALSE;
		LOOP
			m := 0;  n := csize DIV BS;
			WHILE (m # n) & ~found DO
				vol.dev.transfer(vol.dev, Disks.Read, vol.partitionoffset + (dir-2)*n + datab + m, 1, b, 0, res);
				found := FindName(b, disk, attr, start, fsize);
				INC(m)
			END;
			IF found THEN EXIT END;
			m := -1;  dir := NextCluster(vol, b, m, fatb, dir);
			IF dir <= 0 THEN
				IF dir = 0 THEN EXIT	(* last cluster *)
				ELSE COPY("cluster error", error); RETURN FALSE	(* error *)
				END
			END
		END
	END;
	IF found & (file[k] = 0X) & (attr * {3,4} = {}) THEN RETURN TRUE
	ELSE COPY("file not found", error); RETURN FALSE
	END
END FindFile;

(* InitMapping - Initialise sector mapping for file system in a file *)

PROCEDURE InitMapping(vol: Volume; dev: Disks.Device; partnum: LONGINT; VAR file, error: ARRAY OF CHAR);
VAR
	i, j, m, csize, fatb, rootb, datab, rootsize, start, fsize, cache: LONGINT;
	b: ARRAY BS OF CHAR;
BEGIN
	vol.partitionoffset := dev.table[partnum].start;
	vol.partitionlen := dev.table[partnum].size;
	ASSERT(vol.partitionoffset > 0);
	IF GetBPB(vol, csize, fatb, rootb, rootsize, datab, error) THEN
		IF FindFile(vol, file, error, csize, fatb, rootb, rootsize, datab, start, fsize) THEN
			vol.size := fsize DIV vol.blockSize;
			IF (vol.size < 8) OR (start = 0) THEN
				COPY("file too small", error); vol.size := 0; RETURN
			END;
			IF csize MOD vol.blockSize # 0 THEN
				COPY("bad cluster size", error); vol.size := 0; RETURN
			END;
			NEW(vol.map, vol.size+1);
			vol.map[0] := -1;	(* reserve sector 0 (illegal to use) *)
			i := start;  j := 1;  cache := -1;
			LOOP
				FOR m := 0 TO (csize DIV vol.blockSize)-1 DO	(* the next csize bytes of the file are stored in cluster i *)
					vol.map[j] := (i-2)*(csize DIV BS) + datab + m*vol.dbpvb;  INC(j);	(* i-2 for 1st 2 FAT entries *)
					IF j > vol.size THEN EXIT END	(* done *)
				END;
				i := NextCluster(vol, b, cache, fatb, i);	(* step to next *)
				IF i <= 0 THEN
					IF i = 0 THEN EXIT	(* last cluster *)
					ELSE vol.size := 0; COPY("bad cluster", error); RETURN	(* error *)
					END
				END
			END;
			FOR i := 1 TO vol.size DO
				ASSERT((vol.map[i] >= datab) & (vol.map[i] < vol.partitionoffset+vol.partitionlen))
			END;
			vol.nummaxdisk := vol.size
		END
	END
END InitMapping;

PROCEDURE InitVol(vol: Volume; readonly: BOOLEAN; VAR file: ARRAY OF CHAR);
BEGIN
	COPY(file, vol.name);
	vol.flags := {};
	IF Disks.ReadOnly IN vol.dev.flags THEN INCL(vol.flags, OFS.ReadOnly) END;
	IF Disks.Removable IN vol.dev.flags THEN INCL(vol.flags, OFS.Removable) END;
	vol.AllocBlock := AllocBlock;  vol.FreeBlock := FreeBlock;
	vol.MarkBlock := MarkBlock;  vol.Marked := Marked;
	vol.Available := Available;
	vol.GetBlock := GetBlock;  vol.PutBlock := PutBlock;
	vol.Sync := OFS.Sync; vol.Finalize := Finalize;
	IF ~readonly THEN vol.numused := 0
	ELSE vol.numused := vol.size
	END
END InitVol;

PROCEDURE EqualVolume(v1, v2: Volume): BOOLEAN;
BEGIN
	RETURN (v1.dev = v2.dev) & (v1.partitionoffset = v2.partitionoffset) & (ABS(v1.map[1]) = ABS(v2.map[1]))
END EqualVolume;

PROCEDURE FindPartition(VAR file: ARRAY OF CHAR; VAR dev: Disks.Device; VAR partnum: LONGINT);
VAR table: Disks.DeviceTable; i, j, res, dosnum: LONGINT;
BEGIN
	dev := NIL; partnum := -1;
	Disks.GetRegistered(table);
	IF table # NIL THEN
		i := 0; dosnum := 2;	(* assigned like in Partitions and FATFiles *)
		WHILE (i # LEN(table)) & (dev = NIL) DO
			IF table[i].blockSize = BS THEN
				Disks.Open(table[i], res);
				IF res = Disks.Ok THEN
					j := 0;
					WHILE (j # LEN(table[i].table)) & (dev = NIL) DO
						IF IsDOS(table[i].table[j].type) THEN
							IF CHR(ORD("A")+dosnum) = file[0] THEN
								dev := table[i]; partnum := j	(* found *)
							END;
							INC(dosnum)
						END;
						INC(j)
					END;
					IF dev = NIL THEN Disks.Close(table[i], res) END	(* close again - ignore res *)
				END
			END;
			INC(i)
		END
	END
END FindPartition;

PROCEDURE GetParam(VAR file: ARRAY OF CHAR; VAR readonly: BOOLEAN);
VAR i: LONGINT; ch: CHAR;
BEGIN
	readonly := FALSE;
	REPEAT OFS.ReadPar(ch) UNTIL ch # " ";
	i := 0;
	WHILE (ch > " ") & (ch # ",") DO
		IF ch = "\" THEN ch := "/" END;
		file[i] := Cap(ch); INC(i); OFS.ReadPar(ch)
	END;
	file[i] := 0X;
	IF ch = "," THEN
		REPEAT
			OFS.ReadPar(ch);
			IF Cap(ch) = "R" THEN readonly := TRUE END
		UNTIL ch <= " "
	END
END GetParam;

(** Generate a new dos based volume object. OFS.Par: dosfilename [",R"] *)
PROCEDURE New*;
VAR
	vol, vt: Volume; partnum: LONGINT; dev: Disks.Device; readonly, retry: BOOLEAN;
	file, error: ARRAY 64 OF CHAR;
BEGIN
	OFS.NewVol := NIL; error := "";
	GetParam(file, readonly);
	IF file[0] = "?" THEN retry := TRUE; file[0] := "C" ELSE retry := FALSE END;
	LOOP
		Kernel.SetLogMark;	(* start saving log output *)
		Kernel.WriteString("OFSDosBasedVolumes: "); Kernel.WriteString(file);
		IF (file[0] >= "C") & (file[0] <= MaxDrive) & (file[1] = ":") & (file[2] = "/") & (file[3] # 0X) THEN
			FindPartition(file, dev, partnum);
			IF dev # NIL THEN
				NEW(vol); vol.dev := dev; vol.blockSize := VBS;
				vol.dbpvb := vol.blockSize DIV vol.dev.blockSize;
				InitMapping(vol, dev, partnum, file, error);
				IF vol.size # 0 THEN
					vt := volumes;
					WHILE (vt # NIL) & ~EqualVolume(vt, vol) DO vt := vt.next END;
					IF vt = NIL THEN	(* not a duplicate volume *)
						InitVol(vol, readonly, file);
						IF readonly THEN INCL(vol.flags, OFS.ReadOnly) END;
						vol.next := volumes; volumes := vol;
						OFS.NewVol := vol	(* commit the volume *)
					ELSE
						error := "already mounted"
					END
				ELSE
					IF error = "" THEN error := "bad volume" END
				END
			ELSE
				error := "partition not found"
			END
		ELSE
			error := "bad drive name"; retry := FALSE
		END;
		IF OFS.NewVol = NIL THEN Kernel.WriteChar(" "); Kernel.WriteString(error) END;
		Kernel.WriteLn;
		IF (OFS.NewVol = NIL) & retry & (file[0] < MaxDrive) THEN	(* failure - try next drive *)
			file[0] := CHR(ORD(file[0])+1)
		ELSE
			EXIT
		END
	END;
	Kernel.GetMarkedLog(file); OFS.SetPar(file)	(* copy error message out for OFSTools *)
END New;

BEGIN
	volumes := NIL
END OFSDosBasedVolumes.

System.Free OFSDosBasedVolumes ~

(*
to do:
o check if host file has been deleted while still mounted
*)
