(************************************************************************

Name          : Sound
Zweck         : Sound out of internal loudspeaker as background task
Version       : 1.1
Vorgaenger	: 1.2
Aenderungen   : Adapted to Version 2.2 of BTasks, melodies only coded as
strings
Status, Bugs  : StringToMel still to be done (completed)
Autor         : Frank Hrebabetzky
Computer      : PC >=386
Compiler      : Oberon
Betriebssystem: Oberon System 3 Version 2.0
Datum         : January 1996

************************************************************************)

MODULE Sound;

IMPORT
  SYSTEM, Oberon, Math, Texts, BTasks;

CONST
  SIL      = 0.0;    (* silence *)
  MinFreq* = 19.0;   MaxFreq* = 500000.0;  (* min. intern.val.=2 (sil, stop)
*)
  A4       = 440.0;
  TONES   = 12;     OCTAVES = 10;
  c = 0;   cis = 1;   d = 2;   dis = 3;   e = 4;   f = 5;   fis = 6;
  g = 7;   gis = 8;   a = 9;   ais =10;   h =11;
  ON = TRUE;   OFF = FALSE;
  MLEN* = 256;   (** max. melody length incl. stop *)
  sil = 0;   stop = 1;
  cucaracha = "o4c4c4c4f4p8a4p4c4c4c4f4p8a4p20f8f4e4e4d4d4c16p20s";

TYPE
  Tone     = RECORD
                freq, len: INTEGER;			(* intern. freq. represent.   *)
                stop     : BOOLEAN;			(* intern.counterval., tUnits *)
              END;
  MelTone   = RECORD                            (* intern. freq. represent.  
*)
                freq, len: INTEGER;             (* intern.counterval., tUnits
*)
              END;
              (* min. len of 2 required, because 1 is subtracted for pause   
*)
  WORD      = ARRAY 2 OF CHAR;

VAR
  timerActive          : BOOLEAN;                 (* semaphore                
*)
  TF                   : ARRAY TONES,OCTAVES OF INTEGER; (* tone frequencies  
*)
  beep                 : Tone;
  mtone                : MelTone;
  beeper, player       : BTasks.BTask;
  repeat*              : BOOLEAN;
  aTUnit*, mc, tc, oct : INTEGER;			(* melody, tone count, octave *)
  ams*                 : ARRAY MLEN OF CHAR;	(** active melody string *)

(**Melodies have to be coded as strings like "o4c1d1e1p16f2s" in the following
way:
   o4 selects octave no. 4
   c  tone c
   C  tone cis
   d  tone d
   D  tone dis
   and so on. p stands for pause. The number following a tone or pause
designates
   its duration in units of aTUnit. It can have one or two decimal places and
its
   minimal value is 2 because 1 is subtracted for the pause between 2 tones.
   IF repeat is FALSE, the melody will be played only once, else over and over
   again. The melody can be stopped any time by a call to StopPlay. *)



(*                   H A R D W A R E   C O N T R O L
                     -------------------------------
*)

PROCEDURE InitTimer;
CONST noiseMode=182;
      timerPort2=43H;
BEGIN
  SYSTEM.PORTOUT (timerPort2, noiseMode);
END InitTimer;



PROCEDURE SetFreq (f:WORD);
CONST timerPort1 = 42H;
BEGIN
  SYSTEM.PORTOUT (timerPort1, f[0]);
  SYSTEM.PORTOUT (timerPort1, f[1]);
END SetFreq;



PROCEDURE switch (b:BOOLEAN);
CONST loudspPort=61H;
VAR   ctrl: INTEGER;
BEGIN
  SYSTEM.PORTIN (loudspPort, ctrl);
  IF b THEN
    ctrl:= SYSTEM.VAL(INTEGER,SYSTEM.VAL(SET,ctrl) + {0,1});
  ELSE
    ctrl:= SYSTEM.VAL(INTEGER,SYSTEM.VAL(SET,ctrl) - {0,1});
  END;
  SYSTEM.PORTOUT (loudspPort, ctrl);
END switch;



(*            C O N V E R S I O N   A N D   I N I T I A L I Z A T I O N
              ---------------------------------------------------------
*)


PROCEDURE IntFreq (f:REAL): INTEGER;
(* Converts frequency to internal representation as in Tone. *)
CONST Counter  = 1193180.0;
BEGIN
  IF f=SIL THEN RETURN sil END;
  RETURN SYSTEM.VAL(INTEGER,ENTIER(Counter/f+0.5));
  (* VAL instead SHORT in order to enable values > MAX(INTEGER) *)
END IntFreq;



PROCEDURE CalcToneFreq;
VAR t, o     : INTEGER;
    f, f0, df: REAL;
BEGIN
  (* calculate tone frequencies *)
  df:= Math.exp (Math.ln(2.0)/12.0);
  f0:= A4 * df * df * df / 32.0;          (* = c0 *)
  FOR o:=0 TO OCTAVES-1 DO
    f:= f0;
    FOR t:=0 TO TONES-1 DO
      TF[t,o]:= IntFreq(f);
      f:= f * df;
    END;
    f0:= f0 * 2.0;
  END;
END CalcToneFreq;



(*                                    B E E P
                                      -------
*)


PROCEDURE playTone;
BEGIN
    IF beep.stop THEN
      switch(OFF);
      BTasks.Remove(beeper);
      timerActive:= FALSE;
    ELSE
      switch(ON);
      beep.stop:= TRUE;
    END;
END playTone;



PROCEDURE Beep*;
BEGIN
  (* handle semaphore *)
  IF timerActive THEN RETURN END;
  timerActive:= TRUE;
  (* initialize tone *)
  beep.stop:= FALSE;
  (* initialize timer *)
  InitTimer;
  SetFreq (SYSTEM.VAL(WORD,beep.freq));
  (* install task *)
  beeper:= BTasks.Install (playTone, "Beep", beep.len, 0, 0, 0);
END Beep;



PROCEDURE SetBeep*;
(* Get frequency [Hz] and length [ms].
   Example command:
     SetBeep 440.0 1000 
   The next Beep will be of 440 Hz and 1 s. *)
VAR s: Texts.Scanner;
    f: REAL;
BEGIN
  Texts.OpenScanner (s, Oberon.Par.text, Oberon.Par.pos);
  Texts.Scan(s);   f:= s.x;
  IF f<MinFreq THEN f:=MinFreq
  ELSIF f>MaxFreq THEN f:=MaxFreq
  END;
  beep.freq:= IntFreq(f);
  Texts.Scan(s);   beep.len:= SHORT(s.i);
END SetBeep;



(*                                  M E L O D Y
                                    -----------
*)


PROCEDURE NextNumber(): INTEGER;
(* Get next number of 1 or 2 places from ams. mc points already to the first
place.*)
VAR c1, c2: CHAR;
    i     : INTEGER;
BEGIN
  c1:= ams[mc];   INC(mc);
  i:= ORD(c1) - ORD("0");
  c2:= ams[mc];
  IF (c2>="0") & (c2<="9") THEN
    i:= 10*i + ORD(c2) - ORD("0");
    INC(mc);
  END;
  RETURN i;
END NextNumber;


PROCEDURE NextTone (VAR tone:MelTone);
(* Get next tone from ams. *)
VAR ch: CHAR;
BEGIN
  ch:= ams[mc];	INC(mc);
  WHILE ch="o" DO
    oct:= NextNumber();
    ch:= ams[mc];	INC(mc);
  END;
  IF ch="s" THEN	(*end of melody*)
    tone.freq:= stop;
    RETURN;
  END;
    tone.len:= NextNumber();
  CASE ch OF
   "p": tone.freq:= sil;
   |"c": tone.freq:= TF[c,oct];
   |"C": tone.freq:= TF[cis,oct];
   |"d": tone.freq:= TF[d,oct];
   |"D": tone.freq:= TF[dis,oct];
   |"e": tone.freq:= TF[e,oct];
   |"f": tone.freq:= TF[f,oct];
   |"F": tone.freq:= TF[fis,oct];
   |"g": tone.freq:= TF[g,oct];
   |"G": tone.freq:= TF[gis,oct];
   |"a": tone.freq:= TF[a,oct];
   |"A": tone.freq:= TF[ais,oct];
   |"h": tone.freq:= TF[h,oct];
  END;
END NextTone;


PROCEDURE PlayMelody;
BEGIN
  (* position within tone *)
  IF tc=0 THEN (*beginning new tone*)
    IF mtone.freq=stop THEN
      IF repeat THEN
        mc:=0;   NextTone (mtone);
      ELSE
        BTasks.Remove (player);
        timerActive:= FALSE;
      END;
      RETURN;
    END;
    IF mtone.freq#sil THEN
      SetFreq (SYSTEM.VAL(WORD,mtone.freq));
      switch(ON);
    END;
    INC(tc);
  ELSIF tc=mtone.len-1 THEN (*end of tone*)
    switch(OFF);
    tc:= 0;
    NextTone (mtone);
  ELSE (*during tone*)
    INC(tc)
  END;
END PlayMelody;



PROCEDURE Play*;
BEGIN
  IF timerActive THEN RETURN END;
  timerActive:= TRUE;
  mc:= 0;   tc:= 0;   NextTone(mtone);
  InitTimer;
  player:= BTasks.Install (PlayMelody, "Play", aTUnit, 0, 0, 0);
END Play;


PROCEDURE StopPlay*;
BEGIN   BTasks.Remove (player);
END StopPlay;


BEGIN
  CalcToneFreq;
  (* initialize beep parameters *)
  beep.freq:= IntFreq(A4);   beep.len:=1000;   beep.stop:=FALSE;
  aTUnit:= 20;	repeat:=TRUE;
  COPY (cucaracha, ams);
  timerActive:= FALSE;
END Sound.

Sound.Beep
Sound.SetBeep 100.0 1000
Sound.Play
Sound.StopMelody
