(**************************************************************************

Name:           Diagrams
Purpose:        Visible gadget for xy diagrams
Version:	2.1
Predecessor:    2.0
Changes:        debugged, layout changed, visible flag in GraphEl, new:line,
				StringUp now without Kernel
Known Bugs:	Print not tested, StringUp/OberUp10.Scn.Fnt somewhat messy
Target platform:PC>=386
Compiler:	Oberon System 3, V2.0
Date:           December 1995
Author:		Frank Hrebabetzky

***************************************************************************)

MODULE Diagrams;

(* In order to avoid overflow for points (x,y) to be plotted, they shouldn't
be
   too far from the plot range [xmin,xmax], [ymin,ymax]. The limits are about:
   |x|<650*(xmax-xmin), where 650=MAX(INTEGER)/DPX, where DPX is the minimum
   diagram size in pixels in x-direction. For 650 a value of 50 was assumed.
   Analogously for y.

   An update message to a frame redraws the frame, the linked model (Graph)
   chain and the GraphEl chain.

   An update message to a model draws this and the subsequent models, but
doesn't
   affect the already existing drawings, it doesn't clear the plot area.

   The complete model chain can be excluded from display by linking the frame
   (diagram) to NIL. Individual models in the chain can be made invisible by
   setting the field f of the graph to NIL or the number of function points
   (f.NP) to 0.

   The complete GraphEl chain is made invisible by setting the graphEl field
   to NIL. Individual graphical elements are hidden by clearing their
   visible flag.
*)

IMPORT
  Out, Display, Fonts, Display3, Printer3, Objects, Gadgets, Files, Printer,
  Math0, Conversion, Graphs, Num1;

CONST
  (* GraphEl.hor, vert *)
  center*=1;   left*=0;   right*=2;   bottom*=0;   top*=2;
  (* lay out *)
  WMIN = 150;			HMIN 		= 130;
  leBorder	= 1;		riBorder	= 15;
  loBorder	= 5;		toBorder 	= 15;
  xDNoLab	= 3;		xDFrameNo 	= 3;
  yDNoLab	= 3;		yDFrameNo 	= 3;
  (* colors *)
  BLACK	= 15;
  DARKGRAY	= 12;
  MEDGRAY	= 13;
  LIGHTGRAY	= 14;
  WHITE	= 0;
  (* font for y label *)
  UPFONT	= "OberUp10.Scn.Fnt";
  (* HALT codes *)
  SMALLFRAME = 103;

TYPE
  GraphEl*	= POINTER TO GraphElDesc;
  GraphElDesc*	= RECORD	(* base type of graphics elements list	*)
                    col*	: INTEGER;	(* color	*)
                    x*, y*	: REAL;		(* coordinates	*)
                    hor*,			(* center, left, right	*)
                    vert*,			(* center, bottom, top	*)
                    W*, H*	: INTEGER;	(* width, height *)
                    visible*	: BOOLEAN;
                    next	: GraphEl
                  END;
  String*	= POINTER TO StringDesc;
  StringDesc*	= RECORD (GraphElDesc)
                    string*	: ARRAY 64 OF CHAR;
                    font*	: Fonts.Font;
                  END;
  Pattern*	= POINTER TO PatternDesc;
  PatternDesc*	= RECORD (GraphElDesc)
                    pat*	: Display.Pattern;
                  END;
  Line*		= POINTER TO LineDesc;
  LineDesc*	= RECORD (GraphElDesc)	(* hor, vert not used *)
  				Dx*, Dy*	: REAL;
  				pat*		 : LONGINT;	(* e.g. Display.solid *)
  			  END;
  Frame*	= POINTER TO FrameDesc;
  FrameDesc*	= RECORD (Gadgets.FrameDesc)
                    (* defining parameters *)
                    xmin*, xmax*, ymin*, ymax*: REAL;
                    xLabel, yLabel	  : ARRAY 64 OF CHAR;
                    gratingCol : INTEGER;
                    graphEl*		: GraphEl;
                    (* derived layout parameters *)
                    msk			: Display3.Mask;(* plot area	*)
                    xyLab, yyLab, xxLab, yxLab,	(* axis label coordinates *)
                    xyNum, yxNum,		(* axis number coordinates *)
                    pxa, pxb, pya, pyb,		(* plot area	*)
                    Nx, Ny		: INTEGER; (* no.of grid line interv.*)
                    x0, Dx, y0, Dy,		(* grid lines: first, dist.*)
                    ax, bx, ay, by,		(* coeff. real->pixel	*)
                    f10x, f10y		: REAL;	(* scaling factor	*)
                    xScale, yScale	: ARRAY 8 OF CHAR; (* scaling factor *)
                    (* efficiency *)
                    recalc*		: BOOLEAN; (* r.layout upon new attr.*)
                    xo, yo		: INTEGER; (* last offsets (dirty) *)
                  END;
  (* All pixel coordinates are relative to the frame.	*)

VAR
  fntHeight, numWidth, h, fntDsr, expW	: INTEGER;
  upfont								: Fonts.Font;


PROCEDURE StringUp (m:Display3.Mask; col,x,y:INTEGER; font:Fonts.Font;
                    s:ARRAY OF CHAR; mode:INTEGER);
(* Writes s at location x,y (x is bottom line) upwards. *)
VAR i, dx, xc, yc, wc, hc	: INTEGER;
	pat					: Display.Pattern;
BEGIN
  IF font=NIL THEN RETURN END;
  FOR i:=0 TO LEN(s)-1 DO
    IF s[i]=0X THEN RETURN END;
    Fonts.GetChar (font, s[i], dx, xc, yc, wc, hc, pat);
    Display3.CopyPattern (m, col, pat, x-dx, y+yc, mode);
    INC (y, yc+hc+1);
  END;
END StringUp;


PROCEDURE CalcTick (Dp,numWidth:INTEGER; zmin,zmax:REAL;
                    VAR n:INTEGER; VAR z0,Dz:REAL);
(* The pixel interval of width Dp corresponds to the real interval
[zmin,zmax].
   numWidth is the width of an axis number (e.g."1.00") including the distance
   to its neighbor. The following will be calculated:
	n	number of large ticks
	z0	position of first large tick
	Dz	distance between large ticks.
   IF Dp is smaller than 1 number, or zmax-zmin<Math0.eps, then n=0. *)
CONST NMAX = 10;
VAR n0	: INTEGER;	(* first tick no. *)
    F10	: REAL;	(* norming factor, integral multiple of 10	*)
(* One tick length Dz shall from {1,2,5}*10^z with integral z. It shall be
   sufficiently large for the axis numbers. *)
BEGIN
  (* max. n in order the numbers to fit in *)
  n:= Math0.mini (NMAX, SHORT(ENTIER(Dp/numWidth)));
  IF n=0 THEN RETURN END;
  Dz:= (zmax-zmin) / n;
  IF Dz<Math0.eps THEN
    n:=0;   RETURN;
  END;
  (* norming Dz to [1,10) *)
  F10:= Math0.exp10 (ENTIER(Math0.lg(Dz)));
  Dz:= Dz / F10;
  (* select a value from {2,5,10} for Dz *)
  IF Dz<2.001 THEN Dz:=2
  ELSIF Dz<5.001 THEN Dz:=5
  ELSE Dz:=10
  END;
  (* normating back *)
  Dz:= Dz * F10;
  (* calculate other results *)
  n0:= SHORT(-ENTIER(-zmin/Dz+10*Math0.eps));
  z0:= n0 * Dz;
  n:= SHORT(ENTIER(zmax/Dz+10*Math0.eps)) - n0;
END CalcTick;


PROCEDURE pix* (F:Frame; x0:INTEGER; x:REAL): INTEGER;
(* Real -> pixel coordinate. x0 is the left border of F. *)
BEGIN RETURN SHORT(ENTIER(F.ax * x + F.bx + x0))
END pix;


PROCEDURE piy* (F:Frame; y0:INTEGER; y:REAL): INTEGER;
(* Real -> pixel coordinate. y0 is the bottom border of F. *)
BEGIN RETURN SHORT(ENTIER(F.ay * y + F.by + y0))
END piy;


PROCEDURE PlotGrid (F:Frame; Msk:Display3.Mask; x0,y0:INTEGER);
(* Erase plot area and plot grid *)
VAR i, p	: INTEGER;
    z	: REAL;
BEGIN
  Display3.FilledRect3D (Msk, DARKGRAY, WHITE, LIGHTGRAY, F.pxa+x0, F.pya+y0,
                 F.pxb-F.pxa, F.pyb-F.pya, 1, Display3.replace);
  IF F.Nx>0 THEN
    z:= F.x0;
    p:= pix(F,x0,z);
    FOR i:=0 TO F.Nx DO
      p:= pix(F,x0,z);
      Display3.Line (F.msk, F.gratingCol, Display.solid, p, F.msk.Y, p,
                     F.msk.Y+F.msk.H, 1, Display3.replace);
      z:= z + F.Dx;
    END;
  END;
  IF F.Ny>0 THEN
    z:= F.y0;
    p:= piy(F,y0,z);
    FOR i:=0 TO F.Ny DO
      p:= piy(F,y0,z);
      Display3.Line (F.msk, F.gratingCol, Display.solid, F.msk.X, p,
                     F.msk.X+F.msk.W, p, 1, Display3.replace);
      z:= z + F.Dy;
    END;
  END;
END PlotGrid;


PROCEDURE CalcLayOut (F:Frame; W,H:INTEGER);
(* Calculate layout variables and transformation coefficients for real->pixel.
*)
VAR exp: INTEGER;

PROCEDURE MakeExp (e:INTEGER; VAR s:ARRAY OF CHAR);
CONST O = ORD("0");
BEGIN
  IF e=0 THEN s[0]:=0X
  ELSE
    s[0]:= "E";
    IF e<0 THEN s[1]:="-" ELSE s[1]:="+" END;
    e:= ABS(e);
    IF e>99 THEN s[2]:="*";  s[3]:="*";
    ELSE
      s[2]:= CHR (e DIV 10 + O);
      s[3]:= CHR (e MOD 10 + O);
    END;
    s[4]:= 0X;
  END;
END MakeExp;

BEGIN
  (* x coordinates *)
  F.xyLab	:= leBorder + fntHeight;
  F.xyNum	:= F.xyLab + fntDsr + xDNoLab;
  F.pxa	:= F.xyNum + numWidth + xDFrameNo;
  F.pxb	:= W - riBorder;
  F.xxLab	:= (F.pxa+F.pxb) DIV 2;

  (* y coordinates *)
  F.yxLab	:= loBorder + fntDsr;
  F.yxNum	:= F.yxLab + fntHeight + yDNoLab;
  F.pya	:= F.yxNum + fntHeight + yDFrameNo;
  F.pyb	:= H - toBorder;
  F.yyLab	:= (F.pya + F.pyb) DIV 2;

  (* transformation coefficients *)
  F.ax	:= (F.pxb-F.pxa) / (F.xmax-F.xmin);
  F.bx	:= (F.pxa*F.xmax - F.pxb*F.xmin);
  F.bx	:= F.bx / (F.xmax - F.xmin) + 0.5;	(* not enough registers ...*)
  F.ay	:= (F.pyb-F.pya) / (F.ymax-F.ymin);
  F.by	:= (F.pya*F.ymax - F.pyb*F.ymin);
  F.by	:= F.by / (F.ymax - F.ymin) + 0.5;	(* not enough registers ...*)

  (* grid *)
  CalcTick (F.pxb-F.pxa, numWidth, F.xmin, F.xmax, F.Nx, F.x0, F.Dx);
  CalcTick (F.pyb-F.pya, SHORT(ENTIER(1.8*Fonts.Default.height)), F.ymin,
F.ymax,
            F.Ny, F.y0, F.Dy);
  IF (F.Nx=0) OR (F.Ny=0) THEN HALT(SMALLFRAME) END;

  (* skaling factor for x axis *)
  exp:= SHORT(ENTIER(Math0.lg(Math0.maxr
        (ABS(F.x0),ABS(F.x0+F.Nx*F.Dx)))+10*Math0.eps));
  IF exp=1 THEN exp:=0 END; (* numbers from [1, 100) without scaling factor *)
  F.f10x:= Math0.exp10 (exp);
  MakeExp (exp, F.xScale);

  (* skaling factor for y axis *)
  exp:= SHORT(ENTIER(Math0.lg(Math0.maxr
        (ABS(F.y0),ABS(F.y0+F.Ny*F.Dy)))+10*Math0.eps));
  IF exp=1 THEN exp:=0 END; (* numbers from [1, 100) without scaling factor *)
  F.f10y:= Math0.exp10 (exp);
  MakeExp (exp, F.yScale);

END CalcLayOut;


PROCEDURE DrawLayOut (F:Frame; Msk:Display3.Mask; x0,y0:INTEGER);
(* Draw coordinate system. *)
VAR
  i, p, labW	: INTEGER;
  z	: REAL;
  number	: ARRAY 8 OF CHAR;
BEGIN
  (* numbers for x-axis *)
  z:= F.x0;
  FOR i:=0 TO F.Nx DO
    Conversion.RealToStr (z/F.f10x, 5, number);
    Display3.String (Msk, Display3.FG, pix(F,x0,z) - numWidth DIV 2,
                     F.yxNum+y0, Fonts.Default, number, Display3.paint);
    z:= z + F.Dx;
  END;

  (* y-label using x-label-area as Buffer *)
  Display3.StringSize (F.yLabel, Fonts.Default, labW, i, p);
  StringUp (Msk, Display3.FG, F.xyLab+x0, F.yyLab - labW DIV 2 + y0,
            upfont, F.yLabel, Display3.paint);

  (* x-label *)
  Display3.StringSize (F.xLabel, Fonts.Default, labW, i, p);
  Display3.String (Msk, Display3.FG, F.xxLab - labW DIV 2 + x0, F.yxLab+y0,
                   Fonts.Default, F.xLabel, Display3.paint);

  (* numbers for y-axis *)
  z:= F.y0;
  FOR i:=0 TO F.Ny DO
    Conversion.RealToStr (z/F.f10y, 5, number);
    Display3.String (Msk, Display3.FG, F.xyNum+x0,
                     piy(F,y0,z) - Fonts.Default.height DIV 2 + 3,
                     Fonts.Default, number, Display3.paint);
    z:= z + F.Dy;
  END;

  (* x skaling factor *)
  Display3.String (Msk, Display3.FG, F.pxb-expW+x0, F.yxLab+y0, Fonts.Default,
                   F.xScale, Display3.paint);

  (* y skaling factor *)
  StringUp (Msk, Display3.FG, F.xyLab+x0, F.pyb-expW+y0, upfont,
            F.yScale, Display3.paint);

END DrawLayOut;



PROCEDURE PlotGraph (F:Frame; G:Graphs.Graph; x0,y0:INTEGER);
(* Draw G and all subsequent graphs in the list.
   (x0,y0) is the origin of F. *)
VAR i, x1, x2, y1, y2	: INTEGER;
    x			: REAL;
    f			: Num1.Func;
BEGIN
  WHILE G#NIL DO
    f:= G.f;
    IF (f#NIL)&(f.NP>1) THEN
      IF (f IS Num1.FuncGen) THEN
        WITH f:Num1.FuncGen DO
          (* search first value within plot area, no need to be exact
             (clipping) *)
          i:=0;
          WHILE (i<f.NP) & (f.x[i]<F.xmin) DO INC(i) END;
          IF i>=f.NP THEN RETURN END;
          (* plot *)
          x1:= pix (F, x0, f.x[i]);   y1:= piy (F, y0, f.y[i]);
          WHILE (f.x[i]<=F.xmax) & (i+1<f.NP) DO
            x2:= pix (F, x0, f.x[i+1]);   y2:= piy (F, y0, f.y[i+1]);
            Display3.Line (F.msk, G.col, G.line, x1, y1, x2, y2, 1,
                           Display3.replace);
            x1:=x2;   y1:=y2;
            INC(i);
          END;
        END;
      ELSE
      (* search first value within plot area, no need to be exact (clipping)*)
        IF ABS(f.dx)<1.0E-10 THEN RETURN END;
        i:= Math0.maxi(SHORT(ENTIER((F.xmin-f.x0)/f.dx)),0);
        IF i>=f.NP THEN RETURN END;
        x:= f.x0 + i*f.dx;
        x1:= pix(F,x0,x);   y1:= piy(F,y0,f.y[i]);
        (* plot *)
        WHILE (x<=F.xmax) & (i+1<f.NP) DO
          x:= x + f.dx;   INC(i);
          x2:= pix(F,x0,x);   y2:= piy(F,y0,f.y[i]);
          Display3.Line (F.msk, G.col, G.line, x1, y1, x2, y2, 1,
                         Display3.replace);
          x1:=x2;   y1:=y2;
        END;
      END;
    END;
    G:= G.next;
  END;
END PlotGraph;


PROCEDURE DrawGraphEl (F:Frame; x0,y0:INTEGER);
(* Draw the whole list of graphic elements. *)
VAR x, y, x1, y1: INTEGER;
    ge		: GraphEl;
BEGIN
  ge:= F.graphEl;
  WHILE ge#NIL DO
    IF ge.visible THEN
      IF ge IS Line THEN
        x:= pix(F,x0,ge.x);
    	y:= piy(F,y0,ge.y);
      ELSE
        x:= pix(F,x0,ge.x) - ge.W*ge.hor DIV 2;
    	y:= piy(F,y0,ge.y) - ge.H*ge.vert DIV 2;
      END;
      IF ge IS String THEN
        WITH ge:String DO
          Display3.String (F.msk, ge.col, x, y, ge.font, ge.string,
                           Display3.paint);
        END
      ELSIF ge IS Pattern THEN
        WITH ge:Pattern DO
          Display3.CopyPattern (F.msk, ge.col, ge.pat, x, y, Display.paint);
        END
      ELSIF ge IS Line THEN
        WITH ge:Line DO
          x1:= pix (F, x0, ge.x+ge.Dx);
          y1:= piy (F, y0, ge.y+ge.Dy);
          Display3.Line (F.msk, ge.col, ge.pat, x, y, x1, y1, 1,
Display3.replace);
        END
      END
    END;
    ge:= ge.next
  END;
END DrawGraphEl;



PROCEDURE Restore (F:Frame; diagMsk:Display3.Mask; x,y,w,h:INTEGER);
BEGIN
  (* draw outer rectangle *)
  Display3.Rect3D (diagMsk, Display3.bottomC, Display3.topC,
                         x, y, w, h, 1, Display.replace);
  Display3.FilledRect3D (diagMsk, Display3.topC, Display3.bottomC, MEDGRAY,
                         x+1, y+1, w-2, h-2, 1, Display.replace);

  (* adjust plot mask *)
  Display3.Copy (diagMsk, F.msk); (*allocs F.msk! No other
msk-copy-proc.found*)
  Display3.AdjustMask (F.msk, F.pxa+1+x, F.pya+1+y, F.pxb-F.pxa-2,
                       F.pyb-F.pya-2);
  (* draw everything *)
  DrawLayOut (F, diagMsk, x, y);
  PlotGrid (F, diagMsk, x, y);
  IF F.obj#NIL THEN PlotGraph (F, F.obj(Graphs.Graph),x,y)  END;
  DrawGraphEl (F, x, y);
  IF Gadgets.selected IN F.state THEN
    Display3.FillPattern (diagMsk, Display3.white, Display3.selectpat,
                          x, y, x, y, w, h, Display.paint);
  END;
END Restore;



PROCEDURE Attributes (F:Frame; VAR M:Objects.AttrMsg);
BEGIN
  IF M.id=Objects.get THEN
    IF M.name="Gen" THEN
      M.class:= Objects.String;
      COPY("Diagrams.New", M.s);
      M.res:= 0
    ELSIF M.name="Min. x-Value" THEN
      M.class:= Objects.Real;
      M.x:= F.xmin;
      M.res:= 0 
    ELSIF M.name="Max. x-Value" THEN
      M.class:= Objects.Real;
      M.x:= F.xmax;
      M.res:= 0 
    ELSIF M.name="Min. y-Value" THEN
      M.class:= Objects.Real;
      M.x:= F.ymin;
      M.res:= 0 
    ELSIF M.name="Max. y-Value" THEN
      M.class:= Objects.Real;
      M.x:= F.ymax;
      M.res:= 0 
    ELSIF M.name="x Label" THEN
      M.class:= Objects.String;
      COPY (F.xLabel, M.s);
      M.res:= 0 
    ELSIF M.name="y Label" THEN
      M.class:= Objects.String;
      COPY (F.yLabel, M.s);
      M.res:= 0 
    ELSIF M.name="Grating Color" THEN
      M.class:= Objects.Int;
      M.i:= F.gratingCol;
      M.res:= 0 
    ELSIF M.name="Cmd" THEN
      Gadgets.framehandle (F, M);
      IF M.res<0 THEN (* no such attribute, simulate one *)
        M.class:= Objects.String;
        M.s:= "";
        M.res:= 0; 
      END
    ELSE Gadgets.framehandle(F, M)
    END
  ELSIF M.id=Objects.set THEN
  (* LONGREAL and LONGINT components of M are used only in order to be
     compatible with the inspector.
     Only the REAL and INTEGER range is allowed. *) 
    IF M.name="Min. x-Value" THEN
      F.xmin:= SHORT(M.y);
      M.res:= 0;
      IF F.recalc THEN CalcLayOut (F, F.W, F.H) END;
    ELSIF M.name="Max. x-Value" THEN
      F.xmax:= SHORT(M.y);
      M.res:= 0;
      IF F.recalc THEN CalcLayOut (F, F.W, F.H) END;
    ELSIF M.name="Min. y-Value" THEN
      F.ymin:= SHORT(M.y);
      M.res:= 0;
      IF F.recalc THEN CalcLayOut (F, F.W, F.H) END;
    ELSIF M.name="Max. y-Value" THEN
      F.ymax:= SHORT(M.y);
      M.res:= 0;
      IF F.recalc THEN CalcLayOut (F, F.W, F.H) END;
    ELSIF M.name="x Label" THEN
      COPY (M.s, F.xLabel);
      M.res:= 0;
    ELSIF M.name="y Label" THEN
      COPY (M.s, F.yLabel);
      M.res:= 0 
    ELSIF M.name="Grating Color" THEN
      F.gratingCol:= SHORT(M.i);
      M.res:= 0;
    ELSE Gadgets.framehandle(F, M);
    END;
  ELSIF M.id=Objects.enum THEN
    M.Enum("Min. x-Value");
    M.Enum("Max. x-Value");
    M.Enum("Min. y-Value");
    M.Enum("Max. y-Value");
    M.Enum("x Label");
    M.Enum("y Label");
    M.Enum("Grating Color");
    M.Enum("Cmd");
    Gadgets.framehandle(F, M)
  END
END Attributes;


PROCEDURE Print (F:Frame; VAR M:Display.PrintMsg);
VAR Q: Display3.Mask;

  PROCEDURE P (x:INTEGER): INTEGER;
  BEGIN RETURN SHORT(x * LONG(10000) DIV Printer.Unit)
  END P;

BEGIN
  Gadgets.MakePrinterMask (F, M.x, M.y, M.dlink, Q);
  Printer3.ReplConst (Q, MEDGRAY, M.x, M.y, P(F.W), P(F.H), Display.replace)
END Print;


PROCEDURE FileIO (F:Frame; VAR M:Objects.FileMsg);
BEGIN
  IF M.id = Objects.store THEN
    Files.WriteReal (M.R, F.xmin);
    Files.WriteReal (M.R, F.xmax);
    Files.WriteReal (M.R, F.ymin);
    Files.WriteReal (M.R, F.ymax);
    Files.WriteString (M.R, F.xLabel);
    Files.WriteString (M.R, F.yLabel);
    Files.WriteInt (M.R, F.gratingCol);
    Files.WriteBool(M.R, F.recalc);
    Gadgets.framehandle (F, M);
 ELSIF M.id = Objects.load THEN
    Files.ReadReal (M.R, F.xmin);
    Files.ReadReal (M.R, F.xmax);
    Files.ReadReal (M.R, F.ymin);
    Files.ReadReal (M.R, F.ymax);
    Files.ReadString (M.R, F.xLabel);
    Files.ReadString (M.R, F.yLabel);
    Files.ReadInt (M.R, F.gratingCol);
    Files.ReadBool(M.R, F.recalc);
    Gadgets.framehandle (F, M);
    F.graphEl:= NIL;	F.msk:= NIL;
    CalcLayOut (F, F.W, F.H);
  END;
END FileIO;


PROCEDURE Copy* (VAR M:Objects.CopyMsg; from,to:Frame);
BEGIN
  to.xmin	:= from.xmin;
  to.xmax	:= from.xmax;
  to.ymin	:= from.ymin;
  to.ymax	:= from.ymax;
  COPY (from.xLabel, to.xLabel);
  COPY (from.yLabel, to.yLabel);
  to.gratingCol	:= from.gratingCol;
  to.xyLab	:= from.xyLab;	to.yyLab	:= from.yyLab;
  to.xxLab	:= from.xxLab;	to.yxLab	:= from.yxLab;
  to.xyNum	:= from.xyNum;	to.yxNum	:= from.yxNum;
  to.pxa	:= from.pxa;	to.pxb	:= from.pxb;
  to.pya	:= from.pya;	to.pyb	:= from.pyb;
  to.Nx	:= from.Nx;	to.Ny	:= from.Ny;
  to.x0	:= from.x0;	to.Dx	:= from.Dx;
  to.y0	:= from.y0;	to.Dy	:= from.Dy;
  to.ax	:= from.ax;	to.bx	:= from.bx;
  to.ay	:= from.ay;	to.by	:= from.by;
  to.f10x	:= from.f10x;	to.f10y	:= from.f10y;
  COPY (from.xScale, to.xScale);	COPY (from.yScale, to.yScale);
  Gadgets.CopyFrame (M, from, to);
END Copy;



PROCEDURE Handle* (F:Objects.Object; VAR M:Objects.ObjMsg);
VAR x, y, w, h	: INTEGER;
    F1		: Frame;
    Q		: Display3.Mask;
BEGIN
  WITH F:Frame DO
    IF M IS Display.FrameMsg THEN
      WITH M:Display.FrameMsg DO
        IF (M.F=NIL) OR (M.F=F) THEN (* message addressed to box *)
          x:= M.x + F.X;   y:= M.y + F.Y;   w:= F.W;   h:= F.H;
          F.xo:= x;	F.yo:= y;
          IF M IS Display.DisplayMsg THEN
            Gadgets.MakeMask (F, x, y, M.dlink, Q);
            WITH M: Display.DisplayMsg DO
              IF (M.id=Display.frame) OR (M.F=NIL) THEN
                Restore (F, Q, x, y, w, h);
              ELSIF M.id=Display.area THEN
                Display3.AdjustMask (Q, x+M.u, y+h-1+M.v, M.w, M.h);
                Restore (F, Q, x, y, w, h);
              END;
            END
          ELSIF M IS Display.ModifyMsg THEN	(* guarantee minimum size *)
            WITH M:Display.ModifyMsg DO
              M.W:= Math0.maxi (M.W, WMIN);   M.H:= Math0.maxi (M.H, HMIN);
              IF (M.id=Display.reduce) OR (M.id=Display.extend) THEN
                CalcLayOut (F, M.W, M.H);
                (* Gadgets Programmer Reference:"Gadgets ignore the id field."
                   Using id field works, but eventually has to be changed with
                   new version. *)
              END;
            END;
            Gadgets.framehandle(F, M);
          ELSIF (M IS Gadgets.UpdateMsg) THEN
            WITH M:Gadgets.UpdateMsg DO
              IF F.obj#NIL THEN F.obj.handle (F.obj, M) END;
              IF M.obj=F.obj THEN
                IF M.stamp#F.stamp THEN
                  F.stamp:=M.stamp;
                  Gadgets.MakeMask (F, x, y, M.dlink, Q);
                  PlotGrid (F, Q, x, y);
                  PlotGraph (F, M.obj(Graphs.Graph), x, y);
                  DrawGraphEl (F, x, y);
                END
              ELSE
                Gadgets.framehandle (F, M)
              END
            END
          ELSIF M IS Display.PrintMsg THEN Print (F, M(Display.PrintMsg));
          ELSE Gadgets.framehandle(F, M);
          END
        END
      END
    ELSIF M IS Objects.AttrMsg THEN Attributes (F, M(Objects.AttrMsg));
    ELSIF M IS Objects.FileMsg THEN FileIO (F, M(Objects.FileMsg));
    ELSIF M IS Objects.CopyMsg THEN
      WITH M:Objects.CopyMsg DO
        IF M.stamp=F.stamp THEN (*non-first arrival*)
          M.obj:= F.dlink
        ELSE (*first arrival*)
          NEW(F1);   F.stamp:= M.stamp;   F.dlink:= F1;
          Copy (M, F, F1);   M.obj := F1
        END
      END
    ELSE Gadgets.framehandle (F, M);
    END
  END;
END Handle;


PROCEDURE New*;
VAR F: Frame;
BEGIN
  NEW(F);
  F.xmin	:= -1.0;	F.xmax		:= 1.0;
  F.ymin	:= -1.0;	F.ymax		:= 1.0;
  COPY (" x", F.xLabel);	COPY (" y", F.yLabel);
  F.gratingCol	:= MEDGRAY;
  F.W		:= WMIN;		F.H	:= HMIN;
  F.recalc	:= TRUE;
  CalcLayOut (F, F.W, F.H);
  F.handle	:= Handle;
  Objects.NewObj:= F;
END New;


PROCEDURE PlotPoints* (F:Frame; G:Graphs.Graph; p1,p2:INTEGER);
(* Draw line sequence p1 - p1+1 - ... -p2.
   Somewhat dirty because it doesn't use messages, but draws directly with the
   pixel offsets of the last frame message, in order to increase speed.
   Application: draw a graph point by point ('real time'), not only after it
   has been completely calculated or measured. *)
VAR p: INTEGER;
    x: REAL;
    f: Num1.Func;
BEGIN
  f:= G.f;
  IF f IS Num1.FuncGen THEN
    WITH f:Num1.FuncGen DO
      FOR p:=p1+1 TO p2 DO
        Display3.Line
          (F.msk, G.col, G.line, pix(F,F.xo,f.x[p-1]), piy(F,F.yo,f.y[p-1]),
           pix(F,F.xo,f.x[p]), piy(F,F.yo,f.y[p]), 1, Display3.replace);
      END;
    END;
  ELSE (* f IS Num1.Func *)
    x:= f.x0 + p1*f.dx;
    FOR p:=p1+1 TO p2 DO
      Display3.Line
        (F.msk, G.col, G.line, pix(F,F.xo,x-f.dx), piy(F,F.yo,f.y[p-1]),
         pix(F,F.xo,x), piy(F,F.yo,f.y[p]), 1, Display3.replace);
    END;
  END;
END PlotPoints;


PROCEDURE AddGraphEl* (F:Frame; ge:GraphEl);
(* Add an allocated graphic element to the list of F.
   ge^ can be changed after adding it to the list, but it must not be used a
   second time in this procedure, before removing it from the list! *)
VAR gf	: GraphEl;
BEGIN
  ge.next:= NIL;
  IF F.graphEl=NIL THEN F.graphEl:= ge
  ELSE
    gf:= F.graphEl;
    WHILE gf.next#NIL DO gf:= gf.next END;
    gf.next:= ge
  END
END AddGraphEl;


PROCEDURE RemGraphEl* (F:Frame; ge:GraphEl);
(* Remove graphic element from the list *)
VAR gf: GraphEl;
BEGIN
  (* search predecessor of ge *)
  gf:= F.graphEl;
  IF gf=NIL THEN RETURN END;
  WHILE (gf.next#NIL)&(gf.next#ge) DO gf:= gf.next END;
  (* exclude ge from list *)
  IF gf.next=ge THEN gf.next:=ge.next END;  
END RemGraphEl;


PROCEDURE RemAllGraphEl* (F:Frame);
(* Remove all graphic elements *)
BEGIN F.graphEl:= NIL
END RemAllGraphEl;



BEGIN
  fntHeight:= Fonts.Default.height;
  Display3.StringSize ("-8.88", Fonts.Default, numWidth, h, fntDsr);
  Display3.StringSize ("E+88", Fonts.Default, expW, h, fntDsr);
  upfont:= Fonts.This (UPFONT);
  IF upfont=NIL THEN
    Out.String("Diagrams: Font not found: ");	Out.String(UPFONT);	Out.Ln;
  END;
END Diagrams.


System.Free Diagrams ~
Gadgets.Insert Diagrams.New ~
Diagr.Test
